/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  04/2002 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//   MDataMember
//
//   This objects corresponds to the data member of another object.
//   You can either specify the object as a string, eg "MHillas.fWidth"
//   where MHillas is the name of the container in the parameterlist
//   and fWidth is it's data member, or you can specify it by giving
//   the pointer corresponding to the instance of your object and
//   a TMethodCall pointer corresponding to the Member function returning
//   the requested value.
//
/////////////////////////////////////////////////////////////////////////////

#include "MDataMember.h"

#include <TMethodCall.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

ClassImp(MDataMember);

// --------------------------------------------------------------------------
//
//  obj is a pointer to the instance of your class from which the data
//  should be requested. TMethodCall (s. root dox) is a pointer
//  to a TMethodCall object which should be the getter function for
//  the data you want to get.
//
MDataMember::MDataMember(MParContainer *obj, TMethodCall *call)
{
    fObject = obj;
    fCall   = call;
}

// --------------------------------------------------------------------------
//
//  obj is a pointer to the instance of your class from which the data
//  should be requested. TMethodCall (s. root dox) is a pointer
//  to a TMethodCall object which should be the getter function for
//  the data you want to get.
//
MDataMember::MDataMember(MParContainer *obj, const TString call)
{
    fObject = obj;
    fCall   = obj->GetterMethod(call);
}

// --------------------------------------------------------------------------
//
// returns the value you requested
//
Double_t MDataMember::GetValue() const
{
    if (!fCall)
    {
        *fLog << err << "No TMethodCall for " << fName << " of ";
        *fLog << fObject->GetName() << " available... returning 0." << endl;
        return 0;
    }

    switch (fCall->ReturnType())
    {
    case TMethodCall::kLong:
        Long_t l;
        fCall->Execute(fObject, l);
        return (Double_t)l;

    case TMethodCall::kDouble:
        Double_t v;
        fCall->Execute(fObject, v);
        return v;

    default:
        *fLog << err << "DataMember " << fName << " of ";
        *fLog << fObject->GetName() << " neither int nor float... returning 0." << endl;
        return 0;
    }
}

// --------------------------------------------------------------------------
//
// If a string was given PreProcess try to resolv the object name and
// tries to get it from the parlist. And also tries to resolve
// the data member (variable) name you requested and tries to get a
// corresponding TMethodCall from the root Dictionary.
// Remark: If your Data Member is called fDataMember the corresponding
//         getter Method in your class must be calles fDataMember
//
Bool_t MDataMember::PreProcess(const MParList *plist)
{
    if (fCall)
        return kTRUE;

    TString cname(fName);
    TString mname(fName);

    const char *dot = strrchr(cname, '.');

    if (dot)
    {
        const int pos = dot-cname;

        cname.Remove(pos);
        mname.Remove(0, pos+1);
    }

    fObject = (MParContainer*)plist->FindObject(cname);
    if (!fObject)
    {
        *fLog << err << "Object '" << cname << "' not in parameter list... aborting." << endl;
        return kFALSE;
    }

    fCall = fObject->GetterMethod(mname);

    return fCall ? kTRUE : kFALSE;
}

// --------------------------------------------------------------------------
//
// Returns the ready-to-save flag of the data member container
//
Bool_t MDataMember::IsReadyToSave() const
{
    return IsValid() ? fObject->IsReadyToSave() : kFALSE;
}

// --------------------------------------------------------------------------
//
// Print the name of the data member without an CR.
//
void MDataMember::Print(Option_t *opt) const
{
    *fLog << fName << flush;
}

