/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Marcos Lopez  2/2005 <mailto:marcos@gae.ucm.es>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MHadAlphaCut
//
//  Container for Hadronness and Alpha cut for each Estimated Energy and theta
//
/////////////////////////////////////////////////////////////////////////////
#include "MHadAlphaCut.h"


#include <TH2F.h>
#include <TCanvas.h>

#include "MBinning.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHadAlphaCut);

using namespace std;


// ------------------------------------------------------------------------
//
//
MHadAlphaCut::MHadAlphaCut(const char* name, const char* title)
{
    fName  = name  ? name  : "MHadAlphaCut";
    fTitle = title ? title : "Container of Hadronness and Alpha cuts";
    

    fHistHadCut = new TH2F;
    fHistHadCut->UseCurrentStyle();
    fHistHadCut->SetNameTitle("HadCut","HadCut");
    fHistHadCut->SetXTitle("Energy");
    fHistHadCut->SetYTitle("Theta");
    fHistHadCut->SetZTitle("Hadronness Cut");

    fHistAlphaCut = new TH2F; 
    fHistAlphaCut->UseCurrentStyle();
    fHistAlphaCut->SetNameTitle("AlphaCut","AlphaCut");
    fHistAlphaCut->SetXTitle("Energy");
    fHistAlphaCut->SetYTitle("Theta");
    fHistAlphaCut->SetZTitle("Alpha Cut");
  

    MBinning binsx;
    MBinning binsy;
    
    binsx.SetEdgesLog(8, 50., 2000);  // Est. Energy binning
    binsy.SetEdges(2,14,31);          // Theta binning

    MH::SetBinning(fHistHadCut, &binsx, &binsy);
    MH::SetBinning(fHistAlphaCut, &binsx, &binsy);
    

    for(int ix=1; ix<=fHistHadCut->GetNbinsX(); ix++)
	for(int iy=1; iy<=fHistHadCut->GetNbinsY(); iy++)
	{
	    fHistAlphaCut->SetBinContent(ix,iy,10);
	    fHistHadCut->SetBinContent(ix,iy,0.3);
	}
}


Float_t MHadAlphaCut::GetAlphaCut(Float_t energy, Float_t zd)
{
    Int_t bin = fHistAlphaCut->FindBin(energy,zd);
    return fHistAlphaCut->GetBinContent(bin);
}

Float_t MHadAlphaCut::GetHadCut(Float_t energy, Float_t zd)
{ 
    Int_t bin = fHistHadCut->FindBin(energy,zd);
    return fHistHadCut->GetBinContent(bin);
} 


//------------------------------------------------------------------------
//
//
void MHadAlphaCut::Print(Option_t *o) const
{

    // *fLog << " Size bin \t HadCut \t AlphaCut" << endl;
    for(Int_t ix=1; ix<=fHistHadCut->GetNbinsX(); ix++)
    {
	for(Int_t iy=1; iy<=fHistHadCut->GetNbinsY(); iy++)
	{	
	    *fLog <<"(" <<ix << ", " << iy << ") = " 
		  << fHistHadCut->GetBinContent(ix,iy) << " " << 
		fHistAlphaCut->GetBinContent(ix,iy) << endl;
	}
    }
}


// -------------------------------------------------------------------------
//
//
void MHadAlphaCut::Draw(Option_t *o)
{
    TCanvas *c1 = new TCanvas;
    c1->SetLogx();
    fHistAlphaCut->Draw("lego");

    TCanvas *c2 = new TCanvas;
    c2->SetLogx();
    fHistHadCut->Draw("lego");
}




