/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Oscar Blanch <mailto:blanch@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MGeomCamMagicHG
//
// This class stores the geometry information of the hipotethic Magic camera
// with high granularity
// All information are copied from tables, see source code.
//
////////////////////////////////////////////////////////////////////////////
#include "MGeomCamMagicHG.h"

#include "MGeomPix.h"

ClassImp(MGeomCamMagicHG);

// --------------------------------------------------------------------------
//
//  Magic high Granularity camera could have 2401 pixels. For geometry and Next Neighbor 
//  info see CreateCam and CreateNN
//
MGeomCamMagicHG::MGeomCamMagicHG(const char *name)
    : MGeomCam(2401, 17, name, "Geometry information of high Granularity Magic Camera")
{
    CreateCam();
    CreateNN();
    InitGeometry();
}

// --------------------------------------------------------------------------
//
//  This fills the geometry information from a table into the pixel objects.
//
void MGeomCamMagicHG::CreateCam()
{

    //
    //   here define the hardwire things of the magic telescope
    //
  const float xtemp[2401] = {
    0, 15, 7.5, -7.5, -15, -7.5, 7.5, 30, // 0
    22.5, 15, 0, -15, -22.5, -30, -22.5, -15, // 8
    0, 15, 22.5, 45, 37.5, 30, 22.5, 7.5, // 16
    -7.5, -22.5, -30, -37.5, -45, -37.5, -30, -22.5, // 24
    -7.5, 7.5, 22.5, 30, 37.5, 60, 52.5, 45, // 32
    37.5, 30, 15, 0, -15, -30, -37.5, -45, // 40
    -52.5, -60, -52.5, -45, -37.5, -30, -15, 0, // 48
    15, 30, 37.5, 45, 52.5, 75, 67.5, 60, // 56
    52.5, 45, 37.5, 22.5, 7.5, -7.5, -22.5, -37.5, // 64
    -45, -52.5, -60, -67.5, -75, -67.5, -60, -52.5, // 72
    -45, -37.5, -22.5, -7.5, 7.5, 22.5, 37.5, 45, // 80
    52.5, 60, 67.5, 90, 82.5, 75, 67.5, 60, // 88
    52.5, 45, 30, 15, 0, -15, -30, -45, // 96
    -52.5, -60, -67.5, -75, -82.5, -90, -82.5, -75, // 104
    -67.5, -60, -52.5, -45, -30, -15, 0, 15, // 112
    30, 45, 52.5, 60, 67.5, 75, 82.5, 105, // 120
    97.5, 90, 82.5, 75, 67.5, 60, 52.5, 37.5, // 128
    22.5, 7.5, -7.5, -22.5, -37.5, -52.5, -60, -67.5, // 136
    -75, -82.5, -90, -97.5, -105, -97.5, -90, -82.5, // 144
    -75, -67.5, -60, -52.5, -37.5, -22.5, -7.5, 7.5, // 152
    22.5, 37.5, 52.5, 60, 67.5, 75, 82.5, 90, // 160
    97.5, 120, 112.5, 105, 97.5, 90, 82.5, 75, // 168
    67.5, 60, 45, 30, 15, 0, -15, -30, // 176
    -45, -60, -67.5, -75, -82.5, -90, -97.5, -105, // 184
    -112.5, -120, -112.5, -105, -97.5, -90, -82.5, -75, // 192
    -67.5, -60, -45, -30, -15, 0, 15, 30, // 200
    45, 60, 67.5, 75, 82.5, 90, 97.5, 105, // 208
    112.5, 135, 127.5, 120, 112.5, 105, 97.5, 90, // 216
    82.5, 75, 67.5, 52.5, 37.5, 22.5, 7.5, -7.5, // 224
    -22.5, -37.5, -52.5, -67.5, -75, -82.5, -90, -97.5, // 232
    -105, -112.5, -120, -127.5, -135, -127.5, -120, -112.5, // 240
    -105, -97.5, -90, -82.5, -75, -67.5, -52.5, -37.5, // 248
    -22.5, -7.5, 7.5, 22.5, 37.5, 52.5, 67.5, 75, // 256
    82.5, 90, 97.5, 105, 112.5, 120, 127.5, 150, // 264
    142.5, 135, 127.5, 120, 112.5, 105, 97.5, 90, // 272
    82.5, 75, 60, 45, 30, 15, 0, -15, // 280
    -30, -45, -60, -75, -82.5, -90, -97.5, -105, // 288
    -112.5, -120, -127.5, -135, -142.5, -150, -142.5, -135, // 296
    -127.5, -120, -112.5, -105, -97.5, -90, -82.5, -75, // 304
    -60, -45, -30, -15, 0, 15, 30, 45, // 312
    60, 75, 82.5, 90, 97.5, 105, 112.5, 120, // 320
    127.5, 135, 142.5, 165, 157.5, 150, 142.5, 135, // 328
    127.5, 120, 112.5, 105, 97.5, 90, 82.5, 67.5, // 336
    52.5, 37.5, 22.5, 7.5, -7.5, -22.5, -37.5, -52.5, // 344
    -67.5, -82.5, -90, -97.5, -105, -112.5, -120, -127.5, // 352
    -135, -142.5, -150, -157.5, -165, -157.5, -150, -142.5, // 360
    -135, -127.5, -120, -112.5, -105, -97.5, -90, -82.5, // 368
    -67.5, -52.5, -37.5, -22.5, -7.5, 7.5, 22.5, 37.5, // 376
    52.5, 67.5, 82.5, 90, 97.5, 105, 112.5, 120, // 384
    127.5, 135, 142.5, 150, 157.5, 180, 172.5, 165, // 392
    157.5, 150, 142.5, 135, 127.5, 120, 112.5, 105, // 400
    97.5, 90, 75, 60, 45, 30, 15, 0, // 408
    -15, -30, -45, -60, -75, -90, -97.5, -105, // 416
    -112.5, -120, -127.5, -135, -142.5, -150, -157.5, -165, // 424
    -172.5, -180, -172.5, -165, -157.5, -150, -142.5, -135, // 432
    -127.5, -120, -112.5, -105, -97.5, -90, -75, -60, // 440
    -45, -30, -15, 0, 15, 30, 45, 60, // 448
    75, 90, 97.5, 105, 112.5, 120, 127.5, 135, // 456
    142.5, 150, 157.5, 165, 172.5, 195, 187.5, 180, // 464
    172.5, 165, 157.5, 150, 142.5, 135, 127.5, 120, // 472
    112.5, 105, 97.5, 82.5, 67.5, 52.5, 37.5, 22.5, // 480
    7.5, -7.5, -22.5, -37.5, -52.5, -67.5, -82.5, -97.5, // 488
    -105, -112.5, -120, -127.5, -135, -142.5, -150, -157.5, // 496
    -165, -172.5, -180, -187.5, -195, -187.5, -180, -172.5, // 504
    -165, -157.5, -150, -142.5, -135, -127.5, -120, -112.5, // 512
    -105, -97.5, -82.5, -67.5, -52.5, -37.5, -22.5, -7.5, // 520
    7.5, 22.5, 37.5, 52.5, 67.5, 82.5, 97.5, 105, // 528
    112.5, 120, 127.5, 135, 142.5, 150, 157.5, 165, // 536
    172.5, 180, 187.5, 210, 202.5, 195, 187.5, 180, // 544
    172.5, 165, 157.5, 150, 142.5, 135, 127.5, 120, // 552
    112.5, 105, 90, 75, 60, 45, 30, 15, // 560
    0, -15, -30, -45, -60, -75, -90, -105, // 568
    -112.5, -120, -127.5, -135, -142.5, -150, -157.5, -165, // 576
    -172.5, -180, -187.5, -195, -202.5, -210, -202.5, -195, // 584
    -187.5, -180, -172.5, -165, -157.5, -150, -142.5, -135, // 592
    -127.5, -120, -112.5, -105, -90, -75, -60, -45, // 600
    -30, -15, 0, 15, 30, 45, 60, 75, // 608
    90, 105, 112.5, 120, 127.5, 135, 142.5, 150, // 616
    157.5, 165, 172.5, 180, 187.5, 195, 202.5, 225, // 624
    217.5, 210, 202.5, 195, 187.5, 180, 172.5, 165, // 632
    157.5, 150, 142.5, 135, 127.5, 120, 112.5, 97.5, // 640
    82.5, 67.5, 52.5, 37.5, 22.5, 7.5, -7.5, -22.5, // 648
    -37.5, -52.5, -67.5, -82.5, -97.5, -112.5, -120, -127.5, // 656
    -135, -142.5, -150, -157.5, -165, -172.5, -180, -187.5, // 664
    -195, -202.5, -210, -217.5, -225, -217.5, -210, -202.5, // 672
    -195, -187.5, -180, -172.5, -165, -157.5, -150, -142.5, // 680
    -135, -127.5, -120, -112.5, -97.5, -82.5, -67.5, -52.5, // 688
    -37.5, -22.5, -7.5, 7.5, 22.5, 37.5, 52.5, 67.5, // 696
    82.5, 97.5, 112.5, 120, 127.5, 135, 142.5, 150, // 704
    157.5, 165, 172.5, 180, 187.5, 195, 202.5, 210, // 712
    217.5, 240, 232.5, 225, 217.5, 210, 202.5, 195, // 720
    187.5, 180, 172.5, 165, 157.5, 150, 142.5, 135, // 728
    127.5, 120, 105, 90, 75, 60, 45, 30, // 736
    15, 0, -15, -30, -45, -60, -75, -90, // 744
    -105, -120, -127.5, -135, -142.5, -150, -157.5, -165, // 752
    -172.5, -180, -187.5, -195, -202.5, -210, -217.5, -225, // 760
    -232.5, -240, -232.5, -225, -217.5, -210, -202.5, -195, // 768
    -187.5, -180, -172.5, -165, -157.5, -150, -142.5, -135, // 776
    -127.5, -120, -105, -90, -75, -60, -45, -30, // 784
    -15, 0, 15, 30, 45, 60, 75, 90, // 792
    105, 120, 127.5, 135, 142.5, 150, 157.5, 165, // 800
    172.5, 180, 187.5, 195, 202.5, 210, 217.5, 225, // 808
    232.5, 255, 247.5, 240, 232.5, 225, 217.5, 210, // 816
    202.5, 195, 187.5, 180, 172.5, 165, 157.5, 150, // 824
    142.5, 135, 127.5, 112.5, 97.5, 82.5, 67.5, 52.5, // 832
    37.5, 22.5, 7.5, -7.5, -22.5, -37.5, -52.5, -67.5, // 840
    -82.5, -97.5, -112.5, -127.5, -135, -142.5, -150, -157.5, // 848
    -165, -172.5, -180, -187.5, -195, -202.5, -210, -217.5, // 856
    -225, -232.5, -240, -247.5, -255, -247.5, -240, -232.5, // 864
    -225, -217.5, -210, -202.5, -195, -187.5, -180, -172.5, // 872
    -165, -157.5, -150, -142.5, -135, -127.5, -112.5, -97.5, // 880
    -82.5, -67.5, -52.5, -37.5, -22.5, -7.5, 7.5, 22.5, // 888
    37.5, 52.5, 67.5, 82.5, 97.5, 112.5, 127.5, 135, // 896
    142.5, 150, 157.5, 165, 172.5, 180, 187.5, 195, // 904
    202.5, 210, 217.5, 225, 232.5, 240, 247.5, 270, // 912
    262.5, 255, 247.5, 240, 232.5, 225, 217.5, 210, // 920
    202.5, 195, 187.5, 180, 172.5, 165, 157.5, 150, // 928
    142.5, 135, 120, 105, 90, 75, 60, 45, // 936
    30, 15, 0, -15, -30, -45, -60, -75, // 944
    -90, -105, -120, -135, -142.5, -150, -157.5, -165, // 952
    -172.5, -180, -187.5, -195, -202.5, -210, -217.5, -225, // 960
    -232.5, -240, -247.5, -255, -262.5, -270, -262.5, -255, // 968
    -247.5, -240, -232.5, -225, -217.5, -210, -202.5, -195, // 976
    -187.5, -180, -172.5, -165, -157.5, -150, -142.5, -135, // 984
    -120, -105, -90, -75, -60, -45, -30, -15, // 992
    0, 15, 30, 45, 60, 75, 90, 105, // 1000
    120, 135, 142.5, 150, 157.5, 165, 172.5, 180, // 1008
    187.5, 195, 202.5, 210, 217.5, 225, 232.5, 240, // 1016
    247.5, 255, 262.5, 285, 277.5, 270, 262.5, 255, // 1024
    247.5, 240, 232.5, 225, 217.5, 210, 202.5, 195, // 1032
    187.5, 180, 172.5, 165, 157.5, 150, 142.5, 127.5, // 1040
    112.5, 97.5, 82.5, 67.5, 52.5, 37.5, 22.5, 7.5, // 1048
    -7.5, -22.5, -37.5, -52.5, -67.5, -82.5, -97.5, -112.5, // 1056
    -127.5, -142.5, -150, -157.5, -165, -172.5, -180, -187.5, // 1064
    -195, -202.5, -210, -217.5, -225, -232.5, -240, -247.5, // 1072
    -255, -262.5, -270, -277.5, -285, -277.5, -270, -262.5, // 1080
    -255, -247.5, -240, -232.5, -225, -217.5, -210, -202.5, // 1088
    -195, -187.5, -180, -172.5, -165, -157.5, -150, -142.5, // 1096
    -127.5, -112.5, -97.5, -82.5, -67.5, -52.5, -37.5, -22.5, // 1104
    -7.5, 7.5, 22.5, 37.5, 52.5, 67.5, 82.5, 97.5, // 1112
    112.5, 127.5, 142.5, 150, 157.5, 165, 172.5, 180, // 1120
    187.5, 195, 202.5, 210, 217.5, 225, 232.5, 240, // 1128
    247.5, 255, 262.5, 270, 277.5, 300, 292.5, 285, // 1136
    277.5, 270, 262.5, 255, 247.5, 240, 232.5, 225, // 1144
    217.5, 210, 202.5, 195, 187.5, 180, 172.5, 165, // 1152
    157.5, 150, 135, 120, 105, 90, 75, 60, // 1160
    45, 30, 15, 0, -15, -30, -45, -60, // 1168
    -75, -90, -105, -120, -135, -150, -157.5, -165, // 1176
    -172.5, -180, -187.5, -195, -202.5, -210, -217.5, -225, // 1184
    -232.5, -240, -247.5, -255, -262.5, -270, -277.5, -285, // 1192
    -292.5, -300, -292.5, -285, -277.5, -270, -262.5, -255, // 1200
    -247.5, -240, -232.5, -225, -217.5, -210, -202.5, -195, // 1208
    -187.5, -180, -172.5, -165, -157.5, -150, -135, -120, // 1216
    -105, -90, -75, -60, -45, -30, -15, 0, // 1224
    15, 30, 45, 60, 75, 90, 105, 120, // 1232
    135, 150, 157.5, 165, 172.5, 180, 187.5, 195, // 1240
    202.5, 210, 217.5, 225, 232.5, 240, 247.5, 255, // 1248
    262.5, 270, 277.5, 285, 292.5, 315, 307.5, 300, // 1256
    292.5, 285, 277.5, 270, 262.5, 255, 247.5, 240, // 1264
    232.5, 225, 217.5, 210, 202.5, 195, 187.5, 180, // 1272
    172.5, 165, 157.5, 142.5, 127.5, 112.5, 97.5, 82.5, // 1280
    67.5, 52.5, 37.5, 22.5, 7.5, -7.5, -22.5, -37.5, // 1288
    -52.5, -67.5, -82.5, -97.5, -112.5, -127.5, -142.5, -157.5, // 1296
    -165, -172.5, -180, -187.5, -195, -202.5, -210, -217.5, // 1304
    -225, -232.5, -240, -247.5, -255, -262.5, -270, -277.5, // 1312
    -285, -292.5, -300, -307.5, -315, -307.5, -300, -292.5, // 1320
    -285, -277.5, -270, -262.5, -255, -247.5, -240, -232.5, // 1328
    -225, -217.5, -210, -202.5, -195, -187.5, -180, -172.5, // 1336
    -165, -157.5, -142.5, -127.5, -112.5, -97.5, -82.5, -67.5, // 1344
    -52.5, -37.5, -22.5, -7.5, 7.5, 22.5, 37.5, 52.5, // 1352
    67.5, 82.5, 97.5, 112.5, 127.5, 142.5, 157.5, 165, // 1360
    172.5, 180, 187.5, 195, 202.5, 210, 217.5, 225, // 1368
    232.5, 240, 247.5, 255, 262.5, 270, 277.5, 285, // 1376
    292.5, 300, 307.5, 330, 322.5, 315, 307.5, 300, // 1384
    292.5, 285, 277.5, 270, 262.5, 255, 247.5, 240, // 1392
    232.5, 225, 217.5, 210, 202.5, 195, 187.5, 180, // 1400
    172.5, 165, 150, 135, 120, 105, 90, 75, // 1408
    60, 45, 30, 15, 0, -15, -30, -45, // 1416
    -60, -75, -90, -105, -120, -135, -150, -165, // 1424
    -172.5, -180, -187.5, -195, -202.5, -210, -217.5, -225, // 1432
    -232.5, -240, -247.5, -255, -262.5, -270, -277.5, -285, // 1440
    -292.5, -300, -307.5, -315, -322.5, -330, -322.5, -315, // 1448
    -307.5, -300, -292.5, -285, -277.5, -270, -262.5, -255, // 1456
    -247.5, -240, -232.5, -225, -217.5, -210, -202.5, -195, // 1464
    -187.5, -180, -172.5, -165, -150, -135, -120, -105, // 1472
    -90, -75, -60, -45, -30, -15, 0, 15, // 1480
    30, 45, 60, 75, 90, 105, 120, 135, // 1488
    150, 165, 172.5, 180, 187.5, 195, 202.5, 210, // 1496
    217.5, 225, 232.5, 240, 247.5, 255, 262.5, 270, // 1504
    277.5, 285, 292.5, 300, 307.5, 315, 322.5, 345, // 1512
    337.5, 330, 322.5, 315, 307.5, 300, 292.5, 285, // 1520
    277.5, 270, 262.5, 255, 247.5, 240, 232.5, 225, // 1528
    217.5, 210, 202.5, 195, 187.5, 180, 172.5, 157.5, // 1536
    142.5, 127.5, 112.5, 97.5, 82.5, 67.5, 52.5, 37.5, // 1544
    22.5, 7.5, -7.5, -22.5, -37.5, -52.5, -67.5, -82.5, // 1552
    -97.5, -112.5, -127.5, -142.5, -157.5, -172.5, -180, -187.5, // 1560
    -195, -202.5, -210, -217.5, -225, -232.5, -240, -247.5, // 1568
    -255, -262.5, -270, -277.5, -285, -292.5, -300, -307.5, // 1576
    -315, -322.5, -330, -337.5, -345, -337.5, -330, -322.5, // 1584
    -315, -307.5, -300, -292.5, -285, -277.5, -270, -262.5, // 1592
    -255, -247.5, -240, -232.5, -225, -217.5, -210, -202.5, // 1600
    -195, -187.5, -180, -172.5, -157.5, -142.5, -127.5, -112.5, // 1608
    -97.5, -82.5, -67.5, -52.5, -37.5, -22.5, -7.5, 7.5, // 1616
    22.5, 37.5, 52.5, 67.5, 82.5, 97.5, 112.5, 127.5, // 1624
    142.5, 157.5, 172.5, 180, 187.5, 195, 202.5, 210, // 1632
    217.5, 225, 232.5, 240, 247.5, 255, 262.5, 270, // 1640
    277.5, 285, 292.5, 300, 307.5, 315, 322.5, 330, // 1648
    337.5, 360, 345, 330, 315, 300, 285, 270, // 1656
    255, 240, 225, 210, 195, 165, 135, 105, // 1664
    75, 45, 15, -15, -45, -75, -105, -135, // 1672
    -165, -195, -210, -225, -240, -255, -270, -285, // 1680
    -300, -315, -330, -345, -360, -360, -345, -330, // 1688
    -315, -300, -285, -270, -255, -240, -225, -210, // 1696
    -195, -165, -135, -105, -75, -45, -15, 15, // 1704
    45, 75, 105, 135, 165, 195, 210, 225, // 1712
    240, 255, 270, 285, 300, 315, 330, 345, // 1720
    360, 390, 375, 360, 345, 330, 315, 300, // 1728
    285, 270, 255, 240, 225, 210, 180, 150, // 1736
    120, 90, 60, 30, 0, -30, -60, -90, // 1744
    -120, -150, -180, -210, -225, -240, -255, -270, // 1752
    -285, -300, -315, -330, -345, -360, -375, -390, // 1760
    -390, -375, -360, -345, -330, -315, -300, -285, // 1768
    -270, -255, -240, -225, -210, -180, -150, -120, // 1776
    -90, -60, -30, 0, 30, 60, 90, 120, // 1784
    150, 180, 210, 225, 240, 255, 270, 285, // 1792
    300, 315, 330, 345, 360, 375, 390, 420, // 1800
    405, 390, 375, 360, 345, 330, 315, 300, // 1808
    285, 270, 255, 240, 225, 195, 165, 135, // 1816
    105, 75, 45, 15, -15, -45, -75, -105, // 1824
    -135, -165, -195, -225, -240, -255, -270, -285, // 1832
    -300, -315, -330, -345, -360, -375, -390, -405, // 1840
    -420, -420, -405, -390, -375, -360, -345, -330, // 1848
    -315, -300, -285, -270, -255, -240, -225, -195, // 1856
    -165, -135, -105, -75, -45, -15, 15, 45, // 1864
    75, 105, 135, 165, 195, 225, 240, 255, // 1872
    270, 285, 300, 315, 330, 345, 360, 375, // 1880
    390, 405, 420, 450, 435, 420, 405, 390, // 1888
    375, 360, 345, 330, 315, 300, 285, 270, // 1896
    255, 240, 210, 180, 150, 120, 90, 60, // 1904
    30, 0, -30, -60, -90, -120, -150, -180, // 1912
    -210, -240, -255, -270, -285, -300, -315, -330, // 1920
    -345, -360, -375, -390, -405, -420, -435, -450, // 1928
    -450, -435, -420, -405, -390, -375, -360, -345, // 1936
    -330, -315, -300, -285, -270, -255, -240, -210, // 1944
    -180, -150, -120, -90, -60, -30, 0, 30, // 1952
    60, 90, 120, 150, 180, 210, 240, 255, // 1960
    270, 285, 300, 315, 330, 345, 360, 375, // 1968
    390, 405, 420, 435, 450, 480, 465, 450, // 1976
    435, 420, 405, 390, 375, 360, 345, 330, // 1984
    315, 300, 285, 270, 255, 225, 195, 165, // 1992
    135, 105, 75, 45, 15, -15, -45, -75, // 2000
    -105, -135, -165, -195, -225, -255, -270, -285, // 2008
    -300, -315, -330, -345, -360, -375, -390, -405, // 2016
    -420, -435, -450, -465, -480, -480, -465, -450, // 2024
    -435, -420, -405, -390, -375, -360, -345, -330, // 2032
    -315, -300, -285, -270, -255, -225, -195, -165, // 2040
    -135, -105, -75, -45, -15, 15, 45, 75, // 2048
    105, 135, 165, 195, 225, 255, 270, 285, // 2056
    300, 315, 330, 345, 360, 375, 390, 405, // 2064
    420, 435, 450, 465, 480, 510, 495, 480, // 2072
    465, 450, 435, 420, 405, 390, 375, 360, // 2080
    345, 330, 315, 300, 285, 270, 240, 210, // 2088
    180, 150, 120, 90, 60, 30, 0, -30, // 2096
    -60, -90, -120, -150, -180, -210, -240, -270, // 2104
    -285, -300, -315, -330, -345, -360, -375, -390, // 2112
    -405, -420, -435, -450, -465, -480, -495, -510, // 2120
    -510, -495, -480, -465, -450, -435, -420, -405, // 2128
    -390, -375, -360, -345, -330, -315, -300, -285, // 2136
    -270, -240, -210, -180, -150, -120, -90, -60, // 2144
    -30, 0, 30, 60, 90, 120, 150, 180, // 2152
    210, 240, 270, 285, 300, 315, 330, 345, // 2160
    360, 375, 390, 405, 420, 435, 450, 465, // 2168
    480, 495, 510, 540, 525, 510, 495, 480, // 2176
    465, 450, 435, 420, 405, 390, 375, 360, // 2184
    345, 330, 315, 300, 285, 255, 225, 195, // 2192
    165, 135, 105, 75, 45, 15, -15, -45, // 2200
    -75, -105, -135, -165, -195, -225, -255, -285, // 2208
    -300, -315, -330, -345, -360, -375, -390, -405, // 2216
    -420, -435, -450, -465, -480, -495, -510, -525, // 2224
    -540, -540, -525, -510, -495, -480, -465, -450, // 2232
    -435, -420, -405, -390, -375, -360, -345, -330, // 2240
    -315, -300, -285, -255, -225, -195, -165, -135, // 2248
    -105, -75, -45, -15, 15, 45, 75, 105, // 2256
    135, 165, 195, 225, 255, 285, 300, 315, // 2264
    330, 345, 360, 375, 390, 405, 420, 435, // 2272
    450, 465, 480, 495, 510, 525, 540, 570, // 2280
    555, 540, 525, 510, 495, 480, 465, 450, // 2288
    435, 420, 405, 390, 375, 360, 345, 330, // 2296
    315, 300, 270, 240, 210, 180, 150, 120, // 2304
    90, 60, 30, 0, -30, -60, -90, -120, // 2312
    -150, -180, -210, -240, -270, -300, -315, -330, // 2320
    -345, -360, -375, -390, -405, -420, -435, -450, // 2328
    -465, -480, -495, -510, -525, -540, -555, -570, // 2336
    -570, -555, -540, -525, -510, -495, -480, -465, // 2344
    -450, -435, -420, -405, -390, -375, -360, -345, // 2352
    -330, -315, -300, -270, -240, -210, -180, -150, // 2360
    -120, -90, -60, -30, 0, 30, 60, 90, // 2368
    120, 150, 180, 210, 240, 270, 300, 315, // 2376
    330, 345, 360, 375, 390, 405, 420, 435, // 2384
    450, 465, 480, 495, 510, 525, 540, 555, // 2392
    570                        //2400
  };

  const float ytemp[2401] = {
    0, 0, 12.9904, 12.9904, -0, -12.9904, -12.9904, 0, // 0
    12.9904, 25.9808, 25.9808, 25.9808, 12.9904, -0, -12.9904, -25.9808, // 8
    -25.9808, -25.9808, -12.9904, 0, 12.9904, 25.9808, 38.9711, 38.9711, // 16
    38.9711, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 24
    -38.9711, -38.9711, -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, // 32
    38.9711, 51.9615, 51.9615, 51.9615, 51.9615, 51.9615, 38.9711, 25.9808, // 40
    12.9904, -0, -12.9904, -25.9808, -38.9711, -51.9615, -51.9615, -51.9615, // 48
    -51.9615, -51.9615, -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, // 56
    38.9711, 51.9615, 64.9519, 64.9519, 64.9519, 64.9519, 64.9519, 64.9519, // 64
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 72
    -51.9615, -64.9519, -64.9519, -64.9519, -64.9519, -64.9519, -64.9519, -51.9615, // 80
    -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, 38.9711, 51.9615, // 88
    64.9519, 77.9423, 77.9423, 77.9423, 77.9423, 77.9423, 77.9423, 77.9423, // 96
    64.9519, 51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, // 104
    -38.9711, -51.9615, -64.9519, -77.9423, -77.9423, -77.9423, -77.9423, -77.9423, // 112
    -77.9423, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, -12.9904, 0, // 120
    12.9904, 25.9808, 38.9711, 51.9615, 64.9519, 77.9423, 90.9327, 90.9327, // 128
    90.9327, 90.9327, 90.9327, 90.9327, 90.9327, 90.9327, 77.9423, 64.9519, // 136
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 144
    -51.9615, -64.9519, -77.9423, -90.9327, -90.9327, -90.9327, -90.9327, -90.9327, // 152
    -90.9327, -90.9327, -90.9327, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, // 160
    -12.9904, 0, 12.9904, 25.9808, 38.9711, 51.9615, 64.9519, 77.9423, // 168
    90.9327, 103.923, 103.923, 103.923, 103.923, 103.923, 103.923, 103.923, // 176
    103.923, 103.923, 90.9327, 77.9423, 64.9519, 51.9615, 38.9711, 25.9808, // 184
    12.9904, -0, -12.9904, -25.9808, -38.9711, -51.9615, -64.9519, -77.9423, // 192
    -90.9327, -103.923, -103.923, -103.923, -103.923, -103.923, -103.923, -103.923, // 200
    -103.923, -103.923, -90.9327, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, // 208
    -12.9904, 0, 12.9904, 25.9808, 38.9711, 51.9615, 64.9519, 77.9423, // 216
    90.9327, 103.923, 116.913, 116.913, 116.913, 116.913, 116.913, 116.913, // 224
    116.913, 116.913, 116.913, 116.913, 103.923, 90.9327, 77.9423, 64.9519, // 232
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 240
    -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -116.913, -116.913, // 248
    -116.913, -116.913, -116.913, -116.913, -116.913, -116.913, -116.913, -103.923, // 256
    -90.9327, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, -12.9904, 0, // 264
    12.9904, 25.9808, 38.9711, 51.9615, 64.9519, 77.9423, 90.9327, 103.923, // 272
    116.913, 129.904, 129.904, 129.904, 129.904, 129.904, 129.904, 129.904, // 280
    129.904, 129.904, 129.904, 129.904, 116.913, 103.923, 90.9327, 77.9423, // 288
    64.9519, 51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, // 296
    -38.9711, -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, // 304
    -129.904, -129.904, -129.904, -129.904, -129.904, -129.904, -129.904, -129.904, // 312
    -129.904, -129.904, -116.913, -103.923, -90.9327, -77.9423, -64.9519, -51.9615, // 320
    -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, 38.9711, 51.9615, // 328
    64.9519, 77.9423, 90.9327, 103.923, 116.913, 129.904, 142.894, 142.894, // 336
    142.894, 142.894, 142.894, 142.894, 142.894, 142.894, 142.894, 142.894, // 344
    142.894, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, 64.9519, // 352
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 360
    -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, -142.894, // 368
    -142.894, -142.894, -142.894, -142.894, -142.894, -142.894, -142.894, -142.894, // 376
    -142.894, -142.894, -142.894, -129.904, -116.913, -103.923, -90.9327, -77.9423, // 384
    -64.9519, -51.9615, -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, // 392
    38.9711, 51.9615, 64.9519, 77.9423, 90.9327, 103.923, 116.913, 129.904, // 400
    142.894, 155.885, 155.885, 155.885, 155.885, 155.885, 155.885, 155.885, // 408
    155.885, 155.885, 155.885, 155.885, 155.885, 155.885, 142.894, 129.904, // 416
    116.913, 103.923, 90.9327, 77.9423, 64.9519, 51.9615, 38.9711, 25.9808, // 424
    12.9904, -0, -12.9904, -25.9808, -38.9711, -51.9615, -64.9519, -77.9423, // 432
    -90.9327, -103.923, -116.913, -129.904, -142.894, -155.885, -155.885, -155.885, // 440
    -155.885, -155.885, -155.885, -155.885, -155.885, -155.885, -155.885, -155.885, // 448
    -155.885, -155.885, -142.894, -129.904, -116.913, -103.923, -90.9327, -77.9423, // 456
    -64.9519, -51.9615, -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, // 464
    38.9711, 51.9615, 64.9519, 77.9423, 90.9327, 103.923, 116.913, 129.904, // 472
    142.894, 155.885, 168.875, 168.875, 168.875, 168.875, 168.875, 168.875, // 480
    168.875, 168.875, 168.875, 168.875, 168.875, 168.875, 168.875, 168.875, // 488
    155.885, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, 64.9519, // 496
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 504
    -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, -142.894, // 512
    -155.885, -168.875, -168.875, -168.875, -168.875, -168.875, -168.875, -168.875, // 520
    -168.875, -168.875, -168.875, -168.875, -168.875, -168.875, -168.875, -155.885, // 528
    -142.894, -129.904, -116.913, -103.923, -90.9327, -77.9423, -64.9519, -51.9615, // 536
    -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, 38.9711, 51.9615, // 544
    64.9519, 77.9423, 90.9327, 103.923, 116.913, 129.904, 142.894, 155.885, // 552
    168.875, 181.865, 181.865, 181.865, 181.865, 181.865, 181.865, 181.865, // 560
    181.865, 181.865, 181.865, 181.865, 181.865, 181.865, 181.865, 181.865, // 568
    168.875, 155.885, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, // 576
    64.9519, 51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, // 584
    -38.9711, -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, // 592
    -142.894, -155.885, -168.875, -181.865, -181.865, -181.865, -181.865, -181.865, // 600
    -181.865, -181.865, -181.865, -181.865, -181.865, -181.865, -181.865, -181.865, // 608
    -181.865, -181.865, -168.875, -155.885, -142.894, -129.904, -116.913, -103.923, // 616
    -90.9327, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, -12.9904, 0, // 624
    12.9904, 25.9808, 38.9711, 51.9615, 64.9519, 77.9423, 90.9327, 103.923, // 632
    116.913, 129.904, 142.894, 155.885, 168.875, 181.865, 194.856, 194.856, // 640
    194.856, 194.856, 194.856, 194.856, 194.856, 194.856, 194.856, 194.856, // 648
    194.856, 194.856, 194.856, 194.856, 194.856, 194.856, 181.865, 168.875, // 656
    155.885, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, 64.9519, // 664
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 672
    -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, -142.894, // 680
    -155.885, -168.875, -181.865, -194.856, -194.856, -194.856, -194.856, -194.856, // 688
    -194.856, -194.856, -194.856, -194.856, -194.856, -194.856, -194.856, -194.856, // 696
    -194.856, -194.856, -194.856, -181.865, -168.875, -155.885, -142.894, -129.904, // 704
    -116.913, -103.923, -90.9327, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, // 712
    -12.9904, 0, 12.9904, 25.9808, 38.9711, 51.9615, 64.9519, 77.9423, // 720
    90.9327, 103.923, 116.913, 129.904, 142.894, 155.885, 168.875, 181.865, // 728
    194.856, 207.846, 207.846, 207.846, 207.846, 207.846, 207.846, 207.846, // 736
    207.846, 207.846, 207.846, 207.846, 207.846, 207.846, 207.846, 207.846, // 744
    207.846, 207.846, 194.856, 181.865, 168.875, 155.885, 142.894, 129.904, // 752
    116.913, 103.923, 90.9327, 77.9423, 64.9519, 51.9615, 38.9711, 25.9808, // 760
    12.9904, -0, -12.9904, -25.9808, -38.9711, -51.9615, -64.9519, -77.9423, // 768
    -90.9327, -103.923, -116.913, -129.904, -142.894, -155.885, -168.875, -181.865, // 776
    -194.856, -207.846, -207.846, -207.846, -207.846, -207.846, -207.846, -207.846, // 784
    -207.846, -207.846, -207.846, -207.846, -207.846, -207.846, -207.846, -207.846, // 792
    -207.846, -207.846, -194.856, -181.865, -168.875, -155.885, -142.894, -129.904, // 800
    -116.913, -103.923, -90.9327, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, // 808
    -12.9904, 0, 12.9904, 25.9808, 38.9711, 51.9615, 64.9519, 77.9423, // 816
    90.9327, 103.923, 116.913, 129.904, 142.894, 155.885, 168.875, 181.865, // 824
    194.856, 207.846, 220.836, 220.836, 220.836, 220.836, 220.836, 220.836, // 832
    220.836, 220.836, 220.836, 220.836, 220.836, 220.836, 220.836, 220.836, // 840
    220.836, 220.836, 220.836, 220.836, 207.846, 194.856, 181.865, 168.875, // 848
    155.885, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, 64.9519, // 856
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 864
    -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, -142.894, // 872
    -155.885, -168.875, -181.865, -194.856, -207.846, -220.836, -220.836, -220.836, // 880
    -220.836, -220.836, -220.836, -220.836, -220.836, -220.836, -220.836, -220.836, // 888
    -220.836, -220.836, -220.836, -220.836, -220.836, -220.836, -220.836, -207.846, // 896
    -194.856, -181.865, -168.875, -155.885, -142.894, -129.904, -116.913, -103.923, // 904
    -90.9327, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, -12.9904, 0, // 912
    12.9904, 25.9808, 38.9711, 51.9615, 64.9519, 77.9423, 90.9327, 103.923, // 920
    116.913, 129.904, 142.894, 155.885, 168.875, 181.865, 194.856, 207.846, // 928
    220.836, 233.827, 233.827, 233.827, 233.827, 233.827, 233.827, 233.827, // 936
    233.827, 233.827, 233.827, 233.827, 233.827, 233.827, 233.827, 233.827, // 944
    233.827, 233.827, 233.827, 233.827, 220.836, 207.846, 194.856, 181.865, // 952
    168.875, 155.885, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, // 960
    64.9519, 51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, // 968
    -38.9711, -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, // 976
    -142.894, -155.885, -168.875, -181.865, -194.856, -207.846, -220.836, -233.827, // 984
    -233.827, -233.827, -233.827, -233.827, -233.827, -233.827, -233.827, -233.827, // 992
    -233.827, -233.827, -233.827, -233.827, -233.827, -233.827, -233.827, -233.827, // 1000
    -233.827, -233.827, -220.836, -207.846, -194.856, -181.865, -168.875, -155.885, // 1008
    -142.894, -129.904, -116.913, -103.923, -90.9327, -77.9423, -64.9519, -51.9615, // 1016
    -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, 38.9711, 51.9615, // 1024
    64.9519, 77.9423, 90.9327, 103.923, 116.913, 129.904, 142.894, 155.885, // 1032
    168.875, 181.865, 194.856, 207.846, 220.836, 233.827, 246.817, 246.817, // 1040
    246.817, 246.817, 246.817, 246.817, 246.817, 246.817, 246.817, 246.817, // 1048
    246.817, 246.817, 246.817, 246.817, 246.817, 246.817, 246.817, 246.817, // 1056
    246.817, 246.817, 233.827, 220.836, 207.846, 194.856, 181.865, 168.875, // 1064
    155.885, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, 64.9519, // 1072
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 1080
    -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, -142.894, // 1088
    -155.885, -168.875, -181.865, -194.856, -207.846, -220.836, -233.827, -246.817, // 1096
    -246.817, -246.817, -246.817, -246.817, -246.817, -246.817, -246.817, -246.817, // 1104
    -246.817, -246.817, -246.817, -246.817, -246.817, -246.817, -246.817, -246.817, // 1112
    -246.817, -246.817, -246.817, -233.827, -220.836, -207.846, -194.856, -181.865, // 1120
    -168.875, -155.885, -142.894, -129.904, -116.913, -103.923, -90.9327, -77.9423, // 1128
    -64.9519, -51.9615, -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, // 1136
    38.9711, 51.9615, 64.9519, 77.9423, 90.9327, 103.923, 116.913, 129.904, // 1144
    142.894, 155.885, 168.875, 181.865, 194.856, 207.846, 220.836, 233.827, // 1152
    246.817, 259.808, 259.808, 259.808, 259.808, 259.808, 259.808, 259.808, // 1160
    259.808, 259.808, 259.808, 259.808, 259.808, 259.808, 259.808, 259.808, // 1168
    259.808, 259.808, 259.808, 259.808, 259.808, 259.808, 246.817, 233.827, // 1176
    220.836, 207.846, 194.856, 181.865, 168.875, 155.885, 142.894, 129.904, // 1184
    116.913, 103.923, 90.9327, 77.9423, 64.9519, 51.9615, 38.9711, 25.9808, // 1192
    12.9904, -0, -12.9904, -25.9808, -38.9711, -51.9615, -64.9519, -77.9423, // 1200
    -90.9327, -103.923, -116.913, -129.904, -142.894, -155.885, -168.875, -181.865, // 1208
    -194.856, -207.846, -220.836, -233.827, -246.817, -259.808, -259.808, -259.808, // 1216
    -259.808, -259.808, -259.808, -259.808, -259.808, -259.808, -259.808, -259.808, // 1224
    -259.808, -259.808, -259.808, -259.808, -259.808, -259.808, -259.808, -259.808, // 1232
    -259.808, -259.808, -246.817, -233.827, -220.836, -207.846, -194.856, -181.865, // 1240
    -168.875, -155.885, -142.894, -129.904, -116.913, -103.923, -90.9327, -77.9423, // 1248
    -64.9519, -51.9615, -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, // 1256
    38.9711, 51.9615, 64.9519, 77.9423, 90.9327, 103.923, 116.913, 129.904, // 1264
    142.894, 155.885, 168.875, 181.865, 194.856, 207.846, 220.836, 233.827, // 1272
    246.817, 259.808, 272.798, 272.798, 272.798, 272.798, 272.798, 272.798, // 1280
    272.798, 272.798, 272.798, 272.798, 272.798, 272.798, 272.798, 272.798, // 1288
    272.798, 272.798, 272.798, 272.798, 272.798, 272.798, 272.798, 272.798, // 1296
    259.808, 246.817, 233.827, 220.836, 207.846, 194.856, 181.865, 168.875, // 1304
    155.885, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, 64.9519, // 1312
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 1320
    -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, -142.894, // 1328
    -155.885, -168.875, -181.865, -194.856, -207.846, -220.836, -233.827, -246.817, // 1336
    -259.808, -272.798, -272.798, -272.798, -272.798, -272.798, -272.798, -272.798, // 1344
    -272.798, -272.798, -272.798, -272.798, -272.798, -272.798, -272.798, -272.798, // 1352
    -272.798, -272.798, -272.798, -272.798, -272.798, -272.798, -272.798, -259.808, // 1360
    -246.817, -233.827, -220.836, -207.846, -194.856, -181.865, -168.875, -155.885, // 1368
    -142.894, -129.904, -116.913, -103.923, -90.9327, -77.9423, -64.9519, -51.9615, // 1376
    -38.9711, -25.9808, -12.9904, 0, 12.9904, 25.9808, 38.9711, 51.9615, // 1384
    64.9519, 77.9423, 90.9327, 103.923, 116.913, 129.904, 142.894, 155.885, // 1392
    168.875, 181.865, 194.856, 207.846, 220.836, 233.827, 246.817, 259.808, // 1400
    272.798, 285.788, 285.788, 285.788, 285.788, 285.788, 285.788, 285.788, // 1408
    285.788, 285.788, 285.788, 285.788, 285.788, 285.788, 285.788, 285.788, // 1416
    285.788, 285.788, 285.788, 285.788, 285.788, 285.788, 285.788, 285.788, // 1424
    272.798, 259.808, 246.817, 233.827, 220.836, 207.846, 194.856, 181.865, // 1432
    168.875, 155.885, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, // 1440
    64.9519, 51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, // 1448
    -38.9711, -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, // 1456
    -142.894, -155.885, -168.875, -181.865, -194.856, -207.846, -220.836, -233.827, // 1464
    -246.817, -259.808, -272.798, -285.788, -285.788, -285.788, -285.788, -285.788, // 1472
    -285.788, -285.788, -285.788, -285.788, -285.788, -285.788, -285.788, -285.788, // 1480
    -285.788, -285.788, -285.788, -285.788, -285.788, -285.788, -285.788, -285.788, // 1488
    -285.788, -285.788, -272.798, -259.808, -246.817, -233.827, -220.836, -207.846, // 1496
    -194.856, -181.865, -168.875, -155.885, -142.894, -129.904, -116.913, -103.923, // 1504
    -90.9327, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, -12.9904, 0, // 1512
    12.9904, 25.9808, 38.9711, 51.9615, 64.9519, 77.9423, 90.9327, 103.923, // 1520
    116.913, 129.904, 142.894, 155.885, 168.875, 181.865, 194.856, 207.846, // 1528
    220.836, 233.827, 246.817, 259.808, 272.798, 285.788, 298.779, 298.779, // 1536
    298.779, 298.779, 298.779, 298.779, 298.779, 298.779, 298.779, 298.779, // 1544
    298.779, 298.779, 298.779, 298.779, 298.779, 298.779, 298.779, 298.779, // 1552
    298.779, 298.779, 298.779, 298.779, 298.779, 298.779, 285.788, 272.798, // 1560
    259.808, 246.817, 233.827, 220.836, 207.846, 194.856, 181.865, 168.875, // 1568
    155.885, 142.894, 129.904, 116.913, 103.923, 90.9327, 77.9423, 64.9519, // 1576
    51.9615, 38.9711, 25.9808, 12.9904, -0, -12.9904, -25.9808, -38.9711, // 1584
    -51.9615, -64.9519, -77.9423, -90.9327, -103.923, -116.913, -129.904, -142.894, // 1592
    -155.885, -168.875, -181.865, -194.856, -207.846, -220.836, -233.827, -246.817, // 1600
    -259.808, -272.798, -285.788, -298.779, -298.779, -298.779, -298.779, -298.779, // 1608
    -298.779, -298.779, -298.779, -298.779, -298.779, -298.779, -298.779, -298.779, // 1616
    -298.779, -298.779, -298.779, -298.779, -298.779, -298.779, -298.779, -298.779, // 1624
    -298.779, -298.779, -298.779, -285.788, -272.798, -259.808, -246.817, -233.827, // 1632
    -220.836, -207.846, -194.856, -181.865, -168.875, -155.885, -142.894, -129.904, // 1640
    -116.913, -103.923, -90.9327, -77.9423, -64.9519, -51.9615, -38.9711, -25.9808, // 1648
    -12.9904, 17.3205, 43.3013, 69.282, 95.2628, 121.244, 147.224, 173.205, // 1656
    199.186, 225.167, 251.147, 277.128, 303.109, 320.429, 320.429, 320.429, // 1664
    320.429, 320.429, 320.429, 320.429, 320.429, 320.429, 320.429, 320.429, // 1672
    320.429, 303.109, 277.128, 251.147, 225.167, 199.186, 173.205, 147.224, // 1680
    121.244, 95.2628, 69.282, 43.3013, 17.3205, -17.3205, -43.3013, -69.282, // 1688
    -95.2628, -121.244, -147.224, -173.205, -199.186, -225.167, -251.147, -277.128, // 1696
    -303.109, -320.429, -320.429, -320.429, -320.429, -320.429, -320.429, -320.429, // 1704
    -320.429, -320.429, -320.429, -320.429, -320.429, -303.109, -277.128, -251.147, // 1712
    -225.167, -199.186, -173.205, -147.224, -121.244, -95.2628, -69.282, -43.3013, // 1720
    -17.3205, 17.3205, 43.3013, 69.282, 95.2628, 121.244, 147.224, 173.205, // 1728
    199.186, 225.167, 251.147, 277.128, 303.109, 329.09, 346.41, 346.41, // 1736
    346.41, 346.41, 346.41, 346.41, 346.41, 346.41, 346.41, 346.41, // 1744
    346.41, 346.41, 346.41, 329.09, 303.109, 277.128, 251.147, 225.167, // 1752
    199.186, 173.205, 147.224, 121.244, 95.2628, 69.282, 43.3013, 17.3205, // 1760
    -17.3205, -43.3013, -69.282, -95.2628, -121.244, -147.224, -173.205, -199.186, // 1768
    -225.167, -251.147, -277.128, -303.109, -329.09, -346.41, -346.41, -346.41, // 1776
    -346.41, -346.41, -346.41, -346.41, -346.41, -346.41, -346.41, -346.41, // 1784
    -346.41, -346.41, -329.09, -303.109, -277.128, -251.147, -225.167, -199.186, // 1792
    -173.205, -147.224, -121.244, -95.2628, -69.282, -43.3013, -17.3205, 17.3205, // 1800
    43.3013, 69.282, 95.2628, 121.244, 147.224, 173.205, 199.186, 225.167, // 1808
    251.147, 277.128, 303.109, 329.09, 355.07, 372.391, 372.391, 372.391, // 1816
    372.391, 372.391, 372.391, 372.391, 372.391, 372.391, 372.391, 372.391, // 1824
    372.391, 372.391, 372.391, 355.07, 329.09, 303.109, 277.128, 251.147, // 1832
    225.167, 199.186, 173.205, 147.224, 121.244, 95.2628, 69.282, 43.3013, // 1840
    17.3205, -17.3205, -43.3013, -69.282, -95.2628, -121.244, -147.224, -173.205, // 1848
    -199.186, -225.167, -251.147, -277.128, -303.109, -329.09, -355.07, -372.391, // 1856
    -372.391, -372.391, -372.391, -372.391, -372.391, -372.391, -372.391, -372.391, // 1864
    -372.391, -372.391, -372.391, -372.391, -372.391, -355.07, -329.09, -303.109, // 1872
    -277.128, -251.147, -225.167, -199.186, -173.205, -147.224, -121.244, -95.2628, // 1880
    -69.282, -43.3013, -17.3205, 17.3205, 43.3013, 69.282, 95.2628, 121.244, // 1888
    147.224, 173.205, 199.186, 225.167, 251.147, 277.128, 303.109, 329.09, // 1896
    355.07, 381.051, 398.372, 398.372, 398.372, 398.372, 398.372, 398.372, // 1904
    398.372, 398.372, 398.372, 398.372, 398.372, 398.372, 398.372, 398.372, // 1912
    398.372, 381.051, 355.07, 329.09, 303.109, 277.128, 251.147, 225.167, // 1920
    199.186, 173.205, 147.224, 121.244, 95.2628, 69.282, 43.3013, 17.3205, // 1928
    -17.3205, -43.3013, -69.282, -95.2628, -121.244, -147.224, -173.205, -199.186, // 1936
    -225.167, -251.147, -277.128, -303.109, -329.09, -355.07, -381.051, -398.372, // 1944
    -398.372, -398.372, -398.372, -398.372, -398.372, -398.372, -398.372, -398.372, // 1952
    -398.372, -398.372, -398.372, -398.372, -398.372, -398.372, -381.051, -355.07, // 1960
    -329.09, -303.109, -277.128, -251.147, -225.167, -199.186, -173.205, -147.224, // 1968
    -121.244, -95.2628, -69.282, -43.3013, -17.3205, 17.3205, 43.3013, 69.282, // 1976
    95.2628, 121.244, 147.224, 173.205, 199.186, 225.167, 251.147, 277.128, // 1984
    303.109, 329.09, 355.07, 381.051, 407.032, 424.352, 424.352, 424.352, // 1992
    424.352, 424.352, 424.352, 424.352, 424.352, 424.352, 424.352, 424.352, // 2000
    424.352, 424.352, 424.352, 424.352, 424.352, 407.032, 381.051, 355.07, // 2008
    329.09, 303.109, 277.128, 251.147, 225.167, 199.186, 173.205, 147.224, // 2016
    121.244, 95.2628, 69.282, 43.3013, 17.3205, -17.3205, -43.3013, -69.282, // 2024
    -95.2628, -121.244, -147.224, -173.205, -199.186, -225.167, -251.147, -277.128, // 2032
    -303.109, -329.09, -355.07, -381.051, -407.032, -424.352, -424.352, -424.352, // 2040
    -424.352, -424.352, -424.352, -424.352, -424.352, -424.352, -424.352, -424.352, // 2048
    -424.352, -424.352, -424.352, -424.352, -424.352, -407.032, -381.051, -355.07, // 2056
    -329.09, -303.109, -277.128, -251.147, -225.167, -199.186, -173.205, -147.224, // 2064
    -121.244, -95.2628, -69.282, -43.3013, -17.3205, 17.3205, 43.3013, 69.282, // 2072
    95.2628, 121.244, 147.224, 173.205, 199.186, 225.167, 251.147, 277.128, // 2080
    303.109, 329.09, 355.07, 381.051, 407.032, 433.013, 450.333, 450.333, // 2088
    450.333, 450.333, 450.333, 450.333, 450.333, 450.333, 450.333, 450.333, // 2096
    450.333, 450.333, 450.333, 450.333, 450.333, 450.333, 450.333, 433.013, // 2104
    407.032, 381.051, 355.07, 329.09, 303.109, 277.128, 251.147, 225.167, // 2112
    199.186, 173.205, 147.224, 121.244, 95.2628, 69.282, 43.3013, 17.3205, // 2120
    -17.3205, -43.3013, -69.282, -95.2628, -121.244, -147.224, -173.205, -199.186, // 2128
    -225.167, -251.147, -277.128, -303.109, -329.09, -355.07, -381.051, -407.032, // 2136
    -433.013, -450.333, -450.333, -450.333, -450.333, -450.333, -450.333, -450.333, // 2144
    -450.333, -450.333, -450.333, -450.333, -450.333, -450.333, -450.333, -450.333, // 2152
    -450.333, -450.333, -433.013, -407.032, -381.051, -355.07, -329.09, -303.109, // 2160
    -277.128, -251.147, -225.167, -199.186, -173.205, -147.224, -121.244, -95.2628, // 2168
    -69.282, -43.3013, -17.3205, 17.3205, 43.3013, 69.282, 95.2628, 121.244, // 2176
    147.224, 173.205, 199.186, 225.167, 251.147, 277.128, 303.109, 329.09, // 2184
    355.07, 381.051, 407.032, 433.013, 458.993, 476.314, 476.314, 476.314, // 2192
    476.314, 476.314, 476.314, 476.314, 476.314, 476.314, 476.314, 476.314, // 2200
    476.314, 476.314, 476.314, 476.314, 476.314, 476.314, 476.314, 458.993, // 2208
    433.013, 407.032, 381.051, 355.07, 329.09, 303.109, 277.128, 251.147, // 2216
    225.167, 199.186, 173.205, 147.224, 121.244, 95.2628, 69.282, 43.3013, // 2224
    17.3205, -17.3205, -43.3013, -69.282, -95.2628, -121.244, -147.224, -173.205, // 2232
    -199.186, -225.167, -251.147, -277.128, -303.109, -329.09, -355.07, -381.051, // 2240
    -407.032, -433.013, -458.993, -476.314, -476.314, -476.314, -476.314, -476.314, // 2248
    -476.314, -476.314, -476.314, -476.314, -476.314, -476.314, -476.314, -476.314, // 2256
    -476.314, -476.314, -476.314, -476.314, -476.314, -458.993, -433.013, -407.032, // 2264
    -381.051, -355.07, -329.09, -303.109, -277.128, -251.147, -225.167, -199.186, // 2272
    -173.205, -147.224, -121.244, -95.2628, -69.282, -43.3013, -17.3205, 17.3205, // 2280
    43.3013, 69.282, 95.2628, 121.244, 147.224, 173.205, 199.186, 225.167, // 2288
    251.147, 277.128, 303.109, 329.09, 355.07, 381.051, 407.032, 433.013, // 2296
    458.993, 484.974, 502.295, 502.295, 502.295, 502.295, 502.295, 502.295, // 2304
    502.295, 502.295, 502.295, 502.295, 502.295, 502.295, 502.295, 502.295, // 2312
    502.295, 502.295, 502.295, 502.295, 502.295, 484.974, 458.993, 433.013, // 2320
    407.032, 381.051, 355.07, 329.09, 303.109, 277.128, 251.147, 225.167, // 2328
    199.186, 173.205, 147.224, 121.244, 95.2628, 69.282, 43.3013, 17.3205, // 2336
    -17.3205, -43.3013, -69.282, -95.2628, -121.244, -147.224, -173.205, -199.186, // 2344
    -225.167, -251.147, -277.128, -303.109, -329.09, -355.07, -381.051, -407.032, // 2352
    -433.013, -458.993, -484.974, -502.295, -502.295, -502.295, -502.295, -502.295, // 2360
    -502.295, -502.295, -502.295, -502.295, -502.295, -502.295, -502.295, -502.295, // 2368
    -502.295, -502.295, -502.295, -502.295, -502.295, -502.295, -484.974, -458.993, // 2376
    -433.013, -407.032, -381.051, -355.07, -329.09, -303.109, -277.128, -251.147, // 2384
    -225.167, -199.186, -173.205, -147.224, -121.244, -95.2628, -69.282, -43.3013, // 2392
    -17.3205                        //2400
  };

  const float dtemp[2401] = {
    15, 15, 15, 15, 15, 15, 15, 15, // 0
    15, 15, 15, 15, 15, 15, 15, 15, // 8
    15, 15, 15, 15, 15, 15, 15, 15, // 16
    15, 15, 15, 15, 15, 15, 15, 15, // 24
    15, 15, 15, 15, 15, 15, 15, 15, // 32
    15, 15, 15, 15, 15, 15, 15, 15, // 40
    15, 15, 15, 15, 15, 15, 15, 15, // 48
    15, 15, 15, 15, 15, 15, 15, 15, // 56
    15, 15, 15, 15, 15, 15, 15, 15, // 64
    15, 15, 15, 15, 15, 15, 15, 15, // 72
    15, 15, 15, 15, 15, 15, 15, 15, // 80
    15, 15, 15, 15, 15, 15, 15, 15, // 88
    15, 15, 15, 15, 15, 15, 15, 15, // 96
    15, 15, 15, 15, 15, 15, 15, 15, // 104
    15, 15, 15, 15, 15, 15, 15, 15, // 112
    15, 15, 15, 15, 15, 15, 15, 15, // 120
    15, 15, 15, 15, 15, 15, 15, 15, // 128
    15, 15, 15, 15, 15, 15, 15, 15, // 136
    15, 15, 15, 15, 15, 15, 15, 15, // 144
    15, 15, 15, 15, 15, 15, 15, 15, // 152
    15, 15, 15, 15, 15, 15, 15, 15, // 160
    15, 15, 15, 15, 15, 15, 15, 15, // 168
    15, 15, 15, 15, 15, 15, 15, 15, // 176
    15, 15, 15, 15, 15, 15, 15, 15, // 184
    15, 15, 15, 15, 15, 15, 15, 15, // 192
    15, 15, 15, 15, 15, 15, 15, 15, // 200
    15, 15, 15, 15, 15, 15, 15, 15, // 208
    15, 15, 15, 15, 15, 15, 15, 15, // 216
    15, 15, 15, 15, 15, 15, 15, 15, // 224
    15, 15, 15, 15, 15, 15, 15, 15, // 232
    15, 15, 15, 15, 15, 15, 15, 15, // 240
    15, 15, 15, 15, 15, 15, 15, 15, // 248
    15, 15, 15, 15, 15, 15, 15, 15, // 256
    15, 15, 15, 15, 15, 15, 15, 15, // 264
    15, 15, 15, 15, 15, 15, 15, 15, // 272
    15, 15, 15, 15, 15, 15, 15, 15, // 280
    15, 15, 15, 15, 15, 15, 15, 15, // 288
    15, 15, 15, 15, 15, 15, 15, 15, // 296
    15, 15, 15, 15, 15, 15, 15, 15, // 304
    15, 15, 15, 15, 15, 15, 15, 15, // 312
    15, 15, 15, 15, 15, 15, 15, 15, // 320
    15, 15, 15, 15, 15, 15, 15, 15, // 328
    15, 15, 15, 15, 15, 15, 15, 15, // 336
    15, 15, 15, 15, 15, 15, 15, 15, // 344
    15, 15, 15, 15, 15, 15, 15, 15, // 352
    15, 15, 15, 15, 15, 15, 15, 15, // 360
    15, 15, 15, 15, 15, 15, 15, 15, // 368
    15, 15, 15, 15, 15, 15, 15, 15, // 376
    15, 15, 15, 15, 15, 15, 15, 15, // 384
    15, 15, 15, 15, 15, 15, 15, 15, // 392
    15, 15, 15, 15, 15, 15, 15, 15, // 400
    15, 15, 15, 15, 15, 15, 15, 15, // 408
    15, 15, 15, 15, 15, 15, 15, 15, // 416
    15, 15, 15, 15, 15, 15, 15, 15, // 424
    15, 15, 15, 15, 15, 15, 15, 15, // 432
    15, 15, 15, 15, 15, 15, 15, 15, // 440
    15, 15, 15, 15, 15, 15, 15, 15, // 448
    15, 15, 15, 15, 15, 15, 15, 15, // 456
    15, 15, 15, 15, 15, 15, 15, 15, // 464
    15, 15, 15, 15, 15, 15, 15, 15, // 472
    15, 15, 15, 15, 15, 15, 15, 15, // 480
    15, 15, 15, 15, 15, 15, 15, 15, // 488
    15, 15, 15, 15, 15, 15, 15, 15, // 496
    15, 15, 15, 15, 15, 15, 15, 15, // 504
    15, 15, 15, 15, 15, 15, 15, 15, // 512
    15, 15, 15, 15, 15, 15, 15, 15, // 520
    15, 15, 15, 15, 15, 15, 15, 15, // 528
    15, 15, 15, 15, 15, 15, 15, 15, // 536
    15, 15, 15, 15, 15, 15, 15, 15, // 544
    15, 15, 15, 15, 15, 15, 15, 15, // 552
    15, 15, 15, 15, 15, 15, 15, 15, // 560
    15, 15, 15, 15, 15, 15, 15, 15, // 568
    15, 15, 15, 15, 15, 15, 15, 15, // 576
    15, 15, 15, 15, 15, 15, 15, 15, // 584
    15, 15, 15, 15, 15, 15, 15, 15, // 592
    15, 15, 15, 15, 15, 15, 15, 15, // 600
    15, 15, 15, 15, 15, 15, 15, 15, // 608
    15, 15, 15, 15, 15, 15, 15, 15, // 616
    15, 15, 15, 15, 15, 15, 15, 15, // 624
    15, 15, 15, 15, 15, 15, 15, 15, // 632
    15, 15, 15, 15, 15, 15, 15, 15, // 640
    15, 15, 15, 15, 15, 15, 15, 15, // 648
    15, 15, 15, 15, 15, 15, 15, 15, // 656
    15, 15, 15, 15, 15, 15, 15, 15, // 664
    15, 15, 15, 15, 15, 15, 15, 15, // 672
    15, 15, 15, 15, 15, 15, 15, 15, // 680
    15, 15, 15, 15, 15, 15, 15, 15, // 688
    15, 15, 15, 15, 15, 15, 15, 15, // 696
    15, 15, 15, 15, 15, 15, 15, 15, // 704
    15, 15, 15, 15, 15, 15, 15, 15, // 712
    15, 15, 15, 15, 15, 15, 15, 15, // 720
    15, 15, 15, 15, 15, 15, 15, 15, // 728
    15, 15, 15, 15, 15, 15, 15, 15, // 736
    15, 15, 15, 15, 15, 15, 15, 15, // 744
    15, 15, 15, 15, 15, 15, 15, 15, // 752
    15, 15, 15, 15, 15, 15, 15, 15, // 760
    15, 15, 15, 15, 15, 15, 15, 15, // 768
    15, 15, 15, 15, 15, 15, 15, 15, // 776
    15, 15, 15, 15, 15, 15, 15, 15, // 784
    15, 15, 15, 15, 15, 15, 15, 15, // 792
    15, 15, 15, 15, 15, 15, 15, 15, // 800
    15, 15, 15, 15, 15, 15, 15, 15, // 808
    15, 15, 15, 15, 15, 15, 15, 15, // 816
    15, 15, 15, 15, 15, 15, 15, 15, // 824
    15, 15, 15, 15, 15, 15, 15, 15, // 832
    15, 15, 15, 15, 15, 15, 15, 15, // 840
    15, 15, 15, 15, 15, 15, 15, 15, // 848
    15, 15, 15, 15, 15, 15, 15, 15, // 856
    15, 15, 15, 15, 15, 15, 15, 15, // 864
    15, 15, 15, 15, 15, 15, 15, 15, // 872
    15, 15, 15, 15, 15, 15, 15, 15, // 880
    15, 15, 15, 15, 15, 15, 15, 15, // 888
    15, 15, 15, 15, 15, 15, 15, 15, // 896
    15, 15, 15, 15, 15, 15, 15, 15, // 904
    15, 15, 15, 15, 15, 15, 15, 15, // 912
    15, 15, 15, 15, 15, 15, 15, 15, // 920
    15, 15, 15, 15, 15, 15, 15, 15, // 928
    15, 15, 15, 15, 15, 15, 15, 15, // 936
    15, 15, 15, 15, 15, 15, 15, 15, // 944
    15, 15, 15, 15, 15, 15, 15, 15, // 952
    15, 15, 15, 15, 15, 15, 15, 15, // 960
    15, 15, 15, 15, 15, 15, 15, 15, // 968
    15, 15, 15, 15, 15, 15, 15, 15, // 976
    15, 15, 15, 15, 15, 15, 15, 15, // 984
    15, 15, 15, 15, 15, 15, 15, 15, // 992
    15, 15, 15, 15, 15, 15, 15, 15, // 1000
    15, 15, 15, 15, 15, 15, 15, 15, // 1008
    15, 15, 15, 15, 15, 15, 15, 15, // 1016
    15, 15, 15, 15, 15, 15, 15, 15, // 1024
    15, 15, 15, 15, 15, 15, 15, 15, // 1032
    15, 15, 15, 15, 15, 15, 15, 15, // 1040
    15, 15, 15, 15, 15, 15, 15, 15, // 1048
    15, 15, 15, 15, 15, 15, 15, 15, // 1056
    15, 15, 15, 15, 15, 15, 15, 15, // 1064
    15, 15, 15, 15, 15, 15, 15, 15, // 1072
    15, 15, 15, 15, 15, 15, 15, 15, // 1080
    15, 15, 15, 15, 15, 15, 15, 15, // 1088
    15, 15, 15, 15, 15, 15, 15, 15, // 1096
    15, 15, 15, 15, 15, 15, 15, 15, // 1104
    15, 15, 15, 15, 15, 15, 15, 15, // 1112
    15, 15, 15, 15, 15, 15, 15, 15, // 1120
    15, 15, 15, 15, 15, 15, 15, 15, // 1128
    15, 15, 15, 15, 15, 15, 15, 15, // 1136
    15, 15, 15, 15, 15, 15, 15, 15, // 1144
    15, 15, 15, 15, 15, 15, 15, 15, // 1152
    15, 15, 15, 15, 15, 15, 15, 15, // 1160
    15, 15, 15, 15, 15, 15, 15, 15, // 1168
    15, 15, 15, 15, 15, 15, 15, 15, // 1176
    15, 15, 15, 15, 15, 15, 15, 15, // 1184
    15, 15, 15, 15, 15, 15, 15, 15, // 1192
    15, 15, 15, 15, 15, 15, 15, 15, // 1200
    15, 15, 15, 15, 15, 15, 15, 15, // 1208
    15, 15, 15, 15, 15, 15, 15, 15, // 1216
    15, 15, 15, 15, 15, 15, 15, 15, // 1224
    15, 15, 15, 15, 15, 15, 15, 15, // 1232
    15, 15, 15, 15, 15, 15, 15, 15, // 1240
    15, 15, 15, 15, 15, 15, 15, 15, // 1248
    15, 15, 15, 15, 15, 15, 15, 15, // 1256
    15, 15, 15, 15, 15, 15, 15, 15, // 1264
    15, 15, 15, 15, 15, 15, 15, 15, // 1272
    15, 15, 15, 15, 15, 15, 15, 15, // 1280
    15, 15, 15, 15, 15, 15, 15, 15, // 1288
    15, 15, 15, 15, 15, 15, 15, 15, // 1296
    15, 15, 15, 15, 15, 15, 15, 15, // 1304
    15, 15, 15, 15, 15, 15, 15, 15, // 1312
    15, 15, 15, 15, 15, 15, 15, 15, // 1320
    15, 15, 15, 15, 15, 15, 15, 15, // 1328
    15, 15, 15, 15, 15, 15, 15, 15, // 1336
    15, 15, 15, 15, 15, 15, 15, 15, // 1344
    15, 15, 15, 15, 15, 15, 15, 15, // 1352
    15, 15, 15, 15, 15, 15, 15, 15, // 1360
    15, 15, 15, 15, 15, 15, 15, 15, // 1368
    15, 15, 15, 15, 15, 15, 15, 15, // 1376
    15, 15, 15, 15, 15, 15, 15, 15, // 1384
    15, 15, 15, 15, 15, 15, 15, 15, // 1392
    15, 15, 15, 15, 15, 15, 15, 15, // 1400
    15, 15, 15, 15, 15, 15, 15, 15, // 1408
    15, 15, 15, 15, 15, 15, 15, 15, // 1416
    15, 15, 15, 15, 15, 15, 15, 15, // 1424
    15, 15, 15, 15, 15, 15, 15, 15, // 1432
    15, 15, 15, 15, 15, 15, 15, 15, // 1440
    15, 15, 15, 15, 15, 15, 15, 15, // 1448
    15, 15, 15, 15, 15, 15, 15, 15, // 1456
    15, 15, 15, 15, 15, 15, 15, 15, // 1464
    15, 15, 15, 15, 15, 15, 15, 15, // 1472
    15, 15, 15, 15, 15, 15, 15, 15, // 1480
    15, 15, 15, 15, 15, 15, 15, 15, // 1488
    15, 15, 15, 15, 15, 15, 15, 15, // 1496
    15, 15, 15, 15, 15, 15, 15, 15, // 1504
    15, 15, 15, 15, 15, 15, 15, 15, // 1512
    15, 15, 15, 15, 15, 15, 15, 15, // 1520
    15, 15, 15, 15, 15, 15, 15, 15, // 1528
    15, 15, 15, 15, 15, 15, 15, 15, // 1536
    15, 15, 15, 15, 15, 15, 15, 15, // 1544
    15, 15, 15, 15, 15, 15, 15, 15, // 1552
    15, 15, 15, 15, 15, 15, 15, 15, // 1560
    15, 15, 15, 15, 15, 15, 15, 15, // 1568
    15, 15, 15, 15, 15, 15, 15, 15, // 1576
    15, 15, 15, 15, 15, 15, 15, 15, // 1584
    15, 15, 15, 15, 15, 15, 15, 15, // 1592
    15, 15, 15, 15, 15, 15, 15, 15, // 1600
    15, 15, 15, 15, 15, 15, 15, 15, // 1608
    15, 15, 15, 15, 15, 15, 15, 15, // 1616
    15, 15, 15, 15, 15, 15, 15, 15, // 1624
    15, 15, 15, 15, 15, 15, 15, 15, // 1632
    15, 15, 15, 15, 15, 15, 15, 15, // 1640
    15, 15, 15, 15, 15, 15, 15, 15, // 1648
    15, 30, 30, 30, 30, 30, 30, 30, // 1656
    30, 30, 30, 30, 30, 30, 30, 30, // 1664
    30, 30, 30, 30, 30, 30, 30, 30, // 1672
    30, 30, 30, 30, 30, 30, 30, 30, // 1680
    30, 30, 30, 30, 30, 30, 30, 30, // 1688
    30, 30, 30, 30, 30, 30, 30, 30, // 1696
    30, 30, 30, 30, 30, 30, 30, 30, // 1704
    30, 30, 30, 30, 30, 30, 30, 30, // 1712
    30, 30, 30, 30, 30, 30, 30, 30, // 1720
    30, 30, 30, 30, 30, 30, 30, 30, // 1728
    30, 30, 30, 30, 30, 30, 30, 30, // 1736
    30, 30, 30, 30, 30, 30, 30, 30, // 1744
    30, 30, 30, 30, 30, 30, 30, 30, // 1752
    30, 30, 30, 30, 30, 30, 30, 30, // 1760
    30, 30, 30, 30, 30, 30, 30, 30, // 1768
    30, 30, 30, 30, 30, 30, 30, 30, // 1776
    30, 30, 30, 30, 30, 30, 30, 30, // 1784
    30, 30, 30, 30, 30, 30, 30, 30, // 1792
    30, 30, 30, 30, 30, 30, 30, 30, // 1800
    30, 30, 30, 30, 30, 30, 30, 30, // 1808
    30, 30, 30, 30, 30, 30, 30, 30, // 1816
    30, 30, 30, 30, 30, 30, 30, 30, // 1824
    30, 30, 30, 30, 30, 30, 30, 30, // 1832
    30, 30, 30, 30, 30, 30, 30, 30, // 1840
    30, 30, 30, 30, 30, 30, 30, 30, // 1848
    30, 30, 30, 30, 30, 30, 30, 30, // 1856
    30, 30, 30, 30, 30, 30, 30, 30, // 1864
    30, 30, 30, 30, 30, 30, 30, 30, // 1872
    30, 30, 30, 30, 30, 30, 30, 30, // 1880
    30, 30, 30, 30, 30, 30, 30, 30, // 1888
    30, 30, 30, 30, 30, 30, 30, 30, // 1896
    30, 30, 30, 30, 30, 30, 30, 30, // 1904
    30, 30, 30, 30, 30, 30, 30, 30, // 1912
    30, 30, 30, 30, 30, 30, 30, 30, // 1920
    30, 30, 30, 30, 30, 30, 30, 30, // 1928
    30, 30, 30, 30, 30, 30, 30, 30, // 1936
    30, 30, 30, 30, 30, 30, 30, 30, // 1944
    30, 30, 30, 30, 30, 30, 30, 30, // 1952
    30, 30, 30, 30, 30, 30, 30, 30, // 1960
    30, 30, 30, 30, 30, 30, 30, 30, // 1968
    30, 30, 30, 30, 30, 30, 30, 30, // 1976
    30, 30, 30, 30, 30, 30, 30, 30, // 1984
    30, 30, 30, 30, 30, 30, 30, 30, // 1992
    30, 30, 30, 30, 30, 30, 30, 30, // 2000
    30, 30, 30, 30, 30, 30, 30, 30, // 2008
    30, 30, 30, 30, 30, 30, 30, 30, // 2016
    30, 30, 30, 30, 30, 30, 30, 30, // 2024
    30, 30, 30, 30, 30, 30, 30, 30, // 2032
    30, 30, 30, 30, 30, 30, 30, 30, // 2040
    30, 30, 30, 30, 30, 30, 30, 30, // 2048
    30, 30, 30, 30, 30, 30, 30, 30, // 2056
    30, 30, 30, 30, 30, 30, 30, 30, // 2064
    30, 30, 30, 30, 30, 30, 30, 30, // 2072
    30, 30, 30, 30, 30, 30, 30, 30, // 2080
    30, 30, 30, 30, 30, 30, 30, 30, // 2088
    30, 30, 30, 30, 30, 30, 30, 30, // 2096
    30, 30, 30, 30, 30, 30, 30, 30, // 2104
    30, 30, 30, 30, 30, 30, 30, 30, // 2112
    30, 30, 30, 30, 30, 30, 30, 30, // 2120
    30, 30, 30, 30, 30, 30, 30, 30, // 2128
    30, 30, 30, 30, 30, 30, 30, 30, // 2136
    30, 30, 30, 30, 30, 30, 30, 30, // 2144
    30, 30, 30, 30, 30, 30, 30, 30, // 2152
    30, 30, 30, 30, 30, 30, 30, 30, // 2160
    30, 30, 30, 30, 30, 30, 30, 30, // 2168
    30, 30, 30, 30, 30, 30, 30, 30, // 2176
    30, 30, 30, 30, 30, 30, 30, 30, // 2184
    30, 30, 30, 30, 30, 30, 30, 30, // 2192
    30, 30, 30, 30, 30, 30, 30, 30, // 2200
    30, 30, 30, 30, 30, 30, 30, 30, // 2208
    30, 30, 30, 30, 30, 30, 30, 30, // 2216
    30, 30, 30, 30, 30, 30, 30, 30, // 2224
    30, 30, 30, 30, 30, 30, 30, 30, // 2232
    30, 30, 30, 30, 30, 30, 30, 30, // 2240
    30, 30, 30, 30, 30, 30, 30, 30, // 2248
    30, 30, 30, 30, 30, 30, 30, 30, // 2256
    30, 30, 30, 30, 30, 30, 30, 30, // 2264
    30, 30, 30, 30, 30, 30, 30, 30, // 2272
    30, 30, 30, 30, 30, 30, 30, 30, // 2280
    30, 30, 30, 30, 30, 30, 30, 30, // 2288
    30, 30, 30, 30, 30, 30, 30, 30, // 2296
    30, 30, 30, 30, 30, 30, 30, 30, // 2304
    30, 30, 30, 30, 30, 30, 30, 30, // 2312
    30, 30, 30, 30, 30, 30, 30, 30, // 2320
    30, 30, 30, 30, 30, 30, 30, 30, // 2328
    30, 30, 30, 30, 30, 30, 30, 30, // 2336
    30, 30, 30, 30, 30, 30, 30, 30, // 2344
    30, 30, 30, 30, 30, 30, 30, 30, // 2352
    30, 30, 30, 30, 30, 30, 30, 30, // 2360
    30, 30, 30, 30, 30, 30, 30, 30, // 2368
    30, 30, 30, 30, 30, 30, 30, 30, // 2376
    30, 30, 30, 30, 30, 30, 30, 30, // 2384
    30, 30, 30, 30, 30, 30, 30, 30, // 2392
    30                        //2400
  };
  
    //
    //   fill the pixels list with this data
    //

    for (UInt_t i=0; i<GetNumPixels(); i++)
        (*this)[i].Set(xtemp[i], ytemp[i], dtemp[i]);
}

// --------------------------------------------------------------------------
//
//  This fills the next neighbor information from a table into the pixel
//  objects.
//
void MGeomCamMagicHG::CreateNN()
{
  const Short_t nn[2401][6] = {       // Neighbours of  #
    {  1,  2,  3,  4,  5,  6}, // 0
    {  0,  2,  6,  7,  8,  18},
    {  0,  1,  3,  8,  9,  10},
    {  0,  2,  4,  10,  11,  12},
    {  0,  3,  5,  12,  13,  14},
    {  0,  4,  6,  14,  15,  16},
    {  0,  1,  5,  16,  17,  18},
    {  1,  8,  18,  19,  20,  36},
    {  1,  2,  7,  9,  20,  21},
    {  2,  8,  10,  21,  22,  23},
    {  2,  3,  9,  11,  23,  24},
    {  3,  10,  12,  24,  25,  26},
    {  3,  4,  11,  13,  26,  27},
    {  4,  12,  14,  27,  28,  29},
    {  4,  5,  13,  15,  29,  30},
    {  5,  14,  16,  30,  31,  32},
    {  5,  6,  15,  17,  32,  33},
    {  6,  16,  18,  33,  34,  35},
    {  1,  6,  7,  17,  35,  36},
    {  7,  20,  36,  37,  38,  60},
    {  7,  8,  19,  21,  38,  39}, // 20
    {  8,  9,  20,  22,  39,  40},
    {  9,  21,  23,  40,  41,  42},
    {  9,  10,  22,  24,  42,  43},
    {  10,  11,  23,  25,  43,  44},
    {  11,  24,  26,  44,  45,  46},
    {  11,  12,  25,  27,  46,  47},
    {  12,  13,  26,  28,  47,  48},
    {  13,  27,  29,  48,  49,  50},
    {  13,  14,  28,  30,  50,  51},
    {  14,  15,  29,  31,  51,  52},
    {  15,  30,  32,  52,  53,  54},
    {  15,  16,  31,  33,  54,  55},
    {  16,  17,  32,  34,  55,  56},
    {  17,  33,  35,  56,  57,  58},
    {  17,  18,  34,  36,  58,  59},
    {  7,  18,  19,  35,  59,  60},
    {  19,  38,  60,  61,  62,  90},
    {  19,  20,  37,  39,  62,  63},
    {  20,  21,  38,  40,  63,  64},
    {  21,  22,  39,  41,  64,  65}, // 40
    {  22,  40,  42,  65,  66,  67},
    {  22,  23,  41,  43,  67,  68},
    {  23,  24,  42,  44,  68,  69},
    {  24,  25,  43,  45,  69,  70},
    {  25,  44,  46,  70,  71,  72},
    {  25,  26,  45,  47,  72,  73},
    {  26,  27,  46,  48,  73,  74},
    {  27,  28,  47,  49,  74,  75},
    {  28,  48,  50,  75,  76,  77},
    {  28,  29,  49,  51,  77,  78},
    {  29,  30,  50,  52,  78,  79},
    {  30,  31,  51,  53,  79,  80},
    {  31,  52,  54,  80,  81,  82},
    {  31,  32,  53,  55,  82,  83},
    {  32,  33,  54,  56,  83,  84},
    {  33,  34,  55,  57,  84,  85},
    {  34,  56,  58,  85,  86,  87},
    {  34,  35,  57,  59,  87,  88},
    {  35,  36,  58,  60,  88,  89},
    {  19,  36,  37,  59,  89,  90}, // 60
    {  37,  62,  90,  91,  92,  126},
    {  37,  38,  61,  63,  92,  93},
    {  38,  39,  62,  64,  93,  94},
    {  39,  40,  63,  65,  94,  95},
    {  40,  41,  64,  66,  95,  96},
    {  41,  65,  67,  96,  97,  98},
    {  41,  42,  66,  68,  98,  99},
    {  42,  43,  67,  69,  99,  100},
    {  43,  44,  68,  70,  100,  101},
    {  44,  45,  69,  71,  101,  102},
    {  45,  70,  72,  102,  103,  104},
    {  45,  46,  71,  73,  104,  105},
    {  46,  47,  72,  74,  105,  106},
    {  47,  48,  73,  75,  106,  107},
    {  48,  49,  74,  76,  107,  108},
    {  49,  75,  77,  108,  109,  110},
    {  49,  50,  76,  78,  110,  111},
    {  50,  51,  77,  79,  111,  112},
    {  51,  52,  78,  80,  112,  113},
    {  52,  53,  79,  81,  113,  114}, // 80
    {  53,  80,  82,  114,  115,  116},
    {  53,  54,  81,  83,  116,  117},
    {  54,  55,  82,  84,  117,  118},
    {  55,  56,  83,  85,  118,  119},
    {  56,  57,  84,  86,  119,  120},
    {  57,  85,  87,  120,  121,  122},
    {  57,  58,  86,  88,  122,  123},
    {  58,  59,  87,  89,  123,  124},
    {  59,  60,  88,  90,  124,  125},
    {  37,  60,  61,  89,  125,  126},
    {  61,  92,  126,  127,  128,  168},
    {  61,  62,  91,  93,  128,  129},
    {  62,  63,  92,  94,  129,  130},
    {  63,  64,  93,  95,  130,  131},
    {  64,  65,  94,  96,  131,  132},
    {  65,  66,  95,  97,  132,  133},
    {  66,  96,  98,  133,  134,  135},
    {  66,  67,  97,  99,  135,  136},
    {  67,  68,  98,  100,  136,  137},
    {  68,  69,  99,  101,  137,  138}, // 100
    {  69,  70,  100,  102,  138,  139},
    {  70,  71,  101,  103,  139,  140},
    {  71,  102,  104,  140,  141,  142},
    {  71,  72,  103,  105,  142,  143},
    {  72,  73,  104,  106,  143,  144},
    {  73,  74,  105,  107,  144,  145},
    {  74,  75,  106,  108,  145,  146},
    {  75,  76,  107,  109,  146,  147},
    {  76,  108,  110,  147,  148,  149},
    {  76,  77,  109,  111,  149,  150},
    {  77,  78,  110,  112,  150,  151},
    {  78,  79,  111,  113,  151,  152},
    {  79,  80,  112,  114,  152,  153},
    {  80,  81,  113,  115,  153,  154},
    {  81,  114,  116,  154,  155,  156},
    {  81,  82,  115,  117,  156,  157},
    {  82,  83,  116,  118,  157,  158},
    {  83,  84,  117,  119,  158,  159},
    {  84,  85,  118,  120,  159,  160},
    {  85,  86,  119,  121,  160,  161}, // 120
    {  86,  120,  122,  161,  162,  163},
    {  86,  87,  121,  123,  163,  164},
    {  87,  88,  122,  124,  164,  165},
    {  88,  89,  123,  125,  165,  166},
    {  89,  90,  124,  126,  166,  167},
    {  61,  90,  91,  125,  167,  168},
    {  91,  128,  168,  169,  170,  216},
    {  91,  92,  127,  129,  170,  171},
    {  92,  93,  128,  130,  171,  172},
    {  93,  94,  129,  131,  172,  173},
    {  94,  95,  130,  132,  173,  174},
    {  95,  96,  131,  133,  174,  175},
    {  96,  97,  132,  134,  175,  176},
    {  97,  133,  135,  176,  177,  178},
    {  97,  98,  134,  136,  178,  179},
    {  98,  99,  135,  137,  179,  180},
    {  99,  100,  136,  138,  180,  181},
    {  100,  101,  137,  139,  181,  182},
    {  101,  102,  138,  140,  182,  183},
    {  102,  103,  139,  141,  183,  184}, // 140
    {  103,  140,  142,  184,  185,  186},
    {  103,  104,  141,  143,  186,  187},
    {  104,  105,  142,  144,  187,  188},
    {  105,  106,  143,  145,  188,  189},
    {  106,  107,  144,  146,  189,  190},
    {  107,  108,  145,  147,  190,  191},
    {  108,  109,  146,  148,  191,  192},
    {  109,  147,  149,  192,  193,  194},
    {  109,  110,  148,  150,  194,  195},
    {  110,  111,  149,  151,  195,  196},
    {  111,  112,  150,  152,  196,  197},
    {  112,  113,  151,  153,  197,  198},
    {  113,  114,  152,  154,  198,  199},
    {  114,  115,  153,  155,  199,  200},
    {  115,  154,  156,  200,  201,  202},
    {  115,  116,  155,  157,  202,  203},
    {  116,  117,  156,  158,  203,  204},
    {  117,  118,  157,  159,  204,  205},
    {  118,  119,  158,  160,  205,  206},
    {  119,  120,  159,  161,  206,  207}, // 160
    {  120,  121,  160,  162,  207,  208},
    {  121,  161,  163,  208,  209,  210},
    {  121,  122,  162,  164,  210,  211},
    {  122,  123,  163,  165,  211,  212},
    {  123,  124,  164,  166,  212,  213},
    {  124,  125,  165,  167,  213,  214},
    {  125,  126,  166,  168,  214,  215},
    {  91,  126,  127,  167,  215,  216},
    {  127,  170,  216,  217,  218,  270},
    {  127,  128,  169,  171,  218,  219},
    {  128,  129,  170,  172,  219,  220},
    {  129,  130,  171,  173,  220,  221},
    {  130,  131,  172,  174,  221,  222},
    {  131,  132,  173,  175,  222,  223},
    {  132,  133,  174,  176,  223,  224},
    {  133,  134,  175,  177,  224,  225},
    {  134,  176,  178,  225,  226,  227},
    {  134,  135,  177,  179,  227,  228},
    {  135,  136,  178,  180,  228,  229},
    {  136,  137,  179,  181,  229,  230}, // 180
    {  137,  138,  180,  182,  230,  231},
    {  138,  139,  181,  183,  231,  232},
    {  139,  140,  182,  184,  232,  233},
    {  140,  141,  183,  185,  233,  234},
    {  141,  184,  186,  234,  235,  236},
    {  141,  142,  185,  187,  236,  237},
    {  142,  143,  186,  188,  237,  238},
    {  143,  144,  187,  189,  238,  239},
    {  144,  145,  188,  190,  239,  240},
    {  145,  146,  189,  191,  240,  241},
    {  146,  147,  190,  192,  241,  242},
    {  147,  148,  191,  193,  242,  243},
    {  148,  192,  194,  243,  244,  245},
    {  148,  149,  193,  195,  245,  246},
    {  149,  150,  194,  196,  246,  247},
    {  150,  151,  195,  197,  247,  248},
    {  151,  152,  196,  198,  248,  249},
    {  152,  153,  197,  199,  249,  250},
    {  153,  154,  198,  200,  250,  251},
    {  154,  155,  199,  201,  251,  252}, // 200
    {  155,  200,  202,  252,  253,  254},
    {  155,  156,  201,  203,  254,  255},
    {  156,  157,  202,  204,  255,  256},
    {  157,  158,  203,  205,  256,  257},
    {  158,  159,  204,  206,  257,  258},
    {  159,  160,  205,  207,  258,  259},
    {  160,  161,  206,  208,  259,  260},
    {  161,  162,  207,  209,  260,  261},
    {  162,  208,  210,  261,  262,  263},
    {  162,  163,  209,  211,  263,  264},
    {  163,  164,  210,  212,  264,  265},
    {  164,  165,  211,  213,  265,  266},
    {  165,  166,  212,  214,  266,  267},
    {  166,  167,  213,  215,  267,  268},
    {  167,  168,  214,  216,  268,  269},
    {  127,  168,  169,  215,  269,  270},
    {  169,  218,  270,  271,  272,  330},
    {  169,  170,  217,  219,  272,  273},
    {  170,  171,  218,  220,  273,  274},
    {  171,  172,  219,  221,  274,  275}, // 220
    {  172,  173,  220,  222,  275,  276},
    {  173,  174,  221,  223,  276,  277},
    {  174,  175,  222,  224,  277,  278},
    {  175,  176,  223,  225,  278,  279},
    {  176,  177,  224,  226,  279,  280},
    {  177,  225,  227,  280,  281,  282},
    {  177,  178,  226,  228,  282,  283},
    {  178,  179,  227,  229,  283,  284},
    {  179,  180,  228,  230,  284,  285},
    {  180,  181,  229,  231,  285,  286},
    {  181,  182,  230,  232,  286,  287},
    {  182,  183,  231,  233,  287,  288},
    {  183,  184,  232,  234,  288,  289},
    {  184,  185,  233,  235,  289,  290},
    {  185,  234,  236,  290,  291,  292},
    {  185,  186,  235,  237,  292,  293},
    {  186,  187,  236,  238,  293,  294},
    {  187,  188,  237,  239,  294,  295},
    {  188,  189,  238,  240,  295,  296},
    {  189,  190,  239,  241,  296,  297}, // 240
    {  190,  191,  240,  242,  297,  298},
    {  191,  192,  241,  243,  298,  299},
    {  192,  193,  242,  244,  299,  300},
    {  193,  243,  245,  300,  301,  302},
    {  193,  194,  244,  246,  302,  303},
    {  194,  195,  245,  247,  303,  304},
    {  195,  196,  246,  248,  304,  305},
    {  196,  197,  247,  249,  305,  306},
    {  197,  198,  248,  250,  306,  307},
    {  198,  199,  249,  251,  307,  308},
    {  199,  200,  250,  252,  308,  309},
    {  200,  201,  251,  253,  309,  310},
    {  201,  252,  254,  310,  311,  312},
    {  201,  202,  253,  255,  312,  313},
    {  202,  203,  254,  256,  313,  314},
    {  203,  204,  255,  257,  314,  315},
    {  204,  205,  256,  258,  315,  316},
    {  205,  206,  257,  259,  316,  317},
    {  206,  207,  258,  260,  317,  318},
    {  207,  208,  259,  261,  318,  319}, // 260
    {  208,  209,  260,  262,  319,  320},
    {  209,  261,  263,  320,  321,  322},
    {  209,  210,  262,  264,  322,  323},
    {  210,  211,  263,  265,  323,  324},
    {  211,  212,  264,  266,  324,  325},
    {  212,  213,  265,  267,  325,  326},
    {  213,  214,  266,  268,  326,  327},
    {  214,  215,  267,  269,  327,  328},
    {  215,  216,  268,  270,  328,  329},
    {  169,  216,  217,  269,  329,  330},
    {  217,  272,  330,  331,  332,  396},
    {  217,  218,  271,  273,  332,  333},
    {  218,  219,  272,  274,  333,  334},
    {  219,  220,  273,  275,  334,  335},
    {  220,  221,  274,  276,  335,  336},
    {  221,  222,  275,  277,  336,  337},
    {  222,  223,  276,  278,  337,  338},
    {  223,  224,  277,  279,  338,  339},
    {  224,  225,  278,  280,  339,  340},
    {  225,  226,  279,  281,  340,  341}, // 280
    {  226,  280,  282,  341,  342,  343},
    {  226,  227,  281,  283,  343,  344},
    {  227,  228,  282,  284,  344,  345},
    {  228,  229,  283,  285,  345,  346},
    {  229,  230,  284,  286,  346,  347},
    {  230,  231,  285,  287,  347,  348},
    {  231,  232,  286,  288,  348,  349},
    {  232,  233,  287,  289,  349,  350},
    {  233,  234,  288,  290,  350,  351},
    {  234,  235,  289,  291,  351,  352},
    {  235,  290,  292,  352,  353,  354},
    {  235,  236,  291,  293,  354,  355},
    {  236,  237,  292,  294,  355,  356},
    {  237,  238,  293,  295,  356,  357},
    {  238,  239,  294,  296,  357,  358},
    {  239,  240,  295,  297,  358,  359},
    {  240,  241,  296,  298,  359,  360},
    {  241,  242,  297,  299,  360,  361},
    {  242,  243,  298,  300,  361,  362},
    {  243,  244,  299,  301,  362,  363}, // 300
    {  244,  300,  302,  363,  364,  365},
    {  244,  245,  301,  303,  365,  366},
    {  245,  246,  302,  304,  366,  367},
    {  246,  247,  303,  305,  367,  368},
    {  247,  248,  304,  306,  368,  369},
    {  248,  249,  305,  307,  369,  370},
    {  249,  250,  306,  308,  370,  371},
    {  250,  251,  307,  309,  371,  372},
    {  251,  252,  308,  310,  372,  373},
    {  252,  253,  309,  311,  373,  374},
    {  253,  310,  312,  374,  375,  376},
    {  253,  254,  311,  313,  376,  377},
    {  254,  255,  312,  314,  377,  378},
    {  255,  256,  313,  315,  378,  379},
    {  256,  257,  314,  316,  379,  380},
    {  257,  258,  315,  317,  380,  381},
    {  258,  259,  316,  318,  381,  382},
    {  259,  260,  317,  319,  382,  383},
    {  260,  261,  318,  320,  383,  384},
    {  261,  262,  319,  321,  384,  385}, // 320
    {  262,  320,  322,  385,  386,  387},
    {  262,  263,  321,  323,  387,  388},
    {  263,  264,  322,  324,  388,  389},
    {  264,  265,  323,  325,  389,  390},
    {  265,  266,  324,  326,  390,  391},
    {  266,  267,  325,  327,  391,  392},
    {  267,  268,  326,  328,  392,  393},
    {  268,  269,  327,  329,  393,  394},
    {  269,  270,  328,  330,  394,  395},
    {  217,  270,  271,  329,  395,  396},
    {  271,  332,  396,  397,  398,  468},
    {  271,  272,  331,  333,  398,  399},
    {  272,  273,  332,  334,  399,  400},
    {  273,  274,  333,  335,  400,  401},
    {  274,  275,  334,  336,  401,  402},
    {  275,  276,  335,  337,  402,  403},
    {  276,  277,  336,  338,  403,  404},
    {  277,  278,  337,  339,  404,  405},
    {  278,  279,  338,  340,  405,  406},
    {  279,  280,  339,  341,  406,  407}, // 340
    {  280,  281,  340,  342,  407,  408},
    {  281,  341,  343,  408,  409,  410},
    {  281,  282,  342,  344,  410,  411},
    {  282,  283,  343,  345,  411,  412},
    {  283,  284,  344,  346,  412,  413},
    {  284,  285,  345,  347,  413,  414},
    {  285,  286,  346,  348,  414,  415},
    {  286,  287,  347,  349,  415,  416},
    {  287,  288,  348,  350,  416,  417},
    {  288,  289,  349,  351,  417,  418},
    {  289,  290,  350,  352,  418,  419},
    {  290,  291,  351,  353,  419,  420},
    {  291,  352,  354,  420,  421,  422},
    {  291,  292,  353,  355,  422,  423},
    {  292,  293,  354,  356,  423,  424},
    {  293,  294,  355,  357,  424,  425},
    {  294,  295,  356,  358,  425,  426},
    {  295,  296,  357,  359,  426,  427},
    {  296,  297,  358,  360,  427,  428},
    {  297,  298,  359,  361,  428,  429}, // 360
    {  298,  299,  360,  362,  429,  430},
    {  299,  300,  361,  363,  430,  431},
    {  300,  301,  362,  364,  431,  432},
    {  301,  363,  365,  432,  433,  434},
    {  301,  302,  364,  366,  434,  435},
    {  302,  303,  365,  367,  435,  436},
    {  303,  304,  366,  368,  436,  437},
    {  304,  305,  367,  369,  437,  438},
    {  305,  306,  368,  370,  438,  439},
    {  306,  307,  369,  371,  439,  440},
    {  307,  308,  370,  372,  440,  441},
    {  308,  309,  371,  373,  441,  442},
    {  309,  310,  372,  374,  442,  443},
    {  310,  311,  373,  375,  443,  444},
    {  311,  374,  376,  444,  445,  446},
    {  311,  312,  375,  377,  446,  447},
    {  312,  313,  376,  378,  447,  448},
    {  313,  314,  377,  379,  448,  449},
    {  314,  315,  378,  380,  449,  450},
    {  315,  316,  379,  381,  450,  451}, // 380
    {  316,  317,  380,  382,  451,  452},
    {  317,  318,  381,  383,  452,  453},
    {  318,  319,  382,  384,  453,  454},
    {  319,  320,  383,  385,  454,  455},
    {  320,  321,  384,  386,  455,  456},
    {  321,  385,  387,  456,  457,  458},
    {  321,  322,  386,  388,  458,  459},
    {  322,  323,  387,  389,  459,  460},
    {  323,  324,  388,  390,  460,  461},
    {  324,  325,  389,  391,  461,  462},
    {  325,  326,  390,  392,  462,  463},
    {  326,  327,  391,  393,  463,  464},
    {  327,  328,  392,  394,  464,  465},
    {  328,  329,  393,  395,  465,  466},
    {  329,  330,  394,  396,  466,  467},
    {  271,  330,  331,  395,  467,  468},
    {  331,  398,  468,  469,  470,  546},
    {  331,  332,  397,  399,  470,  471},
    {  332,  333,  398,  400,  471,  472},
    {  333,  334,  399,  401,  472,  473}, // 400
    {  334,  335,  400,  402,  473,  474},
    {  335,  336,  401,  403,  474,  475},
    {  336,  337,  402,  404,  475,  476},
    {  337,  338,  403,  405,  476,  477},
    {  338,  339,  404,  406,  477,  478},
    {  339,  340,  405,  407,  478,  479},
    {  340,  341,  406,  408,  479,  480},
    {  341,  342,  407,  409,  480,  481},
    {  342,  408,  410,  481,  482,  483},
    {  342,  343,  409,  411,  483,  484},
    {  343,  344,  410,  412,  484,  485},
    {  344,  345,  411,  413,  485,  486},
    {  345,  346,  412,  414,  486,  487},
    {  346,  347,  413,  415,  487,  488},
    {  347,  348,  414,  416,  488,  489},
    {  348,  349,  415,  417,  489,  490},
    {  349,  350,  416,  418,  490,  491},
    {  350,  351,  417,  419,  491,  492},
    {  351,  352,  418,  420,  492,  493},
    {  352,  353,  419,  421,  493,  494}, // 420
    {  353,  420,  422,  494,  495,  496},
    {  353,  354,  421,  423,  496,  497},
    {  354,  355,  422,  424,  497,  498},
    {  355,  356,  423,  425,  498,  499},
    {  356,  357,  424,  426,  499,  500},
    {  357,  358,  425,  427,  500,  501},
    {  358,  359,  426,  428,  501,  502},
    {  359,  360,  427,  429,  502,  503},
    {  360,  361,  428,  430,  503,  504},
    {  361,  362,  429,  431,  504,  505},
    {  362,  363,  430,  432,  505,  506},
    {  363,  364,  431,  433,  506,  507},
    {  364,  432,  434,  507,  508,  509},
    {  364,  365,  433,  435,  509,  510},
    {  365,  366,  434,  436,  510,  511},
    {  366,  367,  435,  437,  511,  512},
    {  367,  368,  436,  438,  512,  513},
    {  368,  369,  437,  439,  513,  514},
    {  369,  370,  438,  440,  514,  515},
    {  370,  371,  439,  441,  515,  516}, // 440
    {  371,  372,  440,  442,  516,  517},
    {  372,  373,  441,  443,  517,  518},
    {  373,  374,  442,  444,  518,  519},
    {  374,  375,  443,  445,  519,  520},
    {  375,  444,  446,  520,  521,  522},
    {  375,  376,  445,  447,  522,  523},
    {  376,  377,  446,  448,  523,  524},
    {  377,  378,  447,  449,  524,  525},
    {  378,  379,  448,  450,  525,  526},
    {  379,  380,  449,  451,  526,  527},
    {  380,  381,  450,  452,  527,  528},
    {  381,  382,  451,  453,  528,  529},
    {  382,  383,  452,  454,  529,  530},
    {  383,  384,  453,  455,  530,  531},
    {  384,  385,  454,  456,  531,  532},
    {  385,  386,  455,  457,  532,  533},
    {  386,  456,  458,  533,  534,  535},
    {  386,  387,  457,  459,  535,  536},
    {  387,  388,  458,  460,  536,  537},
    {  388,  389,  459,  461,  537,  538}, // 460
    {  389,  390,  460,  462,  538,  539},
    {  390,  391,  461,  463,  539,  540},
    {  391,  392,  462,  464,  540,  541},
    {  392,  393,  463,  465,  541,  542},
    {  393,  394,  464,  466,  542,  543},
    {  394,  395,  465,  467,  543,  544},
    {  395,  396,  466,  468,  544,  545},
    {  331,  396,  397,  467,  545,  546},
    {  397,  470,  546,  547,  548,  630},
    {  397,  398,  469,  471,  548,  549},
    {  398,  399,  470,  472,  549,  550},
    {  399,  400,  471,  473,  550,  551},
    {  400,  401,  472,  474,  551,  552},
    {  401,  402,  473,  475,  552,  553},
    {  402,  403,  474,  476,  553,  554},
    {  403,  404,  475,  477,  554,  555},
    {  404,  405,  476,  478,  555,  556},
    {  405,  406,  477,  479,  556,  557},
    {  406,  407,  478,  480,  557,  558},
    {  407,  408,  479,  481,  558,  559}, // 480
    {  408,  409,  480,  482,  559,  560},
    {  409,  481,  483,  560,  561,  562},
    {  409,  410,  482,  484,  562,  563},
    {  410,  411,  483,  485,  563,  564},
    {  411,  412,  484,  486,  564,  565},
    {  412,  413,  485,  487,  565,  566},
    {  413,  414,  486,  488,  566,  567},
    {  414,  415,  487,  489,  567,  568},
    {  415,  416,  488,  490,  568,  569},
    {  416,  417,  489,  491,  569,  570},
    {  417,  418,  490,  492,  570,  571},
    {  418,  419,  491,  493,  571,  572},
    {  419,  420,  492,  494,  572,  573},
    {  420,  421,  493,  495,  573,  574},
    {  421,  494,  496,  574,  575,  576},
    {  421,  422,  495,  497,  576,  577},
    {  422,  423,  496,  498,  577,  578},
    {  423,  424,  497,  499,  578,  579},
    {  424,  425,  498,  500,  579,  580},
    {  425,  426,  499,  501,  580,  581}, // 500
    {  426,  427,  500,  502,  581,  582},
    {  427,  428,  501,  503,  582,  583},
    {  428,  429,  502,  504,  583,  584},
    {  429,  430,  503,  505,  584,  585},
    {  430,  431,  504,  506,  585,  586},
    {  431,  432,  505,  507,  586,  587},
    {  432,  433,  506,  508,  587,  588},
    {  433,  507,  509,  588,  589,  590},
    {  433,  434,  508,  510,  590,  591},
    {  434,  435,  509,  511,  591,  592},
    {  435,  436,  510,  512,  592,  593},
    {  436,  437,  511,  513,  593,  594},
    {  437,  438,  512,  514,  594,  595},
    {  438,  439,  513,  515,  595,  596},
    {  439,  440,  514,  516,  596,  597},
    {  440,  441,  515,  517,  597,  598},
    {  441,  442,  516,  518,  598,  599},
    {  442,  443,  517,  519,  599,  600},
    {  443,  444,  518,  520,  600,  601},
    {  444,  445,  519,  521,  601,  602}, // 520
    {  445,  520,  522,  602,  603,  604},
    {  445,  446,  521,  523,  604,  605},
    {  446,  447,  522,  524,  605,  606},
    {  447,  448,  523,  525,  606,  607},
    {  448,  449,  524,  526,  607,  608},
    {  449,  450,  525,  527,  608,  609},
    {  450,  451,  526,  528,  609,  610},
    {  451,  452,  527,  529,  610,  611},
    {  452,  453,  528,  530,  611,  612},
    {  453,  454,  529,  531,  612,  613},
    {  454,  455,  530,  532,  613,  614},
    {  455,  456,  531,  533,  614,  615},
    {  456,  457,  532,  534,  615,  616},
    {  457,  533,  535,  616,  617,  618},
    {  457,  458,  534,  536,  618,  619},
    {  458,  459,  535,  537,  619,  620},
    {  459,  460,  536,  538,  620,  621},
    {  460,  461,  537,  539,  621,  622},
    {  461,  462,  538,  540,  622,  623},
    {  462,  463,  539,  541,  623,  624}, // 540
    {  463,  464,  540,  542,  624,  625},
    {  464,  465,  541,  543,  625,  626},
    {  465,  466,  542,  544,  626,  627},
    {  466,  467,  543,  545,  627,  628},
    {  467,  468,  544,  546,  628,  629},
    {  397,  468,  469,  545,  629,  630},
    {  469,  548,  630,  631,  632,  720},
    {  469,  470,  547,  549,  632,  633},
    {  470,  471,  548,  550,  633,  634},
    {  471,  472,  549,  551,  634,  635},
    {  472,  473,  550,  552,  635,  636},
    {  473,  474,  551,  553,  636,  637},
    {  474,  475,  552,  554,  637,  638},
    {  475,  476,  553,  555,  638,  639},
    {  476,  477,  554,  556,  639,  640},
    {  477,  478,  555,  557,  640,  641},
    {  478,  479,  556,  558,  641,  642},
    {  479,  480,  557,  559,  642,  643},
    {  480,  481,  558,  560,  643,  644},
    {  481,  482,  559,  561,  644,  645}, // 560
    {  482,  560,  562,  645,  646,  647},
    {  482,  483,  561,  563,  647,  648},
    {  483,  484,  562,  564,  648,  649},
    {  484,  485,  563,  565,  649,  650},
    {  485,  486,  564,  566,  650,  651},
    {  486,  487,  565,  567,  651,  652},
    {  487,  488,  566,  568,  652,  653},
    {  488,  489,  567,  569,  653,  654},
    {  489,  490,  568,  570,  654,  655},
    {  490,  491,  569,  571,  655,  656},
    {  491,  492,  570,  572,  656,  657},
    {  492,  493,  571,  573,  657,  658},
    {  493,  494,  572,  574,  658,  659},
    {  494,  495,  573,  575,  659,  660},
    {  495,  574,  576,  660,  661,  662},
    {  495,  496,  575,  577,  662,  663},
    {  496,  497,  576,  578,  663,  664},
    {  497,  498,  577,  579,  664,  665},
    {  498,  499,  578,  580,  665,  666},
    {  499,  500,  579,  581,  666,  667}, // 580
    {  500,  501,  580,  582,  667,  668},
    {  501,  502,  581,  583,  668,  669},
    {  502,  503,  582,  584,  669,  670},
    {  503,  504,  583,  585,  670,  671},
    {  504,  505,  584,  586,  671,  672},
    {  505,  506,  585,  587,  672,  673},
    {  506,  507,  586,  588,  673,  674},
    {  507,  508,  587,  589,  674,  675},
    {  508,  588,  590,  675,  676,  677},
    {  508,  509,  589,  591,  677,  678},
    {  509,  510,  590,  592,  678,  679},
    {  510,  511,  591,  593,  679,  680},
    {  511,  512,  592,  594,  680,  681},
    {  512,  513,  593,  595,  681,  682},
    {  513,  514,  594,  596,  682,  683},
    {  514,  515,  595,  597,  683,  684},
    {  515,  516,  596,  598,  684,  685},
    {  516,  517,  597,  599,  685,  686},
    {  517,  518,  598,  600,  686,  687},
    {  518,  519,  599,  601,  687,  688}, // 600
    {  519,  520,  600,  602,  688,  689},
    {  520,  521,  601,  603,  689,  690},
    {  521,  602,  604,  690,  691,  692},
    {  521,  522,  603,  605,  692,  693},
    {  522,  523,  604,  606,  693,  694},
    {  523,  524,  605,  607,  694,  695},
    {  524,  525,  606,  608,  695,  696},
    {  525,  526,  607,  609,  696,  697},
    {  526,  527,  608,  610,  697,  698},
    {  527,  528,  609,  611,  698,  699},
    {  528,  529,  610,  612,  699,  700},
    {  529,  530,  611,  613,  700,  701},
    {  530,  531,  612,  614,  701,  702},
    {  531,  532,  613,  615,  702,  703},
    {  532,  533,  614,  616,  703,  704},
    {  533,  534,  615,  617,  704,  705},
    {  534,  616,  618,  705,  706,  707},
    {  534,  535,  617,  619,  707,  708},
    {  535,  536,  618,  620,  708,  709},
    {  536,  537,  619,  621,  709,  710}, // 620
    {  537,  538,  620,  622,  710,  711},
    {  538,  539,  621,  623,  711,  712},
    {  539,  540,  622,  624,  712,  713},
    {  540,  541,  623,  625,  713,  714},
    {  541,  542,  624,  626,  714,  715},
    {  542,  543,  625,  627,  715,  716},
    {  543,  544,  626,  628,  716,  717},
    {  544,  545,  627,  629,  717,  718},
    {  545,  546,  628,  630,  718,  719},
    {  469,  546,  547,  629,  719,  720},
    {  547,  632,  720,  721,  722,  816},
    {  547,  548,  631,  633,  722,  723},
    {  548,  549,  632,  634,  723,  724},
    {  549,  550,  633,  635,  724,  725},
    {  550,  551,  634,  636,  725,  726},
    {  551,  552,  635,  637,  726,  727},
    {  552,  553,  636,  638,  727,  728},
    {  553,  554,  637,  639,  728,  729},
    {  554,  555,  638,  640,  729,  730},
    {  555,  556,  639,  641,  730,  731}, // 640
    {  556,  557,  640,  642,  731,  732},
    {  557,  558,  641,  643,  732,  733},
    {  558,  559,  642,  644,  733,  734},
    {  559,  560,  643,  645,  734,  735},
    {  560,  561,  644,  646,  735,  736},
    {  561,  645,  647,  736,  737,  738},
    {  561,  562,  646,  648,  738,  739},
    {  562,  563,  647,  649,  739,  740},
    {  563,  564,  648,  650,  740,  741},
    {  564,  565,  649,  651,  741,  742},
    {  565,  566,  650,  652,  742,  743},
    {  566,  567,  651,  653,  743,  744},
    {  567,  568,  652,  654,  744,  745},
    {  568,  569,  653,  655,  745,  746},
    {  569,  570,  654,  656,  746,  747},
    {  570,  571,  655,  657,  747,  748},
    {  571,  572,  656,  658,  748,  749},
    {  572,  573,  657,  659,  749,  750},
    {  573,  574,  658,  660,  750,  751},
    {  574,  575,  659,  661,  751,  752}, // 660
    {  575,  660,  662,  752,  753,  754},
    {  575,  576,  661,  663,  754,  755},
    {  576,  577,  662,  664,  755,  756},
    {  577,  578,  663,  665,  756,  757},
    {  578,  579,  664,  666,  757,  758},
    {  579,  580,  665,  667,  758,  759},
    {  580,  581,  666,  668,  759,  760},
    {  581,  582,  667,  669,  760,  761},
    {  582,  583,  668,  670,  761,  762},
    {  583,  584,  669,  671,  762,  763},
    {  584,  585,  670,  672,  763,  764},
    {  585,  586,  671,  673,  764,  765},
    {  586,  587,  672,  674,  765,  766},
    {  587,  588,  673,  675,  766,  767},
    {  588,  589,  674,  676,  767,  768},
    {  589,  675,  677,  768,  769,  770},
    {  589,  590,  676,  678,  770,  771},
    {  590,  591,  677,  679,  771,  772},
    {  591,  592,  678,  680,  772,  773},
    {  592,  593,  679,  681,  773,  774}, // 680
    {  593,  594,  680,  682,  774,  775},
    {  594,  595,  681,  683,  775,  776},
    {  595,  596,  682,  684,  776,  777},
    {  596,  597,  683,  685,  777,  778},
    {  597,  598,  684,  686,  778,  779},
    {  598,  599,  685,  687,  779,  780},
    {  599,  600,  686,  688,  780,  781},
    {  600,  601,  687,  689,  781,  782},
    {  601,  602,  688,  690,  782,  783},
    {  602,  603,  689,  691,  783,  784},
    {  603,  690,  692,  784,  785,  786},
    {  603,  604,  691,  693,  786,  787},
    {  604,  605,  692,  694,  787,  788},
    {  605,  606,  693,  695,  788,  789},
    {  606,  607,  694,  696,  789,  790},
    {  607,  608,  695,  697,  790,  791},
    {  608,  609,  696,  698,  791,  792},
    {  609,  610,  697,  699,  792,  793},
    {  610,  611,  698,  700,  793,  794},
    {  611,  612,  699,  701,  794,  795}, // 700
    {  612,  613,  700,  702,  795,  796},
    {  613,  614,  701,  703,  796,  797},
    {  614,  615,  702,  704,  797,  798},
    {  615,  616,  703,  705,  798,  799},
    {  616,  617,  704,  706,  799,  800},
    {  617,  705,  707,  800,  801,  802},
    {  617,  618,  706,  708,  802,  803},
    {  618,  619,  707,  709,  803,  804},
    {  619,  620,  708,  710,  804,  805},
    {  620,  621,  709,  711,  805,  806},
    {  621,  622,  710,  712,  806,  807},
    {  622,  623,  711,  713,  807,  808},
    {  623,  624,  712,  714,  808,  809},
    {  624,  625,  713,  715,  809,  810},
    {  625,  626,  714,  716,  810,  811},
    {  626,  627,  715,  717,  811,  812},
    {  627,  628,  716,  718,  812,  813},
    {  628,  629,  717,  719,  813,  814},
    {  629,  630,  718,  720,  814,  815},
    {  547,  630,  631,  719,  815,  816}, // 720
    {  631,  722,  816,  817,  818,  918},
    {  631,  632,  721,  723,  818,  819},
    {  632,  633,  722,  724,  819,  820},
    {  633,  634,  723,  725,  820,  821},
    {  634,  635,  724,  726,  821,  822},
    {  635,  636,  725,  727,  822,  823},
    {  636,  637,  726,  728,  823,  824},
    {  637,  638,  727,  729,  824,  825},
    {  638,  639,  728,  730,  825,  826},
    {  639,  640,  729,  731,  826,  827},
    {  640,  641,  730,  732,  827,  828},
    {  641,  642,  731,  733,  828,  829},
    {  642,  643,  732,  734,  829,  830},
    {  643,  644,  733,  735,  830,  831},
    {  644,  645,  734,  736,  831,  832},
    {  645,  646,  735,  737,  832,  833},
    {  646,  736,  738,  833,  834,  835},
    {  646,  647,  737,  739,  835,  836},
    {  647,  648,  738,  740,  836,  837},
    {  648,  649,  739,  741,  837,  838}, // 740
    {  649,  650,  740,  742,  838,  839},
    {  650,  651,  741,  743,  839,  840},
    {  651,  652,  742,  744,  840,  841},
    {  652,  653,  743,  745,  841,  842},
    {  653,  654,  744,  746,  842,  843},
    {  654,  655,  745,  747,  843,  844},
    {  655,  656,  746,  748,  844,  845},
    {  656,  657,  747,  749,  845,  846},
    {  657,  658,  748,  750,  846,  847},
    {  658,  659,  749,  751,  847,  848},
    {  659,  660,  750,  752,  848,  849},
    {  660,  661,  751,  753,  849,  850},
    {  661,  752,  754,  850,  851,  852},
    {  661,  662,  753,  755,  852,  853},
    {  662,  663,  754,  756,  853,  854},
    {  663,  664,  755,  757,  854,  855},
    {  664,  665,  756,  758,  855,  856},
    {  665,  666,  757,  759,  856,  857},
    {  666,  667,  758,  760,  857,  858},
    {  667,  668,  759,  761,  858,  859}, // 760
    {  668,  669,  760,  762,  859,  860},
    {  669,  670,  761,  763,  860,  861},
    {  670,  671,  762,  764,  861,  862},
    {  671,  672,  763,  765,  862,  863},
    {  672,  673,  764,  766,  863,  864},
    {  673,  674,  765,  767,  864,  865},
    {  674,  675,  766,  768,  865,  866},
    {  675,  676,  767,  769,  866,  867},
    {  676,  768,  770,  867,  868,  869},
    {  676,  677,  769,  771,  869,  870},
    {  677,  678,  770,  772,  870,  871},
    {  678,  679,  771,  773,  871,  872},
    {  679,  680,  772,  774,  872,  873},
    {  680,  681,  773,  775,  873,  874},
    {  681,  682,  774,  776,  874,  875},
    {  682,  683,  775,  777,  875,  876},
    {  683,  684,  776,  778,  876,  877},
    {  684,  685,  777,  779,  877,  878},
    {  685,  686,  778,  780,  878,  879},
    {  686,  687,  779,  781,  879,  880}, // 780
    {  687,  688,  780,  782,  880,  881},
    {  688,  689,  781,  783,  881,  882},
    {  689,  690,  782,  784,  882,  883},
    {  690,  691,  783,  785,  883,  884},
    {  691,  784,  786,  884,  885,  886},
    {  691,  692,  785,  787,  886,  887},
    {  692,  693,  786,  788,  887,  888},
    {  693,  694,  787,  789,  888,  889},
    {  694,  695,  788,  790,  889,  890},
    {  695,  696,  789,  791,  890,  891},
    {  696,  697,  790,  792,  891,  892},
    {  697,  698,  791,  793,  892,  893},
    {  698,  699,  792,  794,  893,  894},
    {  699,  700,  793,  795,  894,  895},
    {  700,  701,  794,  796,  895,  896},
    {  701,  702,  795,  797,  896,  897},
    {  702,  703,  796,  798,  897,  898},
    {  703,  704,  797,  799,  898,  899},
    {  704,  705,  798,  800,  899,  900},
    {  705,  706,  799,  801,  900,  901}, // 800
    {  706,  800,  802,  901,  902,  903},
    {  706,  707,  801,  803,  903,  904},
    {  707,  708,  802,  804,  904,  905},
    {  708,  709,  803,  805,  905,  906},
    {  709,  710,  804,  806,  906,  907},
    {  710,  711,  805,  807,  907,  908},
    {  711,  712,  806,  808,  908,  909},
    {  712,  713,  807,  809,  909,  910},
    {  713,  714,  808,  810,  910,  911},
    {  714,  715,  809,  811,  911,  912},
    {  715,  716,  810,  812,  912,  913},
    {  716,  717,  811,  813,  913,  914},
    {  717,  718,  812,  814,  914,  915},
    {  718,  719,  813,  815,  915,  916},
    {  719,  720,  814,  816,  916,  917},
    {  631,  720,  721,  815,  917,  918},
    {  721,  818,  918,  919,  920,  1026},
    {  721,  722,  817,  819,  920,  921},
    {  722,  723,  818,  820,  921,  922},
    {  723,  724,  819,  821,  922,  923}, // 820
    {  724,  725,  820,  822,  923,  924},
    {  725,  726,  821,  823,  924,  925},
    {  726,  727,  822,  824,  925,  926},
    {  727,  728,  823,  825,  926,  927},
    {  728,  729,  824,  826,  927,  928},
    {  729,  730,  825,  827,  928,  929},
    {  730,  731,  826,  828,  929,  930},
    {  731,  732,  827,  829,  930,  931},
    {  732,  733,  828,  830,  931,  932},
    {  733,  734,  829,  831,  932,  933},
    {  734,  735,  830,  832,  933,  934},
    {  735,  736,  831,  833,  934,  935},
    {  736,  737,  832,  834,  935,  936},
    {  737,  833,  835,  936,  937,  938},
    {  737,  738,  834,  836,  938,  939},
    {  738,  739,  835,  837,  939,  940},
    {  739,  740,  836,  838,  940,  941},
    {  740,  741,  837,  839,  941,  942},
    {  741,  742,  838,  840,  942,  943},
    {  742,  743,  839,  841,  943,  944}, // 840
    {  743,  744,  840,  842,  944,  945},
    {  744,  745,  841,  843,  945,  946},
    {  745,  746,  842,  844,  946,  947},
    {  746,  747,  843,  845,  947,  948},
    {  747,  748,  844,  846,  948,  949},
    {  748,  749,  845,  847,  949,  950},
    {  749,  750,  846,  848,  950,  951},
    {  750,  751,  847,  849,  951,  952},
    {  751,  752,  848,  850,  952,  953},
    {  752,  753,  849,  851,  953,  954},
    {  753,  850,  852,  954,  955,  956},
    {  753,  754,  851,  853,  956,  957},
    {  754,  755,  852,  854,  957,  958},
    {  755,  756,  853,  855,  958,  959},
    {  756,  757,  854,  856,  959,  960},
    {  757,  758,  855,  857,  960,  961},
    {  758,  759,  856,  858,  961,  962},
    {  759,  760,  857,  859,  962,  963},
    {  760,  761,  858,  860,  963,  964},
    {  761,  762,  859,  861,  964,  965}, // 860
    {  762,  763,  860,  862,  965,  966},
    {  763,  764,  861,  863,  966,  967},
    {  764,  765,  862,  864,  967,  968},
    {  765,  766,  863,  865,  968,  969},
    {  766,  767,  864,  866,  969,  970},
    {  767,  768,  865,  867,  970,  971},
    {  768,  769,  866,  868,  971,  972},
    {  769,  867,  869,  972,  973,  974},
    {  769,  770,  868,  870,  974,  975},
    {  770,  771,  869,  871,  975,  976},
    {  771,  772,  870,  872,  976,  977},
    {  772,  773,  871,  873,  977,  978},
    {  773,  774,  872,  874,  978,  979},
    {  774,  775,  873,  875,  979,  980},
    {  775,  776,  874,  876,  980,  981},
    {  776,  777,  875,  877,  981,  982},
    {  777,  778,  876,  878,  982,  983},
    {  778,  779,  877,  879,  983,  984},
    {  779,  780,  878,  880,  984,  985},
    {  780,  781,  879,  881,  985,  986}, // 880
    {  781,  782,  880,  882,  986,  987},
    {  782,  783,  881,  883,  987,  988},
    {  783,  784,  882,  884,  988,  989},
    {  784,  785,  883,  885,  989,  990},
    {  785,  884,  886,  990,  991,  992},
    {  785,  786,  885,  887,  992,  993},
    {  786,  787,  886,  888,  993,  994},
    {  787,  788,  887,  889,  994,  995},
    {  788,  789,  888,  890,  995,  996},
    {  789,  790,  889,  891,  996,  997},
    {  790,  791,  890,  892,  997,  998},
    {  791,  792,  891,  893,  998,  999},
    {  792,  793,  892,  894,  999,  1000},
    {  793,  794,  893,  895,  1000,  1001},
    {  794,  795,  894,  896,  1001,  1002},
    {  795,  796,  895,  897,  1002,  1003},
    {  796,  797,  896,  898,  1003,  1004},
    {  797,  798,  897,  899,  1004,  1005},
    {  798,  799,  898,  900,  1005,  1006},
    {  799,  800,  899,  901,  1006,  1007}, // 900
    {  800,  801,  900,  902,  1007,  1008},
    {  801,  901,  903,  1008,  1009,  1010},
    {  801,  802,  902,  904,  1010,  1011},
    {  802,  803,  903,  905,  1011,  1012},
    {  803,  804,  904,  906,  1012,  1013},
    {  804,  805,  905,  907,  1013,  1014},
    {  805,  806,  906,  908,  1014,  1015},
    {  806,  807,  907,  909,  1015,  1016},
    {  807,  808,  908,  910,  1016,  1017},
    {  808,  809,  909,  911,  1017,  1018},
    {  809,  810,  910,  912,  1018,  1019},
    {  810,  811,  911,  913,  1019,  1020},
    {  811,  812,  912,  914,  1020,  1021},
    {  812,  813,  913,  915,  1021,  1022},
    {  813,  814,  914,  916,  1022,  1023},
    {  814,  815,  915,  917,  1023,  1024},
    {  815,  816,  916,  918,  1024,  1025},
    {  721,  816,  817,  917,  1025,  1026},
    {  817,  920,  1026,  1027,  1028,  1140},
    {  817,  818,  919,  921,  1028,  1029}, // 920
    {  818,  819,  920,  922,  1029,  1030},
    {  819,  820,  921,  923,  1030,  1031},
    {  820,  821,  922,  924,  1031,  1032},
    {  821,  822,  923,  925,  1032,  1033},
    {  822,  823,  924,  926,  1033,  1034},
    {  823,  824,  925,  927,  1034,  1035},
    {  824,  825,  926,  928,  1035,  1036},
    {  825,  826,  927,  929,  1036,  1037},
    {  826,  827,  928,  930,  1037,  1038},
    {  827,  828,  929,  931,  1038,  1039},
    {  828,  829,  930,  932,  1039,  1040},
    {  829,  830,  931,  933,  1040,  1041},
    {  830,  831,  932,  934,  1041,  1042},
    {  831,  832,  933,  935,  1042,  1043},
    {  832,  833,  934,  936,  1043,  1044},
    {  833,  834,  935,  937,  1044,  1045},
    {  834,  936,  938,  1045,  1046,  1047},
    {  834,  835,  937,  939,  1047,  1048},
    {  835,  836,  938,  940,  1048,  1049},
    {  836,  837,  939,  941,  1049,  1050}, // 940
    {  837,  838,  940,  942,  1050,  1051},
    {  838,  839,  941,  943,  1051,  1052},
    {  839,  840,  942,  944,  1052,  1053},
    {  840,  841,  943,  945,  1053,  1054},
    {  841,  842,  944,  946,  1054,  1055},
    {  842,  843,  945,  947,  1055,  1056},
    {  843,  844,  946,  948,  1056,  1057},
    {  844,  845,  947,  949,  1057,  1058},
    {  845,  846,  948,  950,  1058,  1059},
    {  846,  847,  949,  951,  1059,  1060},
    {  847,  848,  950,  952,  1060,  1061},
    {  848,  849,  951,  953,  1061,  1062},
    {  849,  850,  952,  954,  1062,  1063},
    {  850,  851,  953,  955,  1063,  1064},
    {  851,  954,  956,  1064,  1065,  1066},
    {  851,  852,  955,  957,  1066,  1067},
    {  852,  853,  956,  958,  1067,  1068},
    {  853,  854,  957,  959,  1068,  1069},
    {  854,  855,  958,  960,  1069,  1070},
    {  855,  856,  959,  961,  1070,  1071}, // 960
    {  856,  857,  960,  962,  1071,  1072},
    {  857,  858,  961,  963,  1072,  1073},
    {  858,  859,  962,  964,  1073,  1074},
    {  859,  860,  963,  965,  1074,  1075},
    {  860,  861,  964,  966,  1075,  1076},
    {  861,  862,  965,  967,  1076,  1077},
    {  862,  863,  966,  968,  1077,  1078},
    {  863,  864,  967,  969,  1078,  1079},
    {  864,  865,  968,  970,  1079,  1080},
    {  865,  866,  969,  971,  1080,  1081},
    {  866,  867,  970,  972,  1081,  1082},
    {  867,  868,  971,  973,  1082,  1083},
    {  868,  972,  974,  1083,  1084,  1085},
    {  868,  869,  973,  975,  1085,  1086},
    {  869,  870,  974,  976,  1086,  1087},
    {  870,  871,  975,  977,  1087,  1088},
    {  871,  872,  976,  978,  1088,  1089},
    {  872,  873,  977,  979,  1089,  1090},
    {  873,  874,  978,  980,  1090,  1091},
    {  874,  875,  979,  981,  1091,  1092}, // 980
    {  875,  876,  980,  982,  1092,  1093},
    {  876,  877,  981,  983,  1093,  1094},
    {  877,  878,  982,  984,  1094,  1095},
    {  878,  879,  983,  985,  1095,  1096},
    {  879,  880,  984,  986,  1096,  1097},
    {  880,  881,  985,  987,  1097,  1098},
    {  881,  882,  986,  988,  1098,  1099},
    {  882,  883,  987,  989,  1099,  1100},
    {  883,  884,  988,  990,  1100,  1101},
    {  884,  885,  989,  991,  1101,  1102},
    {  885,  990,  992,  1102,  1103,  1104},
    {  885,  886,  991,  993,  1104,  1105},
    {  886,  887,  992,  994,  1105,  1106},
    {  887,  888,  993,  995,  1106,  1107},
    {  888,  889,  994,  996,  1107,  1108},
    {  889,  890,  995,  997,  1108,  1109},
    {  890,  891,  996,  998,  1109,  1110},
    {  891,  892,  997,  999,  1110,  1111},
    {  892,  893,  998,  1000,  1111,  1112},
    {  893,  894,  999,  1001,  1112,  1113}, // 1000
    {  894,  895,  1000,  1002,  1113,  1114},
    {  895,  896,  1001,  1003,  1114,  1115},
    {  896,  897,  1002,  1004,  1115,  1116},
    {  897,  898,  1003,  1005,  1116,  1117},
    {  898,  899,  1004,  1006,  1117,  1118},
    {  899,  900,  1005,  1007,  1118,  1119},
    {  900,  901,  1006,  1008,  1119,  1120},
    {  901,  902,  1007,  1009,  1120,  1121},
    {  902,  1008,  1010,  1121,  1122,  1123},
    {  902,  903,  1009,  1011,  1123,  1124},
    {  903,  904,  1010,  1012,  1124,  1125},
    {  904,  905,  1011,  1013,  1125,  1126},
    {  905,  906,  1012,  1014,  1126,  1127},
    {  906,  907,  1013,  1015,  1127,  1128},
    {  907,  908,  1014,  1016,  1128,  1129},
    {  908,  909,  1015,  1017,  1129,  1130},
    {  909,  910,  1016,  1018,  1130,  1131},
    {  910,  911,  1017,  1019,  1131,  1132},
    {  911,  912,  1018,  1020,  1132,  1133},
    {  912,  913,  1019,  1021,  1133,  1134}, // 1020
    {  913,  914,  1020,  1022,  1134,  1135},
    {  914,  915,  1021,  1023,  1135,  1136},
    {  915,  916,  1022,  1024,  1136,  1137},
    {  916,  917,  1023,  1025,  1137,  1138},
    {  917,  918,  1024,  1026,  1138,  1139},
    {  817,  918,  919,  1025,  1139,  1140},
    {  919,  1028,  1140,  1141,  1142,  1260},
    {  919,  920,  1027,  1029,  1142,  1143},
    {  920,  921,  1028,  1030,  1143,  1144},
    {  921,  922,  1029,  1031,  1144,  1145},
    {  922,  923,  1030,  1032,  1145,  1146},
    {  923,  924,  1031,  1033,  1146,  1147},
    {  924,  925,  1032,  1034,  1147,  1148},
    {  925,  926,  1033,  1035,  1148,  1149},
    {  926,  927,  1034,  1036,  1149,  1150},
    {  927,  928,  1035,  1037,  1150,  1151},
    {  928,  929,  1036,  1038,  1151,  1152},
    {  929,  930,  1037,  1039,  1152,  1153},
    {  930,  931,  1038,  1040,  1153,  1154},
    {  931,  932,  1039,  1041,  1154,  1155}, // 1040
    {  932,  933,  1040,  1042,  1155,  1156},
    {  933,  934,  1041,  1043,  1156,  1157},
    {  934,  935,  1042,  1044,  1157,  1158},
    {  935,  936,  1043,  1045,  1158,  1159},
    {  936,  937,  1044,  1046,  1159,  1160},
    {  937,  1045,  1047,  1160,  1161,  1162},
    {  937,  938,  1046,  1048,  1162,  1163},
    {  938,  939,  1047,  1049,  1163,  1164},
    {  939,  940,  1048,  1050,  1164,  1165},
    {  940,  941,  1049,  1051,  1165,  1166},
    {  941,  942,  1050,  1052,  1166,  1167},
    {  942,  943,  1051,  1053,  1167,  1168},
    {  943,  944,  1052,  1054,  1168,  1169},
    {  944,  945,  1053,  1055,  1169,  1170},
    {  945,  946,  1054,  1056,  1170,  1171},
    {  946,  947,  1055,  1057,  1171,  1172},
    {  947,  948,  1056,  1058,  1172,  1173},
    {  948,  949,  1057,  1059,  1173,  1174},
    {  949,  950,  1058,  1060,  1174,  1175},
    {  950,  951,  1059,  1061,  1175,  1176}, // 1060
    {  951,  952,  1060,  1062,  1176,  1177},
    {  952,  953,  1061,  1063,  1177,  1178},
    {  953,  954,  1062,  1064,  1178,  1179},
    {  954,  955,  1063,  1065,  1179,  1180},
    {  955,  1064,  1066,  1180,  1181,  1182},
    {  955,  956,  1065,  1067,  1182,  1183},
    {  956,  957,  1066,  1068,  1183,  1184},
    {  957,  958,  1067,  1069,  1184,  1185},
    {  958,  959,  1068,  1070,  1185,  1186},
    {  959,  960,  1069,  1071,  1186,  1187},
    {  960,  961,  1070,  1072,  1187,  1188},
    {  961,  962,  1071,  1073,  1188,  1189},
    {  962,  963,  1072,  1074,  1189,  1190},
    {  963,  964,  1073,  1075,  1190,  1191},
    {  964,  965,  1074,  1076,  1191,  1192},
    {  965,  966,  1075,  1077,  1192,  1193},
    {  966,  967,  1076,  1078,  1193,  1194},
    {  967,  968,  1077,  1079,  1194,  1195},
    {  968,  969,  1078,  1080,  1195,  1196},
    {  969,  970,  1079,  1081,  1196,  1197}, // 1080
    {  970,  971,  1080,  1082,  1197,  1198},
    {  971,  972,  1081,  1083,  1198,  1199},
    {  972,  973,  1082,  1084,  1199,  1200},
    {  973,  1083,  1085,  1200,  1201,  1202},
    {  973,  974,  1084,  1086,  1202,  1203},
    {  974,  975,  1085,  1087,  1203,  1204},
    {  975,  976,  1086,  1088,  1204,  1205},
    {  976,  977,  1087,  1089,  1205,  1206},
    {  977,  978,  1088,  1090,  1206,  1207},
    {  978,  979,  1089,  1091,  1207,  1208},
    {  979,  980,  1090,  1092,  1208,  1209},
    {  980,  981,  1091,  1093,  1209,  1210},
    {  981,  982,  1092,  1094,  1210,  1211},
    {  982,  983,  1093,  1095,  1211,  1212},
    {  983,  984,  1094,  1096,  1212,  1213},
    {  984,  985,  1095,  1097,  1213,  1214},
    {  985,  986,  1096,  1098,  1214,  1215},
    {  986,  987,  1097,  1099,  1215,  1216},
    {  987,  988,  1098,  1100,  1216,  1217},
    {  988,  989,  1099,  1101,  1217,  1218}, // 1100
    {  989,  990,  1100,  1102,  1218,  1219},
    {  990,  991,  1101,  1103,  1219,  1220},
    {  991,  1102,  1104,  1220,  1221,  1222},
    {  991,  992,  1103,  1105,  1222,  1223},
    {  992,  993,  1104,  1106,  1223,  1224},
    {  993,  994,  1105,  1107,  1224,  1225},
    {  994,  995,  1106,  1108,  1225,  1226},
    {  995,  996,  1107,  1109,  1226,  1227},
    {  996,  997,  1108,  1110,  1227,  1228},
    {  997,  998,  1109,  1111,  1228,  1229},
    {  998,  999,  1110,  1112,  1229,  1230},
    {  999,  1000,  1111,  1113,  1230,  1231},
    {  1000,  1001,  1112,  1114,  1231,  1232},
    {  1001,  1002,  1113,  1115,  1232,  1233},
    {  1002,  1003,  1114,  1116,  1233,  1234},
    {  1003,  1004,  1115,  1117,  1234,  1235},
    {  1004,  1005,  1116,  1118,  1235,  1236},
    {  1005,  1006,  1117,  1119,  1236,  1237},
    {  1006,  1007,  1118,  1120,  1237,  1238},
    {  1007,  1008,  1119,  1121,  1238,  1239}, // 1120
    {  1008,  1009,  1120,  1122,  1239,  1240},
    {  1009,  1121,  1123,  1240,  1241,  1242},
    {  1009,  1010,  1122,  1124,  1242,  1243},
    {  1010,  1011,  1123,  1125,  1243,  1244},
    {  1011,  1012,  1124,  1126,  1244,  1245},
    {  1012,  1013,  1125,  1127,  1245,  1246},
    {  1013,  1014,  1126,  1128,  1246,  1247},
    {  1014,  1015,  1127,  1129,  1247,  1248},
    {  1015,  1016,  1128,  1130,  1248,  1249},
    {  1016,  1017,  1129,  1131,  1249,  1250},
    {  1017,  1018,  1130,  1132,  1250,  1251},
    {  1018,  1019,  1131,  1133,  1251,  1252},
    {  1019,  1020,  1132,  1134,  1252,  1253},
    {  1020,  1021,  1133,  1135,  1253,  1254},
    {  1021,  1022,  1134,  1136,  1254,  1255},
    {  1022,  1023,  1135,  1137,  1255,  1256},
    {  1023,  1024,  1136,  1138,  1256,  1257},
    {  1024,  1025,  1137,  1139,  1257,  1258},
    {  1025,  1026,  1138,  1140,  1258,  1259},
    {  919,  1026,  1027,  1139,  1259,  1260}, // 1140
    {  1027,  1142,  1260,  1261,  1262,  1386},
    {  1027,  1028,  1141,  1143,  1262,  1263},
    {  1028,  1029,  1142,  1144,  1263,  1264},
    {  1029,  1030,  1143,  1145,  1264,  1265},
    {  1030,  1031,  1144,  1146,  1265,  1266},
    {  1031,  1032,  1145,  1147,  1266,  1267},
    {  1032,  1033,  1146,  1148,  1267,  1268},
    {  1033,  1034,  1147,  1149,  1268,  1269},
    {  1034,  1035,  1148,  1150,  1269,  1270},
    {  1035,  1036,  1149,  1151,  1270,  1271},
    {  1036,  1037,  1150,  1152,  1271,  1272},
    {  1037,  1038,  1151,  1153,  1272,  1273},
    {  1038,  1039,  1152,  1154,  1273,  1274},
    {  1039,  1040,  1153,  1155,  1274,  1275},
    {  1040,  1041,  1154,  1156,  1275,  1276},
    {  1041,  1042,  1155,  1157,  1276,  1277},
    {  1042,  1043,  1156,  1158,  1277,  1278},
    {  1043,  1044,  1157,  1159,  1278,  1279},
    {  1044,  1045,  1158,  1160,  1279,  1280},
    {  1045,  1046,  1159,  1161,  1280,  1281}, // 1160
    {  1046,  1160,  1162,  1281,  1282,  1283},
    {  1046,  1047,  1161,  1163,  1283,  1284},
    {  1047,  1048,  1162,  1164,  1284,  1285},
    {  1048,  1049,  1163,  1165,  1285,  1286},
    {  1049,  1050,  1164,  1166,  1286,  1287},
    {  1050,  1051,  1165,  1167,  1287,  1288},
    {  1051,  1052,  1166,  1168,  1288,  1289},
    {  1052,  1053,  1167,  1169,  1289,  1290},
    {  1053,  1054,  1168,  1170,  1290,  1291},
    {  1054,  1055,  1169,  1171,  1291,  1292},
    {  1055,  1056,  1170,  1172,  1292,  1293},
    {  1056,  1057,  1171,  1173,  1293,  1294},
    {  1057,  1058,  1172,  1174,  1294,  1295},
    {  1058,  1059,  1173,  1175,  1295,  1296},
    {  1059,  1060,  1174,  1176,  1296,  1297},
    {  1060,  1061,  1175,  1177,  1297,  1298},
    {  1061,  1062,  1176,  1178,  1298,  1299},
    {  1062,  1063,  1177,  1179,  1299,  1300},
    {  1063,  1064,  1178,  1180,  1300,  1301},
    {  1064,  1065,  1179,  1181,  1301,  1302}, // 1180
    {  1065,  1180,  1182,  1302,  1303,  1304},
    {  1065,  1066,  1181,  1183,  1304,  1305},
    {  1066,  1067,  1182,  1184,  1305,  1306},
    {  1067,  1068,  1183,  1185,  1306,  1307},
    {  1068,  1069,  1184,  1186,  1307,  1308},
    {  1069,  1070,  1185,  1187,  1308,  1309},
    {  1070,  1071,  1186,  1188,  1309,  1310},
    {  1071,  1072,  1187,  1189,  1310,  1311},
    {  1072,  1073,  1188,  1190,  1311,  1312},
    {  1073,  1074,  1189,  1191,  1312,  1313},
    {  1074,  1075,  1190,  1192,  1313,  1314},
    {  1075,  1076,  1191,  1193,  1314,  1315},
    {  1076,  1077,  1192,  1194,  1315,  1316},
    {  1077,  1078,  1193,  1195,  1316,  1317},
    {  1078,  1079,  1194,  1196,  1317,  1318},
    {  1079,  1080,  1195,  1197,  1318,  1319},
    {  1080,  1081,  1196,  1198,  1319,  1320},
    {  1081,  1082,  1197,  1199,  1320,  1321},
    {  1082,  1083,  1198,  1200,  1321,  1322},
    {  1083,  1084,  1199,  1201,  1322,  1323}, // 1200
    {  1084,  1200,  1202,  1323,  1324,  1325},
    {  1084,  1085,  1201,  1203,  1325,  1326},
    {  1085,  1086,  1202,  1204,  1326,  1327},
    {  1086,  1087,  1203,  1205,  1327,  1328},
    {  1087,  1088,  1204,  1206,  1328,  1329},
    {  1088,  1089,  1205,  1207,  1329,  1330},
    {  1089,  1090,  1206,  1208,  1330,  1331},
    {  1090,  1091,  1207,  1209,  1331,  1332},
    {  1091,  1092,  1208,  1210,  1332,  1333},
    {  1092,  1093,  1209,  1211,  1333,  1334},
    {  1093,  1094,  1210,  1212,  1334,  1335},
    {  1094,  1095,  1211,  1213,  1335,  1336},
    {  1095,  1096,  1212,  1214,  1336,  1337},
    {  1096,  1097,  1213,  1215,  1337,  1338},
    {  1097,  1098,  1214,  1216,  1338,  1339},
    {  1098,  1099,  1215,  1217,  1339,  1340},
    {  1099,  1100,  1216,  1218,  1340,  1341},
    {  1100,  1101,  1217,  1219,  1341,  1342},
    {  1101,  1102,  1218,  1220,  1342,  1343},
    {  1102,  1103,  1219,  1221,  1343,  1344}, // 1220
    {  1103,  1220,  1222,  1344,  1345,  1346},
    {  1103,  1104,  1221,  1223,  1346,  1347},
    {  1104,  1105,  1222,  1224,  1347,  1348},
    {  1105,  1106,  1223,  1225,  1348,  1349},
    {  1106,  1107,  1224,  1226,  1349,  1350},
    {  1107,  1108,  1225,  1227,  1350,  1351},
    {  1108,  1109,  1226,  1228,  1351,  1352},
    {  1109,  1110,  1227,  1229,  1352,  1353},
    {  1110,  1111,  1228,  1230,  1353,  1354},
    {  1111,  1112,  1229,  1231,  1354,  1355},
    {  1112,  1113,  1230,  1232,  1355,  1356},
    {  1113,  1114,  1231,  1233,  1356,  1357},
    {  1114,  1115,  1232,  1234,  1357,  1358},
    {  1115,  1116,  1233,  1235,  1358,  1359},
    {  1116,  1117,  1234,  1236,  1359,  1360},
    {  1117,  1118,  1235,  1237,  1360,  1361},
    {  1118,  1119,  1236,  1238,  1361,  1362},
    {  1119,  1120,  1237,  1239,  1362,  1363},
    {  1120,  1121,  1238,  1240,  1363,  1364},
    {  1121,  1122,  1239,  1241,  1364,  1365}, // 1240
    {  1122,  1240,  1242,  1365,  1366,  1367},
    {  1122,  1123,  1241,  1243,  1367,  1368},
    {  1123,  1124,  1242,  1244,  1368,  1369},
    {  1124,  1125,  1243,  1245,  1369,  1370},
    {  1125,  1126,  1244,  1246,  1370,  1371},
    {  1126,  1127,  1245,  1247,  1371,  1372},
    {  1127,  1128,  1246,  1248,  1372,  1373},
    {  1128,  1129,  1247,  1249,  1373,  1374},
    {  1129,  1130,  1248,  1250,  1374,  1375},
    {  1130,  1131,  1249,  1251,  1375,  1376},
    {  1131,  1132,  1250,  1252,  1376,  1377},
    {  1132,  1133,  1251,  1253,  1377,  1378},
    {  1133,  1134,  1252,  1254,  1378,  1379},
    {  1134,  1135,  1253,  1255,  1379,  1380},
    {  1135,  1136,  1254,  1256,  1380,  1381},
    {  1136,  1137,  1255,  1257,  1381,  1382},
    {  1137,  1138,  1256,  1258,  1382,  1383},
    {  1138,  1139,  1257,  1259,  1383,  1384},
    {  1139,  1140,  1258,  1260,  1384,  1385},
    {  1027,  1140,  1141,  1259,  1385,  1386}, // 1260
    {  1141,  1262,  1386,  1387,  1388,  1518},
    {  1141,  1142,  1261,  1263,  1388,  1389},
    {  1142,  1143,  1262,  1264,  1389,  1390},
    {  1143,  1144,  1263,  1265,  1390,  1391},
    {  1144,  1145,  1264,  1266,  1391,  1392},
    {  1145,  1146,  1265,  1267,  1392,  1393},
    {  1146,  1147,  1266,  1268,  1393,  1394},
    {  1147,  1148,  1267,  1269,  1394,  1395},
    {  1148,  1149,  1268,  1270,  1395,  1396},
    {  1149,  1150,  1269,  1271,  1396,  1397},
    {  1150,  1151,  1270,  1272,  1397,  1398},
    {  1151,  1152,  1271,  1273,  1398,  1399},
    {  1152,  1153,  1272,  1274,  1399,  1400},
    {  1153,  1154,  1273,  1275,  1400,  1401},
    {  1154,  1155,  1274,  1276,  1401,  1402},
    {  1155,  1156,  1275,  1277,  1402,  1403},
    {  1156,  1157,  1276,  1278,  1403,  1404},
    {  1157,  1158,  1277,  1279,  1404,  1405},
    {  1158,  1159,  1278,  1280,  1405,  1406},
    {  1159,  1160,  1279,  1281,  1406,  1407}, // 1280
    {  1160,  1161,  1280,  1282,  1407,  1408},
    {  1161,  1281,  1283,  1408,  1409,  1410},
    {  1161,  1162,  1282,  1284,  1410,  1411},
    {  1162,  1163,  1283,  1285,  1411,  1412},
    {  1163,  1164,  1284,  1286,  1412,  1413},
    {  1164,  1165,  1285,  1287,  1413,  1414},
    {  1165,  1166,  1286,  1288,  1414,  1415},
    {  1166,  1167,  1287,  1289,  1415,  1416},
    {  1167,  1168,  1288,  1290,  1416,  1417},
    {  1168,  1169,  1289,  1291,  1417,  1418},
    {  1169,  1170,  1290,  1292,  1418,  1419},
    {  1170,  1171,  1291,  1293,  1419,  1420},
    {  1171,  1172,  1292,  1294,  1420,  1421},
    {  1172,  1173,  1293,  1295,  1421,  1422},
    {  1173,  1174,  1294,  1296,  1422,  1423},
    {  1174,  1175,  1295,  1297,  1423,  1424},
    {  1175,  1176,  1296,  1298,  1424,  1425},
    {  1176,  1177,  1297,  1299,  1425,  1426},
    {  1177,  1178,  1298,  1300,  1426,  1427},
    {  1178,  1179,  1299,  1301,  1427,  1428}, // 1300
    {  1179,  1180,  1300,  1302,  1428,  1429},
    {  1180,  1181,  1301,  1303,  1429,  1430},
    {  1181,  1302,  1304,  1430,  1431,  1432},
    {  1181,  1182,  1303,  1305,  1432,  1433},
    {  1182,  1183,  1304,  1306,  1433,  1434},
    {  1183,  1184,  1305,  1307,  1434,  1435},
    {  1184,  1185,  1306,  1308,  1435,  1436},
    {  1185,  1186,  1307,  1309,  1436,  1437},
    {  1186,  1187,  1308,  1310,  1437,  1438},
    {  1187,  1188,  1309,  1311,  1438,  1439},
    {  1188,  1189,  1310,  1312,  1439,  1440},
    {  1189,  1190,  1311,  1313,  1440,  1441},
    {  1190,  1191,  1312,  1314,  1441,  1442},
    {  1191,  1192,  1313,  1315,  1442,  1443},
    {  1192,  1193,  1314,  1316,  1443,  1444},
    {  1193,  1194,  1315,  1317,  1444,  1445},
    {  1194,  1195,  1316,  1318,  1445,  1446},
    {  1195,  1196,  1317,  1319,  1446,  1447},
    {  1196,  1197,  1318,  1320,  1447,  1448},
    {  1197,  1198,  1319,  1321,  1448,  1449}, // 1320
    {  1198,  1199,  1320,  1322,  1449,  1450},
    {  1199,  1200,  1321,  1323,  1450,  1451},
    {  1200,  1201,  1322,  1324,  1451,  1452},
    {  1201,  1323,  1325,  1452,  1453,  1454},
    {  1201,  1202,  1324,  1326,  1454,  1455},
    {  1202,  1203,  1325,  1327,  1455,  1456},
    {  1203,  1204,  1326,  1328,  1456,  1457},
    {  1204,  1205,  1327,  1329,  1457,  1458},
    {  1205,  1206,  1328,  1330,  1458,  1459},
    {  1206,  1207,  1329,  1331,  1459,  1460},
    {  1207,  1208,  1330,  1332,  1460,  1461},
    {  1208,  1209,  1331,  1333,  1461,  1462},
    {  1209,  1210,  1332,  1334,  1462,  1463},
    {  1210,  1211,  1333,  1335,  1463,  1464},
    {  1211,  1212,  1334,  1336,  1464,  1465},
    {  1212,  1213,  1335,  1337,  1465,  1466},
    {  1213,  1214,  1336,  1338,  1466,  1467},
    {  1214,  1215,  1337,  1339,  1467,  1468},
    {  1215,  1216,  1338,  1340,  1468,  1469},
    {  1216,  1217,  1339,  1341,  1469,  1470}, // 1340
    {  1217,  1218,  1340,  1342,  1470,  1471},
    {  1218,  1219,  1341,  1343,  1471,  1472},
    {  1219,  1220,  1342,  1344,  1472,  1473},
    {  1220,  1221,  1343,  1345,  1473,  1474},
    {  1221,  1344,  1346,  1474,  1475,  1476},
    {  1221,  1222,  1345,  1347,  1476,  1477},
    {  1222,  1223,  1346,  1348,  1477,  1478},
    {  1223,  1224,  1347,  1349,  1478,  1479},
    {  1224,  1225,  1348,  1350,  1479,  1480},
    {  1225,  1226,  1349,  1351,  1480,  1481},
    {  1226,  1227,  1350,  1352,  1481,  1482},
    {  1227,  1228,  1351,  1353,  1482,  1483},
    {  1228,  1229,  1352,  1354,  1483,  1484},
    {  1229,  1230,  1353,  1355,  1484,  1485},
    {  1230,  1231,  1354,  1356,  1485,  1486},
    {  1231,  1232,  1355,  1357,  1486,  1487},
    {  1232,  1233,  1356,  1358,  1487,  1488},
    {  1233,  1234,  1357,  1359,  1488,  1489},
    {  1234,  1235,  1358,  1360,  1489,  1490},
    {  1235,  1236,  1359,  1361,  1490,  1491}, // 1360
    {  1236,  1237,  1360,  1362,  1491,  1492},
    {  1237,  1238,  1361,  1363,  1492,  1493},
    {  1238,  1239,  1362,  1364,  1493,  1494},
    {  1239,  1240,  1363,  1365,  1494,  1495},
    {  1240,  1241,  1364,  1366,  1495,  1496},
    {  1241,  1365,  1367,  1496,  1497,  1498},
    {  1241,  1242,  1366,  1368,  1498,  1499},
    {  1242,  1243,  1367,  1369,  1499,  1500},
    {  1243,  1244,  1368,  1370,  1500,  1501},
    {  1244,  1245,  1369,  1371,  1501,  1502},
    {  1245,  1246,  1370,  1372,  1502,  1503},
    {  1246,  1247,  1371,  1373,  1503,  1504},
    {  1247,  1248,  1372,  1374,  1504,  1505},
    {  1248,  1249,  1373,  1375,  1505,  1506},
    {  1249,  1250,  1374,  1376,  1506,  1507},
    {  1250,  1251,  1375,  1377,  1507,  1508},
    {  1251,  1252,  1376,  1378,  1508,  1509},
    {  1252,  1253,  1377,  1379,  1509,  1510},
    {  1253,  1254,  1378,  1380,  1510,  1511},
    {  1254,  1255,  1379,  1381,  1511,  1512}, // 1380
    {  1255,  1256,  1380,  1382,  1512,  1513},
    {  1256,  1257,  1381,  1383,  1513,  1514},
    {  1257,  1258,  1382,  1384,  1514,  1515},
    {  1258,  1259,  1383,  1385,  1515,  1516},
    {  1259,  1260,  1384,  1386,  1516,  1517},
    {  1141,  1260,  1261,  1385,  1517,  1518},
    {  1261,  1388,  1518,  1519,  1520,  1656},
    {  1261,  1262,  1387,  1389,  1520,  1521},
    {  1262,  1263,  1388,  1390,  1521,  1522},
    {  1263,  1264,  1389,  1391,  1522,  1523},
    {  1264,  1265,  1390,  1392,  1523,  1524},
    {  1265,  1266,  1391,  1393,  1524,  1525},
    {  1266,  1267,  1392,  1394,  1525,  1526},
    {  1267,  1268,  1393,  1395,  1526,  1527},
    {  1268,  1269,  1394,  1396,  1527,  1528},
    {  1269,  1270,  1395,  1397,  1528,  1529},
    {  1270,  1271,  1396,  1398,  1529,  1530},
    {  1271,  1272,  1397,  1399,  1530,  1531},
    {  1272,  1273,  1398,  1400,  1531,  1532},
    {  1273,  1274,  1399,  1401,  1532,  1533}, // 1400
    {  1274,  1275,  1400,  1402,  1533,  1534},
    {  1275,  1276,  1401,  1403,  1534,  1535},
    {  1276,  1277,  1402,  1404,  1535,  1536},
    {  1277,  1278,  1403,  1405,  1536,  1537},
    {  1278,  1279,  1404,  1406,  1537,  1538},
    {  1279,  1280,  1405,  1407,  1538,  1539},
    {  1280,  1281,  1406,  1408,  1539,  1540},
    {  1281,  1282,  1407,  1409,  1540,  1541},
    {  1282,  1408,  1410,  1541,  1542,  1543},
    {  1282,  1283,  1409,  1411,  1543,  1544},
    {  1283,  1284,  1410,  1412,  1544,  1545},
    {  1284,  1285,  1411,  1413,  1545,  1546},
    {  1285,  1286,  1412,  1414,  1546,  1547},
    {  1286,  1287,  1413,  1415,  1547,  1548},
    {  1287,  1288,  1414,  1416,  1548,  1549},
    {  1288,  1289,  1415,  1417,  1549,  1550},
    {  1289,  1290,  1416,  1418,  1550,  1551},
    {  1290,  1291,  1417,  1419,  1551,  1552},
    {  1291,  1292,  1418,  1420,  1552,  1553},
    {  1292,  1293,  1419,  1421,  1553,  1554}, // 1420
    {  1293,  1294,  1420,  1422,  1554,  1555},
    {  1294,  1295,  1421,  1423,  1555,  1556},
    {  1295,  1296,  1422,  1424,  1556,  1557},
    {  1296,  1297,  1423,  1425,  1557,  1558},
    {  1297,  1298,  1424,  1426,  1558,  1559},
    {  1298,  1299,  1425,  1427,  1559,  1560},
    {  1299,  1300,  1426,  1428,  1560,  1561},
    {  1300,  1301,  1427,  1429,  1561,  1562},
    {  1301,  1302,  1428,  1430,  1562,  1563},
    {  1302,  1303,  1429,  1431,  1563,  1564},
    {  1303,  1430,  1432,  1564,  1565,  1566},
    {  1303,  1304,  1431,  1433,  1566,  1567},
    {  1304,  1305,  1432,  1434,  1567,  1568},
    {  1305,  1306,  1433,  1435,  1568,  1569},
    {  1306,  1307,  1434,  1436,  1569,  1570},
    {  1307,  1308,  1435,  1437,  1570,  1571},
    {  1308,  1309,  1436,  1438,  1571,  1572},
    {  1309,  1310,  1437,  1439,  1572,  1573},
    {  1310,  1311,  1438,  1440,  1573,  1574},
    {  1311,  1312,  1439,  1441,  1574,  1575}, // 1440
    {  1312,  1313,  1440,  1442,  1575,  1576},
    {  1313,  1314,  1441,  1443,  1576,  1577},
    {  1314,  1315,  1442,  1444,  1577,  1578},
    {  1315,  1316,  1443,  1445,  1578,  1579},
    {  1316,  1317,  1444,  1446,  1579,  1580},
    {  1317,  1318,  1445,  1447,  1580,  1581},
    {  1318,  1319,  1446,  1448,  1581,  1582},
    {  1319,  1320,  1447,  1449,  1582,  1583},
    {  1320,  1321,  1448,  1450,  1583,  1584},
    {  1321,  1322,  1449,  1451,  1584,  1585},
    {  1322,  1323,  1450,  1452,  1585,  1586},
    {  1323,  1324,  1451,  1453,  1586,  1587},
    {  1324,  1452,  1454,  1587,  1588,  1589},
    {  1324,  1325,  1453,  1455,  1589,  1590},
    {  1325,  1326,  1454,  1456,  1590,  1591},
    {  1326,  1327,  1455,  1457,  1591,  1592},
    {  1327,  1328,  1456,  1458,  1592,  1593},
    {  1328,  1329,  1457,  1459,  1593,  1594},
    {  1329,  1330,  1458,  1460,  1594,  1595},
    {  1330,  1331,  1459,  1461,  1595,  1596}, // 1460
    {  1331,  1332,  1460,  1462,  1596,  1597},
    {  1332,  1333,  1461,  1463,  1597,  1598},
    {  1333,  1334,  1462,  1464,  1598,  1599},
    {  1334,  1335,  1463,  1465,  1599,  1600},
    {  1335,  1336,  1464,  1466,  1600,  1601},
    {  1336,  1337,  1465,  1467,  1601,  1602},
    {  1337,  1338,  1466,  1468,  1602,  1603},
    {  1338,  1339,  1467,  1469,  1603,  1604},
    {  1339,  1340,  1468,  1470,  1604,  1605},
    {  1340,  1341,  1469,  1471,  1605,  1606},
    {  1341,  1342,  1470,  1472,  1606,  1607},
    {  1342,  1343,  1471,  1473,  1607,  1608},
    {  1343,  1344,  1472,  1474,  1608,  1609},
    {  1344,  1345,  1473,  1475,  1609,  1610},
    {  1345,  1474,  1476,  1610,  1611,  1612},
    {  1345,  1346,  1475,  1477,  1612,  1613},
    {  1346,  1347,  1476,  1478,  1613,  1614},
    {  1347,  1348,  1477,  1479,  1614,  1615},
    {  1348,  1349,  1478,  1480,  1615,  1616},
    {  1349,  1350,  1479,  1481,  1616,  1617}, // 1480
    {  1350,  1351,  1480,  1482,  1617,  1618},
    {  1351,  1352,  1481,  1483,  1618,  1619},
    {  1352,  1353,  1482,  1484,  1619,  1620},
    {  1353,  1354,  1483,  1485,  1620,  1621},
    {  1354,  1355,  1484,  1486,  1621,  1622},
    {  1355,  1356,  1485,  1487,  1622,  1623},
    {  1356,  1357,  1486,  1488,  1623,  1624},
    {  1357,  1358,  1487,  1489,  1624,  1625},
    {  1358,  1359,  1488,  1490,  1625,  1626},
    {  1359,  1360,  1489,  1491,  1626,  1627},
    {  1360,  1361,  1490,  1492,  1627,  1628},
    {  1361,  1362,  1491,  1493,  1628,  1629},
    {  1362,  1363,  1492,  1494,  1629,  1630},
    {  1363,  1364,  1493,  1495,  1630,  1631},
    {  1364,  1365,  1494,  1496,  1631,  1632},
    {  1365,  1366,  1495,  1497,  1632,  1633},
    {  1366,  1496,  1498,  1633,  1634,  1635},
    {  1366,  1367,  1497,  1499,  1635,  1636},
    {  1367,  1368,  1498,  1500,  1636,  1637},
    {  1368,  1369,  1499,  1501,  1637,  1638}, // 1500
    {  1369,  1370,  1500,  1502,  1638,  1639},
    {  1370,  1371,  1501,  1503,  1639,  1640},
    {  1371,  1372,  1502,  1504,  1640,  1641},
    {  1372,  1373,  1503,  1505,  1641,  1642},
    {  1373,  1374,  1504,  1506,  1642,  1643},
    {  1374,  1375,  1505,  1507,  1643,  1644},
    {  1375,  1376,  1506,  1508,  1644,  1645},
    {  1376,  1377,  1507,  1509,  1645,  1646},
    {  1377,  1378,  1508,  1510,  1646,  1647},
    {  1378,  1379,  1509,  1511,  1647,  1648},
    {  1379,  1380,  1510,  1512,  1648,  1649},
    {  1380,  1381,  1511,  1513,  1649,  1650},
    {  1381,  1382,  1512,  1514,  1650,  1651},
    {  1382,  1383,  1513,  1515,  1651,  1652},
    {  1383,  1384,  1514,  1516,  1652,  1653},
    {  1384,  1385,  1515,  1517,  1653,  1654},
    {  1385,  1386,  1516,  1518,  1654,  1655},
    {  1261,  1386,  1387,  1517,  1655,  1656},
    {  1387,  1520,  1656,  -1,  -1,  -1},
    {  1387,  1388,  1519,  1521,  1657,  -1}, // 1520
    {  1388,  1389,  1520,  1522,  -1,  -1},
    {  1389,  1390,  1521,  1523,  1658,  -1},
    {  1390,  1391,  1522,  1524,  -1,  -1},
    {  1391,  1392,  1523,  1525,  1659,  -1},
    {  1392,  1393,  1524,  1526,  -1,  -1},
    {  1393,  1394,  1525,  1527,  1660,  -1},
    {  1394,  1395,  1526,  1528,  -1,  -1},
    {  1395,  1396,  1527,  1529,  1661,  -1},
    {  1396,  1397,  1528,  1530,  -1,  -1},
    {  1397,  1398,  1529,  1531,  1662,  -1},
    {  1398,  1399,  1530,  1532,  -1,  -1},
    {  1399,  1400,  1531,  1533,  1663,  -1},
    {  1400,  1401,  1532,  1534,  -1,  -1},
    {  1401,  1402,  1533,  1535,  1664,  -1},
    {  1402,  1403,  1534,  1536,  -1,  -1},
    {  1403,  1404,  1535,  1537,  1665,  -1},
    {  1404,  1405,  1536,  1538,  -1,  -1},
    {  1405,  1406,  1537,  1539,  1666,  -1},
    {  1406,  1407,  1538,  1540,  -1,  -1},
    {  1407,  1408,  1539,  1541,  1667,  -1}, // 1540
    {  1408,  1409,  1540,  1542,  -1,  -1},
    {  1409,  1541,  1543,  1668,  -1,  -1},
    {  1409,  1410,  1542,  1544,  -1,  -1},
    {  1410,  1411,  1543,  1545,  -1,  -1},
    {  1411,  1412,  1544,  1546,  -1,  -1},
    {  1412,  1413,  1545,  1547,  -1,  -1},
    {  1413,  1414,  1546,  1548,  -1,  -1},
    {  1414,  1415,  1547,  1549,  -1,  -1},
    {  1415,  1416,  1548,  1550,  -1,  -1},
    {  1416,  1417,  1549,  1551,  -1,  -1},
    {  1417,  1418,  1550,  1552,  -1,  -1},
    {  1418,  1419,  1551,  1553,  -1,  -1},
    {  1419,  1420,  1552,  1554,  -1,  -1},
    {  1420,  1421,  1553,  1555,  -1,  -1},
    {  1421,  1422,  1554,  1556,  -1,  -1},
    {  1422,  1423,  1555,  1557,  -1,  -1},
    {  1423,  1424,  1556,  1558,  -1,  -1},
    {  1424,  1425,  1557,  1559,  -1,  -1},
    {  1425,  1426,  1558,  1560,  -1,  -1},
    {  1426,  1427,  1559,  1561,  -1,  -1}, // 1560
    {  1427,  1428,  1560,  1562,  -1,  -1},
    {  1428,  1429,  1561,  1563,  -1,  -1},
    {  1429,  1430,  1562,  1564,  -1,  -1},
    {  1430,  1431,  1563,  1565,  -1,  -1},
    {  1431,  1564,  1566,  -1,  -1,  -1},
    {  1431,  1432,  1565,  1567,  -1,  -1},
    {  1432,  1433,  1566,  1568,  -1,  -1},
    {  1433,  1434,  1567,  1569,  -1,  -1},
    {  1434,  1435,  1568,  1570,  -1,  -1},
    {  1435,  1436,  1569,  1571,  -1,  -1},
    {  1436,  1437,  1570,  1572,  -1,  -1},
    {  1437,  1438,  1571,  1573,  -1,  -1},
    {  1438,  1439,  1572,  1574,  -1,  -1},
    {  1439,  1440,  1573,  1575,  -1,  -1},
    {  1440,  1441,  1574,  1576,  -1,  -1},
    {  1441,  1442,  1575,  1577,  -1,  -1},
    {  1442,  1443,  1576,  1578,  -1,  -1},
    {  1443,  1444,  1577,  1579,  -1,  -1},
    {  1444,  1445,  1578,  1580,  -1,  -1},
    {  1445,  1446,  1579,  1581,  -1,  -1}, // 1580
    {  1446,  1447,  1580,  1582,  -1,  -1},
    {  1447,  1448,  1581,  1583,  -1,  -1},
    {  1448,  1449,  1582,  1584,  -1,  -1},
    {  1449,  1450,  1583,  1585,  -1,  -1},
    {  1450,  1451,  1584,  1586,  -1,  -1},
    {  1451,  1452,  1585,  1587,  -1,  -1},
    {  1452,  1453,  1586,  1588,  -1,  -1},
    {  1453,  1587,  1589,  -1,  -1,  -1},
    {  1453,  1454,  1588,  1590,  -1,  -1},
    {  1454,  1455,  1589,  1591,  -1,  -1},
    {  1455,  1456,  1590,  1592,  -1,  -1},
    {  1456,  1457,  1591,  1593,  -1,  -1},
    {  1457,  1458,  1592,  1594,  -1,  -1},
    {  1458,  1459,  1593,  1595,  -1,  -1},
    {  1459,  1460,  1594,  1596,  -1,  -1},
    {  1460,  1461,  1595,  1597,  -1,  -1},
    {  1461,  1462,  1596,  1598,  -1,  -1},
    {  1462,  1463,  1597,  1599,  -1,  -1},
    {  1463,  1464,  1598,  1600,  -1,  -1},
    {  1464,  1465,  1599,  1601,  -1,  -1}, // 1600
    {  1465,  1466,  1600,  1602,  -1,  -1},
    {  1466,  1467,  1601,  1603,  -1,  -1},
    {  1467,  1468,  1602,  1604,  -1,  -1},
    {  1468,  1469,  1603,  1605,  -1,  -1},
    {  1469,  1470,  1604,  1606,  -1,  -1},
    {  1470,  1471,  1605,  1607,  -1,  -1},
    {  1471,  1472,  1606,  1608,  -1,  -1},
    {  1472,  1473,  1607,  1609,  -1,  -1},
    {  1473,  1474,  1608,  1610,  -1,  -1},
    {  1474,  1475,  1609,  1611,  -1,  -1},
    {  1475,  1610,  1612,  -1,  -1,  -1},
    {  1475,  1476,  1611,  1613,  -1,  -1},
    {  1476,  1477,  1612,  1614,  -1,  -1},
    {  1477,  1478,  1613,  1615,  -1,  -1},
    {  1478,  1479,  1614,  1616,  -1,  -1},
    {  1479,  1480,  1615,  1617,  -1,  -1},
    {  1480,  1481,  1616,  1618,  -1,  -1},
    {  1481,  1482,  1617,  1619,  -1,  -1},
    {  1482,  1483,  1618,  1620,  -1,  -1},
    {  1483,  1484,  1619,  1621,  -1,  -1}, // 1620
    {  1484,  1485,  1620,  1622,  -1,  -1},
    {  1485,  1486,  1621,  1623,  -1,  -1},
    {  1486,  1487,  1622,  1624,  -1,  -1},
    {  1487,  1488,  1623,  1625,  -1,  -1},
    {  1488,  1489,  1624,  1626,  -1,  -1},
    {  1489,  1490,  1625,  1627,  -1,  -1},
    {  1490,  1491,  1626,  1628,  -1,  -1},
    {  1491,  1492,  1627,  1629,  -1,  -1},
    {  1492,  1493,  1628,  1630,  -1,  -1},
    {  1493,  1494,  1629,  1631,  -1,  -1},
    {  1494,  1495,  1630,  1632,  -1,  -1},
    {  1495,  1496,  1631,  1633,  -1,  -1},
    {  1496,  1497,  1632,  1634,  -1,  -1},
    {  1497,  1633,  1635,  1717,  -1,  -1},
    {  1497,  1498,  1634,  1636,  -1,  -1},
    {  1498,  1499,  1635,  1637,  1718,  -1},
    {  1499,  1500,  1636,  1638,  -1,  -1},
    {  1500,  1501,  1637,  1639,  1719,  -1},
    {  1501,  1502,  1638,  1640,  -1,  -1},
    {  1502,  1503,  1639,  1641,  1720,  -1}, // 1640
    {  1503,  1504,  1640,  1642,  -1,  -1},
    {  1504,  1505,  1641,  1643,  1721,  -1},
    {  1505,  1506,  1642,  1644,  -1,  -1},
    {  1506,  1507,  1643,  1645,  1722,  -1},
    {  1507,  1508,  1644,  1646,  -1,  -1},
    {  1508,  1509,  1645,  1647,  1723,  -1},
    {  1509,  1510,  1646,  1648,  -1,  -1},
    {  1510,  1511,  1647,  1649,  1724,  -1},
    {  1511,  1512,  1648,  1650,  -1,  -1},
    {  1512,  1513,  1649,  1651,  1725,  -1},
    {  1513,  1514,  1650,  1652,  -1,  -1},
    {  1514,  1515,  1651,  1653,  1726,  -1},
    {  1515,  1516,  1652,  1654,  -1,  -1},
    {  1516,  1517,  1653,  1655,  1727,  -1},
    {  1517,  1518,  1654,  1656,  -1,  -1},
    {  1387,  1518,  1519,  1655,  1728,  -1},
    {  1658,  1729,  1730,  -1,  -1,  -1},
    {  1657,  1659,  1730,  1731,  -1,  -1},
    {  1658,  1660,  1731,  1732,  -1,  -1},
    {  1659,  1661,  1732,  1733,  -1,  -1}, // 1660
    {  1660,  1662,  1733,  1734,  -1,  -1},
    {  1661,  1663,  1734,  1735,  -1,  -1},
    {  1662,  1664,  1735,  1736,  -1,  -1},
    {  1663,  1665,  1736,  1737,  -1,  -1},
    {  1664,  1666,  1737,  1738,  -1,  -1},
    {  1665,  1667,  1738,  1739,  -1,  -1},
    {  1666,  1668,  1739,  1740,  -1,  -1},
    {  1667,  1740,  1741,  -1,  -1,  -1},
    {  1670,  1742,  1743,  -1,  -1,  -1},
    {  1669,  1671,  1743,  1744,  -1,  -1},
    {  1670,  1672,  1744,  1745,  -1,  -1},
    {  1671,  1673,  1745,  1746,  -1,  -1},
    {  1672,  1674,  1746,  1747,  -1,  -1},
    {  1673,  1675,  1747,  1748,  -1,  -1},
    {  1674,  1676,  1748,  1749,  -1,  -1},
    {  1675,  1677,  1749,  1750,  -1,  -1},
    {  1676,  1678,  1750,  1751,  -1,  -1},
    {  1677,  1679,  1751,  1752,  -1,  -1},
    {  1678,  1680,  1752,  1753,  -1,  -1},
    {  1679,  1753,  1754,  -1,  -1,  -1}, // 1680
    {  1682,  1755,  1756,  -1,  -1,  -1},
    {  1681,  1683,  1756,  1757,  -1,  -1},
    {  1682,  1684,  1757,  1758,  -1,  -1},
    {  1683,  1685,  1758,  1759,  -1,  -1},
    {  1684,  1686,  1759,  1760,  -1,  -1},
    {  1685,  1687,  1760,  1761,  -1,  -1},
    {  1686,  1688,  1761,  1762,  -1,  -1},
    {  1687,  1689,  1762,  1763,  -1,  -1},
    {  1688,  1690,  1763,  1764,  -1,  -1},
    {  1689,  1691,  1764,  1765,  -1,  -1},
    {  1690,  1692,  1765,  1766,  -1,  -1},
    {  1691,  1766,  1767,  -1,  -1,  -1},
    {  1694,  1768,  1769,  -1,  -1,  -1},
    {  1693,  1695,  1769,  1770,  -1,  -1},
    {  1694,  1696,  1770,  1771,  -1,  -1},
    {  1695,  1697,  1771,  1772,  -1,  -1},
    {  1696,  1698,  1772,  1773,  -1,  -1},
    {  1697,  1699,  1773,  1774,  -1,  -1},
    {  1698,  1700,  1774,  1775,  -1,  -1},
    {  1699,  1701,  1775,  1776,  -1,  -1}, // 1700
    {  1700,  1702,  1776,  1777,  -1,  -1},
    {  1701,  1703,  1777,  1778,  -1,  -1},
    {  1702,  1704,  1778,  1779,  -1,  -1},
    {  1703,  1779,  1780,  -1,  -1,  -1},
    {  1706,  1781,  1782,  -1,  -1,  -1},
    {  1705,  1707,  1782,  1783,  -1,  -1},
    {  1706,  1708,  1783,  1784,  -1,  -1},
    {  1707,  1709,  1784,  1785,  -1,  -1},
    {  1708,  1710,  1785,  1786,  -1,  -1},
    {  1709,  1711,  1786,  1787,  -1,  -1},
    {  1710,  1712,  1787,  1788,  -1,  -1},
    {  1711,  1713,  1788,  1789,  -1,  -1},
    {  1712,  1714,  1789,  1790,  -1,  -1},
    {  1713,  1715,  1790,  1791,  -1,  -1},
    {  1714,  1716,  1791,  1792,  -1,  -1},
    {  1715,  1792,  1793,  -1,  -1,  -1},
    {  1718,  1794,  1795,  -1,  -1,  -1},
    {  1717,  1719,  1795,  1796,  -1,  -1},
    {  1718,  1720,  1796,  1797,  -1,  -1},
    {  1719,  1721,  1797,  1798,  -1,  -1}, // 1720
    {  1720,  1722,  1798,  1799,  -1,  -1},
    {  1721,  1723,  1799,  1800,  -1,  -1},
    {  1722,  1724,  1800,  1801,  -1,  -1},
    {  1723,  1725,  1801,  1802,  -1,  -1},
    {  1724,  1726,  1802,  1803,  -1,  -1},
    {  1725,  1727,  1803,  1804,  -1,  -1},
    {  1726,  1728,  1804,  1805,  -1,  -1},
    {  1727,  1805,  1806,  -1,  -1,  -1},
    {  1657,  1730,  1807,  1808,  -1,  -1},
    {  1657,  1658,  1729,  1731,  1808,  1809},
    {  1658,  1659,  1730,  1732,  1809,  1810},
    {  1659,  1660,  1731,  1733,  1810,  1811},
    {  1660,  1661,  1732,  1734,  1811,  1812},
    {  1661,  1662,  1733,  1735,  1812,  1813},
    {  1662,  1663,  1734,  1736,  1813,  1814},
    {  1663,  1664,  1735,  1737,  1814,  1815},
    {  1664,  1665,  1736,  1738,  1815,  1816},
    {  1665,  1666,  1737,  1739,  1816,  1817},
    {  1666,  1667,  1738,  1740,  1817,  1818},
    {  1667,  1668,  1739,  1741,  1818,  1819}, // 1740
    {  1668,  1740,  1819,  1820,  -1,  -1},
    {  1669,  1743,  1821,  1822,  -1,  -1},
    {  1669,  1670,  1742,  1744,  1822,  1823},
    {  1670,  1671,  1743,  1745,  1823,  1824},
    {  1671,  1672,  1744,  1746,  1824,  1825},
    {  1672,  1673,  1745,  1747,  1825,  1826},
    {  1673,  1674,  1746,  1748,  1826,  1827},
    {  1674,  1675,  1747,  1749,  1827,  1828},
    {  1675,  1676,  1748,  1750,  1828,  1829},
    {  1676,  1677,  1749,  1751,  1829,  1830},
    {  1677,  1678,  1750,  1752,  1830,  1831},
    {  1678,  1679,  1751,  1753,  1831,  1832},
    {  1679,  1680,  1752,  1754,  1832,  1833},
    {  1680,  1753,  1833,  1834,  -1,  -1},
    {  1681,  1756,  1835,  1836,  -1,  -1},
    {  1681,  1682,  1755,  1757,  1836,  1837},
    {  1682,  1683,  1756,  1758,  1837,  1838},
    {  1683,  1684,  1757,  1759,  1838,  1839},
    {  1684,  1685,  1758,  1760,  1839,  1840},
    {  1685,  1686,  1759,  1761,  1840,  1841}, // 1760
    {  1686,  1687,  1760,  1762,  1841,  1842},
    {  1687,  1688,  1761,  1763,  1842,  1843},
    {  1688,  1689,  1762,  1764,  1843,  1844},
    {  1689,  1690,  1763,  1765,  1844,  1845},
    {  1690,  1691,  1764,  1766,  1845,  1846},
    {  1691,  1692,  1765,  1767,  1846,  1847},
    {  1692,  1766,  1847,  1848,  -1,  -1},
    {  1693,  1769,  1849,  1850,  -1,  -1},
    {  1693,  1694,  1768,  1770,  1850,  1851},
    {  1694,  1695,  1769,  1771,  1851,  1852},
    {  1695,  1696,  1770,  1772,  1852,  1853},
    {  1696,  1697,  1771,  1773,  1853,  1854},
    {  1697,  1698,  1772,  1774,  1854,  1855},
    {  1698,  1699,  1773,  1775,  1855,  1856},
    {  1699,  1700,  1774,  1776,  1856,  1857},
    {  1700,  1701,  1775,  1777,  1857,  1858},
    {  1701,  1702,  1776,  1778,  1858,  1859},
    {  1702,  1703,  1777,  1779,  1859,  1860},
    {  1703,  1704,  1778,  1780,  1860,  1861},
    {  1704,  1779,  1861,  1862,  -1,  -1}, // 1780
    {  1705,  1782,  1863,  1864,  -1,  -1},
    {  1705,  1706,  1781,  1783,  1864,  1865},
    {  1706,  1707,  1782,  1784,  1865,  1866},
    {  1707,  1708,  1783,  1785,  1866,  1867},
    {  1708,  1709,  1784,  1786,  1867,  1868},
    {  1709,  1710,  1785,  1787,  1868,  1869},
    {  1710,  1711,  1786,  1788,  1869,  1870},
    {  1711,  1712,  1787,  1789,  1870,  1871},
    {  1712,  1713,  1788,  1790,  1871,  1872},
    {  1713,  1714,  1789,  1791,  1872,  1873},
    {  1714,  1715,  1790,  1792,  1873,  1874},
    {  1715,  1716,  1791,  1793,  1874,  1875},
    {  1716,  1792,  1875,  1876,  -1,  -1},
    {  1717,  1795,  1877,  1878,  -1,  -1},
    {  1717,  1718,  1794,  1796,  1878,  1879},
    {  1718,  1719,  1795,  1797,  1879,  1880},
    {  1719,  1720,  1796,  1798,  1880,  1881},
    {  1720,  1721,  1797,  1799,  1881,  1882},
    {  1721,  1722,  1798,  1800,  1882,  1883},
    {  1722,  1723,  1799,  1801,  1883,  1884}, // 1800
    {  1723,  1724,  1800,  1802,  1884,  1885},
    {  1724,  1725,  1801,  1803,  1885,  1886},
    {  1725,  1726,  1802,  1804,  1886,  1887},
    {  1726,  1727,  1803,  1805,  1887,  1888},
    {  1727,  1728,  1804,  1806,  1888,  1889},
    {  1728,  1805,  1889,  1890,  -1,  -1},
    {  1729,  1808,  1891,  1892,  -1,  -1},
    {  1729,  1730,  1807,  1809,  1892,  1893},
    {  1730,  1731,  1808,  1810,  1893,  1894},
    {  1731,  1732,  1809,  1811,  1894,  1895},
    {  1732,  1733,  1810,  1812,  1895,  1896},
    {  1733,  1734,  1811,  1813,  1896,  1897},
    {  1734,  1735,  1812,  1814,  1897,  1898},
    {  1735,  1736,  1813,  1815,  1898,  1899},
    {  1736,  1737,  1814,  1816,  1899,  1900},
    {  1737,  1738,  1815,  1817,  1900,  1901},
    {  1738,  1739,  1816,  1818,  1901,  1902},
    {  1739,  1740,  1817,  1819,  1902,  1903},
    {  1740,  1741,  1818,  1820,  1903,  1904},
    {  1741,  1819,  1904,  1905,  -1,  -1}, // 1820
    {  1742,  1822,  1906,  1907,  -1,  -1},
    {  1742,  1743,  1821,  1823,  1907,  1908},
    {  1743,  1744,  1822,  1824,  1908,  1909},
    {  1744,  1745,  1823,  1825,  1909,  1910},
    {  1745,  1746,  1824,  1826,  1910,  1911},
    {  1746,  1747,  1825,  1827,  1911,  1912},
    {  1747,  1748,  1826,  1828,  1912,  1913},
    {  1748,  1749,  1827,  1829,  1913,  1914},
    {  1749,  1750,  1828,  1830,  1914,  1915},
    {  1750,  1751,  1829,  1831,  1915,  1916},
    {  1751,  1752,  1830,  1832,  1916,  1917},
    {  1752,  1753,  1831,  1833,  1917,  1918},
    {  1753,  1754,  1832,  1834,  1918,  1919},
    {  1754,  1833,  1919,  1920,  -1,  -1},
    {  1755,  1836,  1921,  1922,  -1,  -1},
    {  1755,  1756,  1835,  1837,  1922,  1923},
    {  1756,  1757,  1836,  1838,  1923,  1924},
    {  1757,  1758,  1837,  1839,  1924,  1925},
    {  1758,  1759,  1838,  1840,  1925,  1926},
    {  1759,  1760,  1839,  1841,  1926,  1927}, // 1840
    {  1760,  1761,  1840,  1842,  1927,  1928},
    {  1761,  1762,  1841,  1843,  1928,  1929},
    {  1762,  1763,  1842,  1844,  1929,  1930},
    {  1763,  1764,  1843,  1845,  1930,  1931},
    {  1764,  1765,  1844,  1846,  1931,  1932},
    {  1765,  1766,  1845,  1847,  1932,  1933},
    {  1766,  1767,  1846,  1848,  1933,  1934},
    {  1767,  1847,  1934,  1935,  -1,  -1},
    {  1768,  1850,  1936,  1937,  -1,  -1},
    {  1768,  1769,  1849,  1851,  1937,  1938},
    {  1769,  1770,  1850,  1852,  1938,  1939},
    {  1770,  1771,  1851,  1853,  1939,  1940},
    {  1771,  1772,  1852,  1854,  1940,  1941},
    {  1772,  1773,  1853,  1855,  1941,  1942},
    {  1773,  1774,  1854,  1856,  1942,  1943},
    {  1774,  1775,  1855,  1857,  1943,  1944},
    {  1775,  1776,  1856,  1858,  1944,  1945},
    {  1776,  1777,  1857,  1859,  1945,  1946},
    {  1777,  1778,  1858,  1860,  1946,  1947},
    {  1778,  1779,  1859,  1861,  1947,  1948}, // 1860
    {  1779,  1780,  1860,  1862,  1948,  1949},
    {  1780,  1861,  1949,  1950,  -1,  -1},
    {  1781,  1864,  1951,  1952,  -1,  -1},
    {  1781,  1782,  1863,  1865,  1952,  1953},
    {  1782,  1783,  1864,  1866,  1953,  1954},
    {  1783,  1784,  1865,  1867,  1954,  1955},
    {  1784,  1785,  1866,  1868,  1955,  1956},
    {  1785,  1786,  1867,  1869,  1956,  1957},
    {  1786,  1787,  1868,  1870,  1957,  1958},
    {  1787,  1788,  1869,  1871,  1958,  1959},
    {  1788,  1789,  1870,  1872,  1959,  1960},
    {  1789,  1790,  1871,  1873,  1960,  1961},
    {  1790,  1791,  1872,  1874,  1961,  1962},
    {  1791,  1792,  1873,  1875,  1962,  1963},
    {  1792,  1793,  1874,  1876,  1963,  1964},
    {  1793,  1875,  1964,  1965,  -1,  -1},
    {  1794,  1878,  1966,  1967,  -1,  -1},
    {  1794,  1795,  1877,  1879,  1967,  1968},
    {  1795,  1796,  1878,  1880,  1968,  1969},
    {  1796,  1797,  1879,  1881,  1969,  1970}, // 1880
    {  1797,  1798,  1880,  1882,  1970,  1971},
    {  1798,  1799,  1881,  1883,  1971,  1972},
    {  1799,  1800,  1882,  1884,  1972,  1973},
    {  1800,  1801,  1883,  1885,  1973,  1974},
    {  1801,  1802,  1884,  1886,  1974,  1975},
    {  1802,  1803,  1885,  1887,  1975,  1976},
    {  1803,  1804,  1886,  1888,  1976,  1977},
    {  1804,  1805,  1887,  1889,  1977,  1978},
    {  1805,  1806,  1888,  1890,  1978,  1979},
    {  1806,  1889,  1979,  1980,  -1,  -1},
    {  1807,  1892,  1981,  1982,  -1,  -1},
    {  1807,  1808,  1891,  1893,  1982,  1983},
    {  1808,  1809,  1892,  1894,  1983,  1984},
    {  1809,  1810,  1893,  1895,  1984,  1985},
    {  1810,  1811,  1894,  1896,  1985,  1986},
    {  1811,  1812,  1895,  1897,  1986,  1987},
    {  1812,  1813,  1896,  1898,  1987,  1988},
    {  1813,  1814,  1897,  1899,  1988,  1989},
    {  1814,  1815,  1898,  1900,  1989,  1990},
    {  1815,  1816,  1899,  1901,  1990,  1991}, // 1900
    {  1816,  1817,  1900,  1902,  1991,  1992},
    {  1817,  1818,  1901,  1903,  1992,  1993},
    {  1818,  1819,  1902,  1904,  1993,  1994},
    {  1819,  1820,  1903,  1905,  1994,  1995},
    {  1820,  1904,  1995,  1996,  -1,  -1},
    {  1821,  1907,  1997,  1998,  -1,  -1},
    {  1821,  1822,  1906,  1908,  1998,  1999},
    {  1822,  1823,  1907,  1909,  1999,  2000},
    {  1823,  1824,  1908,  1910,  2000,  2001},
    {  1824,  1825,  1909,  1911,  2001,  2002},
    {  1825,  1826,  1910,  1912,  2002,  2003},
    {  1826,  1827,  1911,  1913,  2003,  2004},
    {  1827,  1828,  1912,  1914,  2004,  2005},
    {  1828,  1829,  1913,  1915,  2005,  2006},
    {  1829,  1830,  1914,  1916,  2006,  2007},
    {  1830,  1831,  1915,  1917,  2007,  2008},
    {  1831,  1832,  1916,  1918,  2008,  2009},
    {  1832,  1833,  1917,  1919,  2009,  2010},
    {  1833,  1834,  1918,  1920,  2010,  2011},
    {  1834,  1919,  2011,  2012,  -1,  -1}, // 1920
    {  1835,  1922,  2013,  2014,  -1,  -1},
    {  1835,  1836,  1921,  1923,  2014,  2015},
    {  1836,  1837,  1922,  1924,  2015,  2016},
    {  1837,  1838,  1923,  1925,  2016,  2017},
    {  1838,  1839,  1924,  1926,  2017,  2018},
    {  1839,  1840,  1925,  1927,  2018,  2019},
    {  1840,  1841,  1926,  1928,  2019,  2020},
    {  1841,  1842,  1927,  1929,  2020,  2021},
    {  1842,  1843,  1928,  1930,  2021,  2022},
    {  1843,  1844,  1929,  1931,  2022,  2023},
    {  1844,  1845,  1930,  1932,  2023,  2024},
    {  1845,  1846,  1931,  1933,  2024,  2025},
    {  1846,  1847,  1932,  1934,  2025,  2026},
    {  1847,  1848,  1933,  1935,  2026,  2027},
    {  1848,  1934,  2027,  2028,  -1,  -1},
    {  1849,  1937,  2029,  2030,  -1,  -1},
    {  1849,  1850,  1936,  1938,  2030,  2031},
    {  1850,  1851,  1937,  1939,  2031,  2032},
    {  1851,  1852,  1938,  1940,  2032,  2033},
    {  1852,  1853,  1939,  1941,  2033,  2034}, // 1940
    {  1853,  1854,  1940,  1942,  2034,  2035},
    {  1854,  1855,  1941,  1943,  2035,  2036},
    {  1855,  1856,  1942,  1944,  2036,  2037},
    {  1856,  1857,  1943,  1945,  2037,  2038},
    {  1857,  1858,  1944,  1946,  2038,  2039},
    {  1858,  1859,  1945,  1947,  2039,  2040},
    {  1859,  1860,  1946,  1948,  2040,  2041},
    {  1860,  1861,  1947,  1949,  2041,  2042},
    {  1861,  1862,  1948,  1950,  2042,  2043},
    {  1862,  1949,  2043,  2044,  -1,  -1},
    {  1863,  1952,  2045,  2046,  -1,  -1},
    {  1863,  1864,  1951,  1953,  2046,  2047},
    {  1864,  1865,  1952,  1954,  2047,  2048},
    {  1865,  1866,  1953,  1955,  2048,  2049},
    {  1866,  1867,  1954,  1956,  2049,  2050},
    {  1867,  1868,  1955,  1957,  2050,  2051},
    {  1868,  1869,  1956,  1958,  2051,  2052},
    {  1869,  1870,  1957,  1959,  2052,  2053},
    {  1870,  1871,  1958,  1960,  2053,  2054},
    {  1871,  1872,  1959,  1961,  2054,  2055}, // 1960
    {  1872,  1873,  1960,  1962,  2055,  2056},
    {  1873,  1874,  1961,  1963,  2056,  2057},
    {  1874,  1875,  1962,  1964,  2057,  2058},
    {  1875,  1876,  1963,  1965,  2058,  2059},
    {  1876,  1964,  2059,  2060,  -1,  -1},
    {  1877,  1967,  2061,  2062,  -1,  -1},
    {  1877,  1878,  1966,  1968,  2062,  2063},
    {  1878,  1879,  1967,  1969,  2063,  2064},
    {  1879,  1880,  1968,  1970,  2064,  2065},
    {  1880,  1881,  1969,  1971,  2065,  2066},
    {  1881,  1882,  1970,  1972,  2066,  2067},
    {  1882,  1883,  1971,  1973,  2067,  2068},
    {  1883,  1884,  1972,  1974,  2068,  2069},
    {  1884,  1885,  1973,  1975,  2069,  2070},
    {  1885,  1886,  1974,  1976,  2070,  2071},
    {  1886,  1887,  1975,  1977,  2071,  2072},
    {  1887,  1888,  1976,  1978,  2072,  2073},
    {  1888,  1889,  1977,  1979,  2073,  2074},
    {  1889,  1890,  1978,  1980,  2074,  2075},
    {  1890,  1979,  2075,  2076,  -1,  -1}, // 1980
    {  1891,  1982,  2077,  2078,  -1,  -1},
    {  1891,  1892,  1981,  1983,  2078,  2079},
    {  1892,  1893,  1982,  1984,  2079,  2080},
    {  1893,  1894,  1983,  1985,  2080,  2081},
    {  1894,  1895,  1984,  1986,  2081,  2082},
    {  1895,  1896,  1985,  1987,  2082,  2083},
    {  1896,  1897,  1986,  1988,  2083,  2084},
    {  1897,  1898,  1987,  1989,  2084,  2085},
    {  1898,  1899,  1988,  1990,  2085,  2086},
    {  1899,  1900,  1989,  1991,  2086,  2087},
    {  1900,  1901,  1990,  1992,  2087,  2088},
    {  1901,  1902,  1991,  1993,  2088,  2089},
    {  1902,  1903,  1992,  1994,  2089,  2090},
    {  1903,  1904,  1993,  1995,  2090,  2091},
    {  1904,  1905,  1994,  1996,  2091,  2092},
    {  1905,  1995,  2092,  2093,  -1,  -1},
    {  1906,  1998,  2094,  2095,  -1,  -1},
    {  1906,  1907,  1997,  1999,  2095,  2096},
    {  1907,  1908,  1998,  2000,  2096,  2097},
    {  1908,  1909,  1999,  2001,  2097,  2098}, // 2000
    {  1909,  1910,  2000,  2002,  2098,  2099},
    {  1910,  1911,  2001,  2003,  2099,  2100},
    {  1911,  1912,  2002,  2004,  2100,  2101},
    {  1912,  1913,  2003,  2005,  2101,  2102},
    {  1913,  1914,  2004,  2006,  2102,  2103},
    {  1914,  1915,  2005,  2007,  2103,  2104},
    {  1915,  1916,  2006,  2008,  2104,  2105},
    {  1916,  1917,  2007,  2009,  2105,  2106},
    {  1917,  1918,  2008,  2010,  2106,  2107},
    {  1918,  1919,  2009,  2011,  2107,  2108},
    {  1919,  1920,  2010,  2012,  2108,  2109},
    {  1920,  2011,  2109,  2110,  -1,  -1},
    {  1921,  2014,  2111,  2112,  -1,  -1},
    {  1921,  1922,  2013,  2015,  2112,  2113},
    {  1922,  1923,  2014,  2016,  2113,  2114},
    {  1923,  1924,  2015,  2017,  2114,  2115},
    {  1924,  1925,  2016,  2018,  2115,  2116},
    {  1925,  1926,  2017,  2019,  2116,  2117},
    {  1926,  1927,  2018,  2020,  2117,  2118},
    {  1927,  1928,  2019,  2021,  2118,  2119}, // 2020
    {  1928,  1929,  2020,  2022,  2119,  2120},
    {  1929,  1930,  2021,  2023,  2120,  2121},
    {  1930,  1931,  2022,  2024,  2121,  2122},
    {  1931,  1932,  2023,  2025,  2122,  2123},
    {  1932,  1933,  2024,  2026,  2123,  2124},
    {  1933,  1934,  2025,  2027,  2124,  2125},
    {  1934,  1935,  2026,  2028,  2125,  2126},
    {  1935,  2027,  2126,  2127,  -1,  -1},
    {  1936,  2030,  2128,  2129,  -1,  -1},
    {  1936,  1937,  2029,  2031,  2129,  2130},
    {  1937,  1938,  2030,  2032,  2130,  2131},
    {  1938,  1939,  2031,  2033,  2131,  2132},
    {  1939,  1940,  2032,  2034,  2132,  2133},
    {  1940,  1941,  2033,  2035,  2133,  2134},
    {  1941,  1942,  2034,  2036,  2134,  2135},
    {  1942,  1943,  2035,  2037,  2135,  2136},
    {  1943,  1944,  2036,  2038,  2136,  2137},
    {  1944,  1945,  2037,  2039,  2137,  2138},
    {  1945,  1946,  2038,  2040,  2138,  2139},
    {  1946,  1947,  2039,  2041,  2139,  2140}, // 2040
    {  1947,  1948,  2040,  2042,  2140,  2141},
    {  1948,  1949,  2041,  2043,  2141,  2142},
    {  1949,  1950,  2042,  2044,  2142,  2143},
    {  1950,  2043,  2143,  2144,  -1,  -1},
    {  1951,  2046,  2145,  2146,  -1,  -1},
    {  1951,  1952,  2045,  2047,  2146,  2147},
    {  1952,  1953,  2046,  2048,  2147,  2148},
    {  1953,  1954,  2047,  2049,  2148,  2149},
    {  1954,  1955,  2048,  2050,  2149,  2150},
    {  1955,  1956,  2049,  2051,  2150,  2151},
    {  1956,  1957,  2050,  2052,  2151,  2152},
    {  1957,  1958,  2051,  2053,  2152,  2153},
    {  1958,  1959,  2052,  2054,  2153,  2154},
    {  1959,  1960,  2053,  2055,  2154,  2155},
    {  1960,  1961,  2054,  2056,  2155,  2156},
    {  1961,  1962,  2055,  2057,  2156,  2157},
    {  1962,  1963,  2056,  2058,  2157,  2158},
    {  1963,  1964,  2057,  2059,  2158,  2159},
    {  1964,  1965,  2058,  2060,  2159,  2160},
    {  1965,  2059,  2160,  2161,  -1,  -1}, // 2060
    {  1966,  2062,  2162,  2163,  -1,  -1},
    {  1966,  1967,  2061,  2063,  2163,  2164},
    {  1967,  1968,  2062,  2064,  2164,  2165},
    {  1968,  1969,  2063,  2065,  2165,  2166},
    {  1969,  1970,  2064,  2066,  2166,  2167},
    {  1970,  1971,  2065,  2067,  2167,  2168},
    {  1971,  1972,  2066,  2068,  2168,  2169},
    {  1972,  1973,  2067,  2069,  2169,  2170},
    {  1973,  1974,  2068,  2070,  2170,  2171},
    {  1974,  1975,  2069,  2071,  2171,  2172},
    {  1975,  1976,  2070,  2072,  2172,  2173},
    {  1976,  1977,  2071,  2073,  2173,  2174},
    {  1977,  1978,  2072,  2074,  2174,  2175},
    {  1978,  1979,  2073,  2075,  2175,  2176},
    {  1979,  1980,  2074,  2076,  2176,  2177},
    {  1980,  2075,  2177,  2178,  -1,  -1},
    {  1981,  2078,  2179,  2180,  -1,  -1},
    {  1981,  1982,  2077,  2079,  2180,  2181},
    {  1982,  1983,  2078,  2080,  2181,  2182},
    {  1983,  1984,  2079,  2081,  2182,  2183}, // 2080
    {  1984,  1985,  2080,  2082,  2183,  2184},
    {  1985,  1986,  2081,  2083,  2184,  2185},
    {  1986,  1987,  2082,  2084,  2185,  2186},
    {  1987,  1988,  2083,  2085,  2186,  2187},
    {  1988,  1989,  2084,  2086,  2187,  2188},
    {  1989,  1990,  2085,  2087,  2188,  2189},
    {  1990,  1991,  2086,  2088,  2189,  2190},
    {  1991,  1992,  2087,  2089,  2190,  2191},
    {  1992,  1993,  2088,  2090,  2191,  2192},
    {  1993,  1994,  2089,  2091,  2192,  2193},
    {  1994,  1995,  2090,  2092,  2193,  2194},
    {  1995,  1996,  2091,  2093,  2194,  2195},
    {  1996,  2092,  2195,  2196,  -1,  -1},
    {  1997,  2095,  2197,  2198,  -1,  -1},
    {  1997,  1998,  2094,  2096,  2198,  2199},
    {  1998,  1999,  2095,  2097,  2199,  2200},
    {  1999,  2000,  2096,  2098,  2200,  2201},
    {  2000,  2001,  2097,  2099,  2201,  2202},
    {  2001,  2002,  2098,  2100,  2202,  2203},
    {  2002,  2003,  2099,  2101,  2203,  2204}, // 2100
    {  2003,  2004,  2100,  2102,  2204,  2205},
    {  2004,  2005,  2101,  2103,  2205,  2206},
    {  2005,  2006,  2102,  2104,  2206,  2207},
    {  2006,  2007,  2103,  2105,  2207,  2208},
    {  2007,  2008,  2104,  2106,  2208,  2209},
    {  2008,  2009,  2105,  2107,  2209,  2210},
    {  2009,  2010,  2106,  2108,  2210,  2211},
    {  2010,  2011,  2107,  2109,  2211,  2212},
    {  2011,  2012,  2108,  2110,  2212,  2213},
    {  2012,  2109,  2213,  2214,  -1,  -1},
    {  2013,  2112,  2215,  2216,  -1,  -1},
    {  2013,  2014,  2111,  2113,  2216,  2217},
    {  2014,  2015,  2112,  2114,  2217,  2218},
    {  2015,  2016,  2113,  2115,  2218,  2219},
    {  2016,  2017,  2114,  2116,  2219,  2220},
    {  2017,  2018,  2115,  2117,  2220,  2221},
    {  2018,  2019,  2116,  2118,  2221,  2222},
    {  2019,  2020,  2117,  2119,  2222,  2223},
    {  2020,  2021,  2118,  2120,  2223,  2224},
    {  2021,  2022,  2119,  2121,  2224,  2225}, // 2120
    {  2022,  2023,  2120,  2122,  2225,  2226},
    {  2023,  2024,  2121,  2123,  2226,  2227},
    {  2024,  2025,  2122,  2124,  2227,  2228},
    {  2025,  2026,  2123,  2125,  2228,  2229},
    {  2026,  2027,  2124,  2126,  2229,  2230},
    {  2027,  2028,  2125,  2127,  2230,  2231},
    {  2028,  2126,  2231,  2232,  -1,  -1},
    {  2029,  2129,  2233,  2234,  -1,  -1},
    {  2029,  2030,  2128,  2130,  2234,  2235},
    {  2030,  2031,  2129,  2131,  2235,  2236},
    {  2031,  2032,  2130,  2132,  2236,  2237},
    {  2032,  2033,  2131,  2133,  2237,  2238},
    {  2033,  2034,  2132,  2134,  2238,  2239},
    {  2034,  2035,  2133,  2135,  2239,  2240},
    {  2035,  2036,  2134,  2136,  2240,  2241},
    {  2036,  2037,  2135,  2137,  2241,  2242},
    {  2037,  2038,  2136,  2138,  2242,  2243},
    {  2038,  2039,  2137,  2139,  2243,  2244},
    {  2039,  2040,  2138,  2140,  2244,  2245},
    {  2040,  2041,  2139,  2141,  2245,  2246}, // 2140
    {  2041,  2042,  2140,  2142,  2246,  2247},
    {  2042,  2043,  2141,  2143,  2247,  2248},
    {  2043,  2044,  2142,  2144,  2248,  2249},
    {  2044,  2143,  2249,  2250,  -1,  -1},
    {  2045,  2146,  2251,  2252,  -1,  -1},
    {  2045,  2046,  2145,  2147,  2252,  2253},
    {  2046,  2047,  2146,  2148,  2253,  2254},
    {  2047,  2048,  2147,  2149,  2254,  2255},
    {  2048,  2049,  2148,  2150,  2255,  2256},
    {  2049,  2050,  2149,  2151,  2256,  2257},
    {  2050,  2051,  2150,  2152,  2257,  2258},
    {  2051,  2052,  2151,  2153,  2258,  2259},
    {  2052,  2053,  2152,  2154,  2259,  2260},
    {  2053,  2054,  2153,  2155,  2260,  2261},
    {  2054,  2055,  2154,  2156,  2261,  2262},
    {  2055,  2056,  2155,  2157,  2262,  2263},
    {  2056,  2057,  2156,  2158,  2263,  2264},
    {  2057,  2058,  2157,  2159,  2264,  2265},
    {  2058,  2059,  2158,  2160,  2265,  2266},
    {  2059,  2060,  2159,  2161,  2266,  2267}, // 2160
    {  2060,  2160,  2267,  2268,  -1,  -1},
    {  2061,  2163,  2269,  2270,  -1,  -1},
    {  2061,  2062,  2162,  2164,  2270,  2271},
    {  2062,  2063,  2163,  2165,  2271,  2272},
    {  2063,  2064,  2164,  2166,  2272,  2273},
    {  2064,  2065,  2165,  2167,  2273,  2274},
    {  2065,  2066,  2166,  2168,  2274,  2275},
    {  2066,  2067,  2167,  2169,  2275,  2276},
    {  2067,  2068,  2168,  2170,  2276,  2277},
    {  2068,  2069,  2169,  2171,  2277,  2278},
    {  2069,  2070,  2170,  2172,  2278,  2279},
    {  2070,  2071,  2171,  2173,  2279,  2280},
    {  2071,  2072,  2172,  2174,  2280,  2281},
    {  2072,  2073,  2173,  2175,  2281,  2282},
    {  2073,  2074,  2174,  2176,  2282,  2283},
    {  2074,  2075,  2175,  2177,  2283,  2284},
    {  2075,  2076,  2176,  2178,  2284,  2285},
    {  2076,  2177,  2285,  2286,  -1,  -1},
    {  2077,  2180,  2287,  2288,  -1,  -1},
    {  2077,  2078,  2179,  2181,  2288,  2289}, // 2180
    {  2078,  2079,  2180,  2182,  2289,  2290},
    {  2079,  2080,  2181,  2183,  2290,  2291},
    {  2080,  2081,  2182,  2184,  2291,  2292},
    {  2081,  2082,  2183,  2185,  2292,  2293},
    {  2082,  2083,  2184,  2186,  2293,  2294},
    {  2083,  2084,  2185,  2187,  2294,  2295},
    {  2084,  2085,  2186,  2188,  2295,  2296},
    {  2085,  2086,  2187,  2189,  2296,  2297},
    {  2086,  2087,  2188,  2190,  2297,  2298},
    {  2087,  2088,  2189,  2191,  2298,  2299},
    {  2088,  2089,  2190,  2192,  2299,  2300},
    {  2089,  2090,  2191,  2193,  2300,  2301},
    {  2090,  2091,  2192,  2194,  2301,  2302},
    {  2091,  2092,  2193,  2195,  2302,  2303},
    {  2092,  2093,  2194,  2196,  2303,  2304},
    {  2093,  2195,  2304,  2305,  -1,  -1},
    {  2094,  2198,  2306,  2307,  -1,  -1},
    {  2094,  2095,  2197,  2199,  2307,  2308},
    {  2095,  2096,  2198,  2200,  2308,  2309},
    {  2096,  2097,  2199,  2201,  2309,  2310}, // 2200
    {  2097,  2098,  2200,  2202,  2310,  2311},
    {  2098,  2099,  2201,  2203,  2311,  2312},
    {  2099,  2100,  2202,  2204,  2312,  2313},
    {  2100,  2101,  2203,  2205,  2313,  2314},
    {  2101,  2102,  2204,  2206,  2314,  2315},
    {  2102,  2103,  2205,  2207,  2315,  2316},
    {  2103,  2104,  2206,  2208,  2316,  2317},
    {  2104,  2105,  2207,  2209,  2317,  2318},
    {  2105,  2106,  2208,  2210,  2318,  2319},
    {  2106,  2107,  2209,  2211,  2319,  2320},
    {  2107,  2108,  2210,  2212,  2320,  2321},
    {  2108,  2109,  2211,  2213,  2321,  2322},
    {  2109,  2110,  2212,  2214,  2322,  2323},
    {  2110,  2213,  2323,  2324,  -1,  -1},
    {  2111,  2216,  2325,  2326,  -1,  -1},
    {  2111,  2112,  2215,  2217,  2326,  2327},
    {  2112,  2113,  2216,  2218,  2327,  2328},
    {  2113,  2114,  2217,  2219,  2328,  2329},
    {  2114,  2115,  2218,  2220,  2329,  2330},
    {  2115,  2116,  2219,  2221,  2330,  2331}, // 2220
    {  2116,  2117,  2220,  2222,  2331,  2332},
    {  2117,  2118,  2221,  2223,  2332,  2333},
    {  2118,  2119,  2222,  2224,  2333,  2334},
    {  2119,  2120,  2223,  2225,  2334,  2335},
    {  2120,  2121,  2224,  2226,  2335,  2336},
    {  2121,  2122,  2225,  2227,  2336,  2337},
    {  2122,  2123,  2226,  2228,  2337,  2338},
    {  2123,  2124,  2227,  2229,  2338,  2339},
    {  2124,  2125,  2228,  2230,  2339,  2340},
    {  2125,  2126,  2229,  2231,  2340,  2341},
    {  2126,  2127,  2230,  2232,  2341,  2342},
    {  2127,  2231,  2342,  2343,  -1,  -1},
    {  2128,  2234,  2344,  2345,  -1,  -1},
    {  2128,  2129,  2233,  2235,  2345,  2346},
    {  2129,  2130,  2234,  2236,  2346,  2347},
    {  2130,  2131,  2235,  2237,  2347,  2348},
    {  2131,  2132,  2236,  2238,  2348,  2349},
    {  2132,  2133,  2237,  2239,  2349,  2350},
    {  2133,  2134,  2238,  2240,  2350,  2351},
    {  2134,  2135,  2239,  2241,  2351,  2352}, // 2240
    {  2135,  2136,  2240,  2242,  2352,  2353},
    {  2136,  2137,  2241,  2243,  2353,  2354},
    {  2137,  2138,  2242,  2244,  2354,  2355},
    {  2138,  2139,  2243,  2245,  2355,  2356},
    {  2139,  2140,  2244,  2246,  2356,  2357},
    {  2140,  2141,  2245,  2247,  2357,  2358},
    {  2141,  2142,  2246,  2248,  2358,  2359},
    {  2142,  2143,  2247,  2249,  2359,  2360},
    {  2143,  2144,  2248,  2250,  2360,  2361},
    {  2144,  2249,  2361,  2362,  -1,  -1},
    {  2145,  2252,  2363,  2364,  -1,  -1},
    {  2145,  2146,  2251,  2253,  2364,  2365},
    {  2146,  2147,  2252,  2254,  2365,  2366},
    {  2147,  2148,  2253,  2255,  2366,  2367},
    {  2148,  2149,  2254,  2256,  2367,  2368},
    {  2149,  2150,  2255,  2257,  2368,  2369},
    {  2150,  2151,  2256,  2258,  2369,  2370},
    {  2151,  2152,  2257,  2259,  2370,  2371},
    {  2152,  2153,  2258,  2260,  2371,  2372},
    {  2153,  2154,  2259,  2261,  2372,  2373}, // 2260
    {  2154,  2155,  2260,  2262,  2373,  2374},
    {  2155,  2156,  2261,  2263,  2374,  2375},
    {  2156,  2157,  2262,  2264,  2375,  2376},
    {  2157,  2158,  2263,  2265,  2376,  2377},
    {  2158,  2159,  2264,  2266,  2377,  2378},
    {  2159,  2160,  2265,  2267,  2378,  2379},
    {  2160,  2161,  2266,  2268,  2379,  2380},
    {  2161,  2267,  2380,  2381,  -1,  -1},
    {  2162,  2270,  2382,  2383,  -1,  -1},
    {  2162,  2163,  2269,  2271,  2383,  2384},
    {  2163,  2164,  2270,  2272,  2384,  2385},
    {  2164,  2165,  2271,  2273,  2385,  2386},
    {  2165,  2166,  2272,  2274,  2386,  2387},
    {  2166,  2167,  2273,  2275,  2387,  2388},
    {  2167,  2168,  2274,  2276,  2388,  2389},
    {  2168,  2169,  2275,  2277,  2389,  2390},
    {  2169,  2170,  2276,  2278,  2390,  2391},
    {  2170,  2171,  2277,  2279,  2391,  2392},
    {  2171,  2172,  2278,  2280,  2392,  2393},
    {  2172,  2173,  2279,  2281,  2393,  2394}, // 2280
    {  2173,  2174,  2280,  2282,  2394,  2395},
    {  2174,  2175,  2281,  2283,  2395,  2396},
    {  2175,  2176,  2282,  2284,  2396,  2397},
    {  2176,  2177,  2283,  2285,  2397,  2398},
    {  2177,  2178,  2284,  2286,  2398,  2399},
    {  2178,  2285,  2399,  2400,  -1,  -1},
    {  2179,  2288,  -1,  -1,  -1,  -1},
    {  2179,  2180,  2287,  2289,  -1,  -1},
    {  2180,  2181,  2288,  2290,  -1,  -1},
    {  2181,  2182,  2289,  2291,  -1,  -1},
    {  2182,  2183,  2290,  2292,  -1,  -1},
    {  2183,  2184,  2291,  2293,  -1,  -1},
    {  2184,  2185,  2292,  2294,  -1,  -1},
    {  2185,  2186,  2293,  2295,  -1,  -1},
    {  2186,  2187,  2294,  2296,  -1,  -1},
    {  2187,  2188,  2295,  2297,  -1,  -1},
    {  2188,  2189,  2296,  2298,  -1,  -1},
    {  2189,  2190,  2297,  2299,  -1,  -1},
    {  2190,  2191,  2298,  2300,  -1,  -1},
    {  2191,  2192,  2299,  2301,  -1,  -1}, // 2300
    {  2192,  2193,  2300,  2302,  -1,  -1},
    {  2193,  2194,  2301,  2303,  -1,  -1},
    {  2194,  2195,  2302,  2304,  -1,  -1},
    {  2195,  2196,  2303,  2305,  -1,  -1},
    {  2196,  2304,  -1,  -1,  -1,  -1},
    {  2197,  2307,  -1,  -1,  -1,  -1},
    {  2197,  2198,  2306,  2308,  -1,  -1},
    {  2198,  2199,  2307,  2309,  -1,  -1},
    {  2199,  2200,  2308,  2310,  -1,  -1},
    {  2200,  2201,  2309,  2311,  -1,  -1},
    {  2201,  2202,  2310,  2312,  -1,  -1},
    {  2202,  2203,  2311,  2313,  -1,  -1},
    {  2203,  2204,  2312,  2314,  -1,  -1},
    {  2204,  2205,  2313,  2315,  -1,  -1},
    {  2205,  2206,  2314,  2316,  -1,  -1},
    {  2206,  2207,  2315,  2317,  -1,  -1},
    {  2207,  2208,  2316,  2318,  -1,  -1},
    {  2208,  2209,  2317,  2319,  -1,  -1},
    {  2209,  2210,  2318,  2320,  -1,  -1},
    {  2210,  2211,  2319,  2321,  -1,  -1}, // 2320
    {  2211,  2212,  2320,  2322,  -1,  -1},
    {  2212,  2213,  2321,  2323,  -1,  -1},
    {  2213,  2214,  2322,  2324,  -1,  -1},
    {  2214,  2323,  -1,  -1,  -1,  -1},
    {  2215,  2326,  -1,  -1,  -1,  -1},
    {  2215,  2216,  2325,  2327,  -1,  -1},
    {  2216,  2217,  2326,  2328,  -1,  -1},
    {  2217,  2218,  2327,  2329,  -1,  -1},
    {  2218,  2219,  2328,  2330,  -1,  -1},
    {  2219,  2220,  2329,  2331,  -1,  -1},
    {  2220,  2221,  2330,  2332,  -1,  -1},
    {  2221,  2222,  2331,  2333,  -1,  -1},
    {  2222,  2223,  2332,  2334,  -1,  -1},
    {  2223,  2224,  2333,  2335,  -1,  -1},
    {  2224,  2225,  2334,  2336,  -1,  -1},
    {  2225,  2226,  2335,  2337,  -1,  -1},
    {  2226,  2227,  2336,  2338,  -1,  -1},
    {  2227,  2228,  2337,  2339,  -1,  -1},
    {  2228,  2229,  2338,  2340,  -1,  -1},
    {  2229,  2230,  2339,  2341,  -1,  -1}, // 2340
    {  2230,  2231,  2340,  2342,  -1,  -1},
    {  2231,  2232,  2341,  2343,  -1,  -1},
    {  2232,  2342,  -1,  -1,  -1,  -1},
    {  2233,  2345,  -1,  -1,  -1,  -1},
    {  2233,  2234,  2344,  2346,  -1,  -1},
    {  2234,  2235,  2345,  2347,  -1,  -1},
    {  2235,  2236,  2346,  2348,  -1,  -1},
    {  2236,  2237,  2347,  2349,  -1,  -1},
    {  2237,  2238,  2348,  2350,  -1,  -1},
    {  2238,  2239,  2349,  2351,  -1,  -1},
    {  2239,  2240,  2350,  2352,  -1,  -1},
    {  2240,  2241,  2351,  2353,  -1,  -1},
    {  2241,  2242,  2352,  2354,  -1,  -1},
    {  2242,  2243,  2353,  2355,  -1,  -1},
    {  2243,  2244,  2354,  2356,  -1,  -1},
    {  2244,  2245,  2355,  2357,  -1,  -1},
    {  2245,  2246,  2356,  2358,  -1,  -1},
    {  2246,  2247,  2357,  2359,  -1,  -1},
    {  2247,  2248,  2358,  2360,  -1,  -1},
    {  2248,  2249,  2359,  2361,  -1,  -1}, // 2360
    {  2249,  2250,  2360,  2362,  -1,  -1},
    {  2250,  2361,  -1,  -1,  -1,  -1},
    {  2251,  2364,  -1,  -1,  -1,  -1},
    {  2251,  2252,  2363,  2365,  -1,  -1},
    {  2252,  2253,  2364,  2366,  -1,  -1},
    {  2253,  2254,  2365,  2367,  -1,  -1},
    {  2254,  2255,  2366,  2368,  -1,  -1},
    {  2255,  2256,  2367,  2369,  -1,  -1},
    {  2256,  2257,  2368,  2370,  -1,  -1},
    {  2257,  2258,  2369,  2371,  -1,  -1},
    {  2258,  2259,  2370,  2372,  -1,  -1},
    {  2259,  2260,  2371,  2373,  -1,  -1},
    {  2260,  2261,  2372,  2374,  -1,  -1},
    {  2261,  2262,  2373,  2375,  -1,  -1},
    {  2262,  2263,  2374,  2376,  -1,  -1},
    {  2263,  2264,  2375,  2377,  -1,  -1},
    {  2264,  2265,  2376,  2378,  -1,  -1},
    {  2265,  2266,  2377,  2379,  -1,  -1},
    {  2266,  2267,  2378,  2380,  -1,  -1},
    {  2267,  2268,  2379,  2381,  -1,  -1}, // 2380
    {  2268,  2380,  -1,  -1,  -1,  -1},
    {  2269,  2383,  -1,  -1,  -1,  -1},
    {  2269,  2270,  2382,  2384,  -1,  -1},
    {  2270,  2271,  2383,  2385,  -1,  -1},
    {  2271,  2272,  2384,  2386,  -1,  -1},
    {  2272,  2273,  2385,  2387,  -1,  -1},
    {  2273,  2274,  2386,  2388,  -1,  -1},
    {  2274,  2275,  2387,  2389,  -1,  -1},
    {  2275,  2276,  2388,  2390,  -1,  -1},
    {  2276,  2277,  2389,  2391,  -1,  -1},
    {  2277,  2278,  2390,  2392,  -1,  -1},
    {  2278,  2279,  2391,  2393,  -1,  -1},
    {  2279,  2280,  2392,  2394,  -1,  -1},
    {  2280,  2281,  2393,  2395,  -1,  -1},
    {  2281,  2282,  2394,  2396,  -1,  -1},
    {  2282,  2283,  2395,  2397,  -1,  -1},
    {  2283,  2284,  2396,  2398,  -1,  -1},
    {  2284,  2285,  2397,  2399,  -1,  -1},
    {  2285,  2286,  2398,  2400,  -1,  -1},
    {  2286,  2399,  -1,  -1,  -1,  -1} // 2400
};

    for (Int_t i=0; i<2401; i++)
        (*this)[i].SetNeighbors(nn[i][0], nn[i][1], nn[i][2],
                                nn[i][3], nn[i][4], nn[i][5]);
}
