/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  2001 <mailto:tbretz@uni-sw.gwdg.de>
!              Wolfgang Wittek  2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHHillas
//
// This class contains histograms for the source independent image parameters
//
/////////////////////////////////////////////////////////////////////////////

#include "MHHillas.h"

#include <math.h>

#include <TH1.h>
#include <TH2.h>
#include <TPad.h>
#include <TStyle.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MHillas.h"
#include "MGeomCam.h"
#include "MBinning.h"


ClassImp(MHHillas);

// --------------------------------------------------------------------------
//
// Setup four histograms for Width, Length
//
MHHillas::MHHillas(const char *name, const char *title)
    : fMm2Deg(1), fUseMmScale(kFALSE)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHHillas";
    fTitle = title ? title : "Source independent image parameters";

    fLength = new TH1F("Length", "Length of Ellipse", 100, 0, 300);
    fLength->SetDirectory(NULL);
    fLength->SetXTitle("Length [mm]");
    fLength->SetYTitle("Counts");

    fWidth  = new TH1F("Width",  "Width of Ellipse",  100, 0, 300);
    fWidth->SetDirectory(NULL);
    fWidth->SetXTitle("Width [mm]");
    fWidth->SetYTitle("Counts");

    fDistC  = new TH1F("DistC",  "Distance from center of camera",  100, 0, 600);
    fDistC->SetDirectory(NULL);
    fDistC->SetXTitle("Distance [mm]");
    fDistC->SetYTitle("Counts");

    fDelta  = new TH1F("Delta",  "Angle (Main axis - x-axis)", 100, -90, 90);
    fDelta->SetDirectory(NULL);
    fDelta->SetXTitle("Delta [\\circ]");
    fDelta->SetYTitle("Counts");

    MBinning bins;
    bins.SetEdgesLog(50, 1, 1e7);

    fSize  = new TH1F;
    fSize->SetName("Size");
    fSize->SetTitle("Number of Photons");
    fSize->SetDirectory(NULL);
    fSize->SetXTitle("Size");
    fSize->SetYTitle("Counts");
    fSize->GetXaxis()->SetTitleOffset(1.2);
    fSize->GetXaxis()->SetLabelOffset(-0.015);

    MH::SetBinning(fSize, &bins);

    fCenter = new TH2F("Center", "Center of Ellipse", 60, -600, 600, 60, -600, 600);
    fCenter->SetDirectory(NULL);
    fCenter->SetXTitle("x [mm]");
    fCenter->SetYTitle("y [mm]");
    fCenter->SetZTitle("Counts");
}

// --------------------------------------------------------------------------
//
// Delete the histograms
//
MHHillas::~MHHillas()
{
    delete fLength;
    delete fWidth;

    delete fDistC;
    delete fDelta;

    delete fSize;
    delete fCenter;
}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning (with the names 'BinningWidth' and 'BinningLength')
// are found in the parameter list
// Use this function if you want to set the conversion factor which
// is used to convert the mm-scale in the camera plain into the deg-scale
// used for histogram presentations. The conversion factor is part of
// the camera geometry. Please create a corresponding MGeomCam container.
//
Bool_t MHHillas::SetupFill(const MParList *plist)
{
    const MBinning *binsw = (MBinning*)plist->FindObject("BinningWidth");
    const MBinning *binsl = (MBinning*)plist->FindObject("BinningLength");
    const MBinning *binsd = (MBinning*)plist->FindObject("BinningDist");
    const MBinning *binsc = (MBinning*)plist->FindObject("BinningCamera");

    if (!binsw || !binsl || !binsd || !binsc)
    {
        *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
        return kFALSE;
    }

    SetBinning(fWidth,  binsw);
    SetBinning(fLength, binsl);
    SetBinning(fDistC,  binsd);
    SetBinning(fCenter, binsc, binsc);

    const MGeomCam *geom = (MGeomCam*)plist->FindObject("MGeomCam");
    if (!geom)
    {
        *fLog << warn << dbginf << "No Camera Geometry available. Using mm-scale for histograms." << endl;
        return kTRUE;
    }

    fMm2Deg     = geom->GetConvMm2Deg();
    fUseMmScale = kFALSE;

    fLength->SetXTitle("Length [\\circ]");
    fWidth->SetXTitle("Width [\\circ]");
    fDistC->SetXTitle("Distance [\\circ]");
    fCenter->SetXTitle("x [\\circ]");
    fCenter->SetYTitle("y [\\circ]");

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Use this function to setup your own conversion factor between degrees
// and millimeters. The conversion factor should be the one calculated in
// MGeomCam. Use this function with Caution: You could create wrong values
// by setting up your own scale factor.
//
void MHHillas::SetMm2Deg(Float_t mmdeg)
{
    if (mmdeg<0)
    {
        *fLog << warn << dbginf << "Warning - Conversion factor < 0 - nonsense. Ignored." << endl;
        return;
    }

    if (fMm2Deg>=0)
        *fLog << warn << dbginf << "Warning - Conversion factor already set. Overwriting" << endl;

    fMm2Deg = mmdeg;
}

// --------------------------------------------------------------------------
//
// With this function you can convert the histogram ('on the fly') between
// degrees and millimeters.
//
void MHHillas::SetMmScale(Bool_t mmscale)
{
    if (fUseMmScale == mmscale)
        return;

    if (fMm2Deg<0)
    {
        *fLog << warn << dbginf << "Warning - Sorry, no conversion factor for conversion available." << endl;
        return;
    }

    if (fUseMmScale)
    {
        MH::ScaleAxis(fLength, 1./fMm2Deg);
        MH::ScaleAxis(fWidth,  1./fMm2Deg);
        MH::ScaleAxis(fDistC,  1./fMm2Deg);
        MH::ScaleAxis(fCenter, 1./fMm2Deg, 1./fMm2Deg);

        fLength->SetXTitle("Length [mm]");
        fWidth->SetXTitle("Width [mm]");
        fCenter->SetXTitle("x [mm]");
        fCenter->SetYTitle("y [mm]");
    }
    else
    {
        MH::ScaleAxis(fLength, fMm2Deg);
        MH::ScaleAxis(fWidth,  fMm2Deg);
        MH::ScaleAxis(fDistC,  fMm2Deg);
        MH::ScaleAxis(fCenter, fMm2Deg, fMm2Deg);

        fLength->SetXTitle("Length [\\circ]");
        fWidth->SetXTitle("Width [\\circ]");
        fDistC->SetXTitle("Distance [\\circ]");
        fCenter->SetXTitle("x [\\circ]");
        fCenter->SetYTitle("y [\\circ]");
    }

    fUseMmScale = mmscale;
}

// --------------------------------------------------------------------------
//
// Fill the histograms with data from a MHillas-Container.
// Be careful: Only call this with an object of type MHillas
//
Bool_t MHHillas::Fill(const MParContainer *par)
{
    const MHillas &h = *(MHillas*)par;

    const Double_t d = sqrt(h.GetMeanX()*h.GetMeanX() + h.GetMeanY()*h.GetMeanY());

    if (fUseMmScale)
    {
        fLength->Fill(h.GetLength());
        fWidth ->Fill(h.GetWidth());
        fDistC ->Fill(d);
        fCenter->Fill(h.GetMeanX(), h.GetMeanY());
    }
    else
    {
        fLength->Fill(fMm2Deg*h.GetLength());
        fWidth ->Fill(fMm2Deg*h.GetWidth());
        fDistC ->Fill(fMm2Deg*d);
        fCenter->Fill(fMm2Deg*h.GetMeanX(), fMm2Deg*h.GetMeanY());
    }

    fDelta->Fill(kRad2Deg*h.GetDelta());
    fSize->Fill(h.GetSize());

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Setup a inversed deep blue sea palette for the fCenter histogram.
//
void MHHillas::SetColors() const
{
    gStyle->SetPalette(51, NULL);
    Int_t c[50];
    for (int i=0; i<50; i++)
        c[49-i] = gStyle->GetColorPalette(i);
    gStyle->SetPalette(50, c);
}

// --------------------------------------------------------------------------
//
// Draw clones of four histograms. So that the object can be deleted
// and the histograms are still visible in the canvas.
// The cloned object are deleted together with the canvas if the canvas is
// destroyed. If you want to handle dostroying the canvas you can get a
// pointer to it from this function
//
TObject *MHHillas::DrawClone(Option_t *opt) const
{
    TCanvas *c = MakeDefCanvas("Hillas", fTitle, 700, 750);
    c->Divide(2,3);

    gROOT->SetSelectedPad(NULL);

    c->cd(1);
    fLength->DrawCopy();

    c->cd(2);
    fWidth->DrawCopy();

    c->cd(3);
    gPad->SetLogx();
    fSize->DrawCopy();

    c->cd(4);
    fDelta->DrawCopy();

    c->cd(5);
    fDistC->DrawCopy();

    c->cd(6);
    SetColors();
    fCenter->DrawCopy("colz");

    c->Modified();
    c->Update();

    return c;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the four histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHHillas::Draw(Option_t *)
{
    if (!gPad)
        MakeDefCanvas("Hillas", fTitle, 700, 750);

    gPad->Divide(2,3);

    gPad->cd(1);
    fLength->Draw();

    gPad->cd(2);
    fWidth->Draw();

    gPad->cd(3);
    gPad->SetLogx();
    fSize->Draw();

    gPad->cd(4);
    fDelta->Draw();

    gPad->cd(5);
    fDistC->Draw();

    gPad->cd(6);
    SetColors();
    fCenter->Draw("colz");

    gPad->Modified();
    gPad->Update();
}
