/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  2001 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHHillasExt
//
// This class contains histograms for every Hillas parameter
//
/////////////////////////////////////////////////////////////////////////////

#include "MHHillasExt.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TLegend.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"

#include "MParList.h"

#include "MBinning.h"
#include "MHillasExt.h"
#include "MHillasSrc.h"

ClassImp(MHHillasExt);

// --------------------------------------------------------------------------
//
// Setup four histograms for Width, Length
//
MHHillasExt::MHHillasExt(const char *name, const char *title,
                         const char *hil)
    : fMm2Deg(1), fUseMmScale(kTRUE)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHHillasExt";
    fTitle = title ? title : "Container for extended Hillas histograms";

    fHilName = hil;
    //*fLog << "MHHillasExt : fHilName = " << fHilName << endl;

    //
    // loop over all Pixels and create two histograms
    // one for the Low and one for the High gain
    // connect all the histogram with the container fHist
    //
    fHConc1.SetLineColor(kBlue);
    fHConc.SetFillStyle(0);

    fHConc.SetDirectory(NULL);
    fHConc1.SetDirectory(NULL);
    fHAsym.SetDirectory(NULL);
    fHM3Long.SetDirectory(NULL);
    fHM3Trans.SetDirectory(NULL);

    fHConc.SetName("Conc2");
    fHConc1.SetName("Conc1");
    fHAsym.SetName("Asymmetry");
    fHM3Long.SetName("3rd Mom Long");
    fHM3Trans.SetName("3rd Mom Trans");

    fHConc.SetTitle("Ratio: Conc");
    fHConc1.SetTitle("Ratio: Conc1");
    fHAsym.SetTitle("Asymmetry");
    fHM3Long.SetTitle("3^{rd} Moment Longitudinal");
    fHM3Trans.SetTitle("3^{rd} Moment Transverse");

    fHConc.SetXTitle("Ratio");
    fHConc1.SetXTitle("Ratio");
    fHAsym.SetXTitle("Asym [mm]");
    fHM3Long.SetXTitle("3^{rd} M_{l} [mm]");
    fHM3Trans.SetXTitle("3^{rd} M_{t} [mm]");

    fHConc.SetYTitle("Counts");
    fHConc1.SetYTitle("Counts");
    fHAsym.SetYTitle("Counts");
    fHM3Long.SetYTitle("Counts");
    fHM3Trans.SetYTitle("Counts");


    MBinning bins;

    bins.SetEdges(100, 0, 1);
    bins.Apply(fHConc);
    bins.Apply(fHConc1);

    bins.SetEdges(101, -326, 326);
    bins.Apply(fHM3Long);
    bins.Apply(fHM3Trans);

    bins.SetEdges(101, -593, 593);
    bins.Apply(fHAsym);
}

// --------------------------------------------------------------------------
//
// Delete the four histograms
//
MHHillasExt::~MHHillasExt()
{
}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning (with the names 'BinningWidth' and 'BinningLength')
// are found in the parameter list
// Use this function if you want to set the conversion factor which
// is used to convert the mm-scale in the camera plain into the deg-scale
// used for histogram presentations. The conversion factor is part of
// the camera geometry. Please create a corresponding MGeomCam container.
//
Bool_t MHHillasExt::SetupFill(const MParList *plist)
{
    TObject *obj = plist->FindObject(fHilName, "MHillas");
    if (!obj)
    {
      *fLog << err << dbginf << "Sorry '" << fHilName 
            << "' not found in parameter list... aborting." << endl;
        return kFALSE;
    }
    if (!obj->InheritsFrom(MHillasExt::Class()))
    {
        *fLog << err << dbginf << "Sorry 'MHillas' doesn't inherit from MHillasExt... aborting." << endl;
        return kFALSE;
    }
    fHillasExt = (MHillasExt*)obj;

    const MGeomCam *geom = (MGeomCam*)plist->FindObject("MGeomCam");
    if (!geom)
        *fLog << warn << GetDescriptor() << ": No Camera Geometry available. Using mm-scale for histograms." << endl;
    else
    {
        fMm2Deg = geom->GetConvMm2Deg();
        SetMmScale(kFALSE);
    }

    ApplyBinning(*plist, "Conc",    &fHConc);
    ApplyBinning(*plist, "Conc1",   &fHConc1);
    ApplyBinning(*plist, "Asym",    &fHAsym);
    ApplyBinning(*plist, "M3Long",  &fHM3Long);
    ApplyBinning(*plist, "M3Trans", &fHM3Trans);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the four histograms with data from a MHillas-Container.
// Be careful: Only call this with an object of type MHillas
//
Bool_t MHHillasExt::Fill(const MParContainer *par)
{
    const MHillasSrc *src = (MHillasSrc*)par;

    const Double_t scale = TMath::Sign(fUseMmScale?1:fMm2Deg, src ? src->GetCosDeltaAlpha() : 1);

    fHConc.Fill(fHillasExt->GetConc());
    fHConc1.Fill(fHillasExt->GetConc1());

    fHAsym.Fill(scale*fHillasExt->GetAsym());
    fHM3Long.Fill(scale*fHillasExt->GetM3Long());
    fHM3Trans.Fill(scale*fHillasExt->GetM3Trans());
    //fHAsymna.Fill(scale*ext.GetAsymna());
    //fHAsym0.Fill(scale*ext.GetAsym0());

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// With this function you can convert the histogram ('on the fly') between
// degrees and millimeters.
//
void MHHillasExt::SetMmScale(Bool_t mmscale)
{
    if (fUseMmScale == mmscale)
        return;

    if (fMm2Deg<0)
    {
        *fLog << warn << dbginf << "Warning - Sorry, no conversion factor for conversion available." << endl;
        return;
    }

    const Double_t scale = mmscale ? 1./fMm2Deg : fMm2Deg;
    MH::ScaleAxis(&fHAsym,    scale);
    MH::ScaleAxis(&fHM3Long,  scale);
    MH::ScaleAxis(&fHM3Trans, scale);

    if (mmscale)
    {
        fHAsym.SetXTitle("Asym [mm]");
        fHM3Long.SetXTitle("3^{rd} M_{l} [mm]");
        fHM3Trans.SetXTitle("3^{rd} M_{t} [mm]");
    }
    else
    {
        fHAsym.SetXTitle("Asym [\\circ]");
        fHM3Long.SetXTitle("3^{rd} M_{l} [\\circ]");
        fHM3Trans.SetXTitle("3^{rd} M_{t} [\\circ]");
    }

    fUseMmScale = mmscale;
}

// --------------------------------------------------------------------------
//
// Use this function to setup your own conversion factor between degrees
// and millimeters. The conversion factor should be the one calculated in
// MGeomCam. Use this function with Caution: You could create wrong values
// by setting up your own scale factor.
//
void MHHillasExt::SetMm2Deg(Float_t mmdeg)
{
    if (mmdeg<0)
    {
        *fLog << warn << dbginf << "Warning - Conversion factor < 0 - nonsense. Ignored." << endl;
        return;
    }

    if (fMm2Deg>=0)
        *fLog << warn << dbginf << "Warning - Conversion factor already set. Overwriting" << endl;

    fMm2Deg = mmdeg;
}

// --------------------------------------------------------------------------
//
// Draw clones of all four histograms. So that the object can be deleted
// and the histograms are still visible in the canvas.
// The cloned object are deleted together with the canvas if the canvas is
// destroyed. If you want to handle dostroying the canvas you can get a
// pointer to it from this function
//
TObject *MHHillasExt::DrawClone(Option_t *opt) const
{
    TCanvas &c = *MakeDefCanvas(this, 720, 540);
    c.Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    c.cd(1);
    DrawCopy(fHConc1, fHConc, "Concentrations");

    c.cd(2);
    ((TH1&)fHAsym).DrawCopy();

    c.cd(3);
    ((TH1&)fHM3Long).DrawCopy();

    c.cd(4);
    ((TH1&)fHM3Trans).DrawCopy();

    c.Modified();
    c.Update();

    return &c;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the four histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHHillasExt::Draw(Option_t *)
{
    if (!gPad)
        MakeDefCanvas(this, 720, 540);

    gPad->Divide(2, 2);

    gPad->cd(1);
    MH::Draw(fHConc1, fHConc, "Concentrations");

    gPad->cd(2);
    fHAsym.Draw();

    gPad->cd(3);
    fHM3Long.Draw();

    gPad->cd(4);
    fHM3Trans.Draw();

    gPad->Modified();
    gPad->Update();
}

TH1 *MHHillasExt::GetHistByName(const TString name)
{
    if (name.Contains("Conc", TString::kIgnoreCase))
        return &fHConc;
    if (name.Contains("Conc1", TString::kIgnoreCase))
        return &fHConc1;
    if (name.Contains("Asym", TString::kIgnoreCase))
        return &fHAsym;
    if (name.Contains("M3Long", TString::kIgnoreCase))
        return &fHM3Long;
    if (name.Contains("M3Trans", TString::kIgnoreCase))
        return &fHM3Trans;

    return NULL;
}
