/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  1/2002 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHEnergyTime                                                            //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHEnergyTime.h" 

#include <TCanvas.h>

#include "MMcEvt.hxx"
#include "MTime.h"

#include "MH.h"
#include "MBinning.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

ClassImp(MHEnergyTime);

using namespace std;

// --------------------------------------------------------------------------
//
//  Creates the three necessary histograms:
//   - selected showers (input)
//   - all showers (input)
//   - collection area (result)
//
MHEnergyTime::MHEnergyTime(const char *name, const char *title)
{ 
    //   initialize the histogram for the distribution r vs E
    //
    //   we set the energy range from 1 Gev to 10000 GeV (in log 5 orders
    //   of magnitude) and for each order we take 10 subdivision --> 50 xbins
    //
    //   we set the radius range from 0 m to 500 m with 10 m bin --> 50 ybins

    fName  = name  ? name  : "MHEnergyTime";
    fTitle = title ? title : "Data to Calculate Collection Area";

    fHist.SetDirectory(NULL);

    fHist.SetXTitle("E [GeV]");
    fHist.SetYTitle("t [s]");
    fHist.SetZTitle("N");
}

Bool_t MHEnergyTime::SetupFill(const MParList *plist)
{
    fTime = (MTime*)plist->FindObject("MTime");
    if (!fTime)
    {
        *fLog << err << dbginf << "MTime not found... aborting." << endl;
        return kFALSE;
    }

    const MBinning* binsenergy = (MBinning*)plist->FindObject("BinningE");
    const MBinning* binstime   = (MBinning*)plist->FindObject("BinningTime");
    if (!binsenergy || !binstime)
    {
        *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
        return kFALSE;
    }

    SetBinning(&fHist, binsenergy, binstime);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill data into the histogram which contains all showers
//
Bool_t MHEnergyTime::Fill(const MParContainer *par, const Stat_t w)
{
    const MMcEvt &mcevt = *(MMcEvt*)par;

    fHist.Fill(mcevt.GetEnergy(), *fTime, w);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw the histogram with all showers
//
void MHEnergyTime::Draw(Option_t* option)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->SetLogy();
    fHist.DrawCopy(option);

    pad->Modified();
    pad->Update();
}

// --------------------------------------------------------------------------
//
//  Calculate the Efficiency (collection area) and set the 'ReadyToSave'
//  flag
//
void MHEnergyTime::Divide(const TH2D *h1, const TH2D *h2)
{
    // Description!

    fHist.Sumw2();
    fHist.Divide((TH2D*)h1, (TH2D*)h2);

    SetReadyToSave();
}
