/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 8/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MSequence
//
/////////////////////////////////////////////////////////////////////////////
#include "MSequence.h"

#include <stdlib.h>

#include <TEnv.h>
#include <TRegexp.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MDirIter.h"

ClassImp(MSequence);

using namespace std;

void MSequence::Split(TString &runs, TArrayI &data) const
{
    const TRegexp regexp("[0-9]+");

    data.Set(0);
    runs = runs.Strip(TString::kTrailing);

    while (!runs.IsNull())
    {
        TString num = runs(regexp);

        const Int_t n = data.GetSize();
        data.Set(n+1);
        data[n] = atoi(num.Data());

        runs.Remove(0, runs.First(num)+num.Length());
    }
}

void MSequence::SetupRuns(MDirIter &iter, const TArrayI &arr, const char *path) const
{
    TString d(path);

    // Setup path
    if (!path)
    {
        d  = Form("/data/MAGIC/Period%03d/rootdata/", fPeriod);
        d += fNight.GetStringFmt("%Y_%m_%d");
    }

    for (int i=0; i<arr.GetSize(); i++)
    {
        TString n;

        // Create file name
        n =  fNight.GetStringFmt("%Y%m%d_");
        n += Form("%05d_*_E.root", arr[i]);

        // Add Path/File to TIter
        iter.AddDirectory(d, n, 0);
    }
}

MSequence::MSequence(const char *fname)
{
    fName  = fname;
    fTitle = Form("Sequence contained in file %s", fName.Data());

    TEnv env(fname);

    TString str;

    fSequence  = env.GetValue("Sequence", -1);
    fLastRun   = env.GetValue("LastRun", -1);
    fNumEvents = env.GetValue("NumEvents", -1);
    fPeriod    = env.GetValue("Period", -1);

    str = env.GetValue("Start", "");
    fStart.SetSqlDateTime(str);
    str = env.GetValue("Night", "");
    str += " 00:00:00";
    fNight.SetSqlDateTime(str);

    fProject      = env.GetValue("Project", "");
    fSource       = env.GetValue("Source", "");
    fTriggerTable = env.GetValue("TriggerTable", "");
    fHvSettings   = env.GetValue("HvSettings", "");

    str = env.GetValue("Runs", "");
    Split(str, fRuns);
    str = env.GetValue("CalRuns", "");
    Split(str, fCalRuns);
    str = env.GetValue("PedRuns", "");
    Split(str, fPedRuns);
    str = env.GetValue("DatRuns", "");
    Split(str, fDatRuns);
}

void MSequence::Print(Option_t *o) const
{
    gLog << all;
    gLog << "Sequence:     " << fSequence << endl;
    gLog << "Period:       " << fPeriod << endl;
    gLog << "Night:        " << fNight << endl << endl;
    gLog << "Start:        " << fStart << endl;
    gLog << "LastRun:      " << fLastRun << endl;
    gLog << "NumEvents:    " << fNumEvents << endl;
    gLog << "Project:      " << fProject << endl;
    gLog << "Source:       " << fSource << endl;
    gLog << "TriggerTable: " << fTriggerTable << endl;
    gLog << "HvSettings:   " << fHvSettings << endl << endl;
    gLog << "Runs:";
    for (int i=0; i<fRuns.GetSize(); i++)
        gLog << " " << fRuns[i];
    gLog << endl;
    gLog << "CalRuns:";
    for (int i=0; i<fCalRuns.GetSize(); i++)
        gLog << " " << fCalRuns[i];
    gLog << endl;
    gLog << "PedRuns:";
    for (int i=0; i<fPedRuns.GetSize(); i++)
        gLog << " " << fPedRuns[i];
    gLog << endl;
    gLog << "DatRuns:";
    for (int i=0; i<fDatRuns.GetSize(); i++)
        gLog << " " << fDatRuns[i];
    gLog << endl;
}

void MSequence::SetupPedRuns(MDirIter &iter, const char *path) const
{
    SetupRuns(iter, fPedRuns, path);
}

void MSequence::SetupDatRuns(MDirIter &iter, const char *path) const
{
    SetupRuns(iter, fDatRuns, path);
}

void MSequence::SetupAllRuns(MDirIter &iter, const char *path) const
{
    SetupRuns(iter, fRuns, path);
}

void MSequence::SetupCalRuns(MDirIter &iter, const char *path) const
{
    SetupRuns(iter, fCalRuns, path);
}
