/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug, 04/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MExtractTime
//
//   Base class for the signal extractors, used the functions 
//   FindTimeHiGain() and FindTimeLoGain() to extract the signal and 
//   substract the pedestal value    
//
//   The following variables have to be set by the derived class and 
//   do not have defaults:
//   - fNumHiGainSamples
//   - fNumLoGainSamples
//   - fSqrtHiGainSamples
//   - fSqrtLoGainSamples
//
// Input Containers:
//   MRawEvtData
//   MRawRunHeader
//   MPedestalCam
//
// Output Containers:
//   MArrivalTimeCam
//
//////////////////////////////////////////////////////////////////////////////
#include "MExtractTime.h"

#include <fstream>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRawEvtData.h"
#include "MRawEvtPixelIter.h"
#include "MRawRunHeader.h"

#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include "MArrivalTimeCam.h"
#include "MArrivalTimePix.h"

ClassImp(MExtractTime);

using namespace std;
const Float_t MExtractTime::fgOffsetLoGain = 1.51;   // 5 ns
// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Set: 
// - all pointers to NULL
// - all variables to 0
// - fSaturationLimit to fgSaturationLimit
//
// Call:
// -  AddToBranchList("MRawEvtData.*")
//
MExtractTime::MExtractTime(const char *name, const char *title)
  : fArrTime(NULL), fNamePedContainer("MPedestalCam")
{

    fName  = name  ? name  : "MExtractTime";
    fTitle = title ? title : "Base class for signal extractors";

    SetOffsetLoGain();
}



// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawEvtData
//  - MRawRunHeader
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MArrivalTimeCam
//
Int_t MExtractTime::PreProcess(MParList *pList)
{

  fRawEvt = (MRawEvtData*)pList->FindObject(AddSerialNumber("MRawEvtData"));
  if (!fRawEvt)
    {
        *fLog << err << AddSerialNumber("MRawEvtData") << " not found... aborting." << endl;
        return kFALSE;
    }
  
  fRunHeader = (MRawRunHeader*)pList->FindObject(AddSerialNumber("MRawRunHeader"));
  if (!fRunHeader)
    {
      *fLog << err << AddSerialNumber("MRawRunHeader") << " not found... aborting." << endl;
      return kFALSE;
    }
  
  
  fArrTime = (MArrivalTimeCam*)pList->FindCreateObj(AddSerialNumber("MArrivalTimeCam"));
  if (!fArrTime)
    return kFALSE;
  

  fPedestals = (MPedestalCam*)pList->FindObject( AddSerialNumber(fNamePedContainer), "MPedestalCam");
  if (!fPedestals)
    {
      *fLog << err << AddSerialNumber("MPedestalCam") << " not found... aborting" << endl;
      return kFALSE;
    }
  
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// The ReInit calls:
// -  MExtractor::ReInit()
//
// Call: 
// - MArrivalTimeCam::SetUsedFADCSlices(fHiGainFirst, fHiGainLast, fNumHiGainSamples,
//                                      fLoGainFirst, fLoGainLast, fNumLoGainSamples);
//
Bool_t MExtractTime::ReInit(MParList *pList)
{
  
  MExtractor::ReInit(pList);

  fArrTime->SetUsedFADCSlices(fHiGainFirst, fHiGainLast+fHiLoLast, fLoGainFirst, fLoGainLast);

  return kTRUE;
}



void MExtractTime::FindTimeHiGain(Byte_t *firstused, Float_t &time, Float_t &dtime, 
                                  Byte_t &sat, const MPedestalPix &ped) const
{
  return;
}

void MExtractTime::FindTimeLoGain(Byte_t *firstused, Float_t &time, Float_t &dtime, 
                                  Byte_t &sat, const MPedestalPix &ped) const
{
  return;
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MArrivalTimeCam container.
//
Int_t MExtractTime::Process()
{


  MRawEvtPixelIter pixel(fRawEvt);
  fArrTime->Clear();

  while (pixel.Next())
    {
      //
      // Find signal in hi- and lo-gain
      //
      Float_t timehi=0., deltatimehi=0.;
      Byte_t sathi=0;

      const Int_t pixid = pixel.GetPixelId();
      const MPedestalPix  &ped = (*fPedestals)[pixid];
      MArrivalTimePix &pix = (*fArrTime)[pixid];

      FindTimeHiGain(pixel.GetHiGainSamples()+fHiGainFirst, timehi, deltatimehi, sathi, ped);

      Float_t timelo=0., deltatimelo=0.;
      Byte_t satlo=0;

      if (sathi)
	  FindTimeLoGain(pixel.GetLoGainSamples()+fLoGainFirst, timelo, deltatimelo, satlo, ped);

      pix.SetArrivalTime(timehi, deltatimehi, timelo-fOffsetLoGain, deltatimelo);
      pix.SetGainSaturation(sathi, sathi, satlo);
 
    } /* while (pixel.Next()) */

    fArrTime->SetReadyToSave();

    return kTRUE;
}

