/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Nicola Galante  12/2004 <mailto:nicola.galante@pi.infn.it>
!   Author(s): Thomas Bretz  12/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2004-2007
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//   MFTriggerPattern
//
//  A Filter for testing the trigger performance using Trigger Pattern.
//
// For files before file version 5 the trigger pattern is set to 00000000.
//
// For more details see: MTriggerPattern
//
/////////////////////////////////////////////////////////////////////////////
#include "MFTriggerPattern.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"
#include "MTriggerPattern.h"

ClassImp(MFTriggerPattern);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. The default is to allow passing any trigger pattern.
//
MFTriggerPattern::MFTriggerPattern(const char *name, const char *title)
    : fPattern(0), fMaskRequiredPrescaled(0), fMaskRequiredUnprescaled(0),
      fMaskDeniedPrescaled(0), fMaskDeniedUnprescaled(0), fDefault(kTRUE)
{
    fName  = name  ? name  : "MFTriggerPattern";
    fTitle = title ? title : "Filter on Trigger Pattern";
}

// --------------------------------------------------------------------------
//
// Copy constructor
//
MFTriggerPattern::MFTriggerPattern(MFTriggerPattern &trigpatt)
: MFilter(trigpatt)
{
    fMaskRequiredPrescaled   = trigpatt.fMaskRequiredPrescaled;
    fMaskRequiredUnprescaled = trigpatt.fMaskRequiredUnprescaled;
}


// --------------------------------------------------------------------------
//
Int_t MFTriggerPattern::PreProcess(MParList *pList)
{
    fPattern = (MTriggerPattern*)pList->FindObject("MTriggerPattern");
    if (!fPattern)
    {
	*fLog << err << "MTriggerPattern not found... abort." << endl;
	return kFALSE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
Int_t MFTriggerPattern::Process()
{
    const Byte_t p = fPattern->GetPrescaled();
    const Byte_t u = fPattern->GetUnprescaled();
    if (p==0 && u==0)
    {
        fResult = fDefault;
        return kTRUE;
    }

    fResult = kFALSE;

    // Check whether all the bits required are ON
    if ( ((p & fMaskRequiredPrescaled)   == fMaskRequiredPrescaled) &&
         ((u & fMaskRequiredUnprescaled) == fMaskRequiredUnprescaled))
      fResult = kTRUE;

    // Now overwrite the result if one of the bits is denied
    if ( (p & fMaskDeniedPrescaled) || (u & fMaskDeniedUnprescaled) )
      fResult = kFALSE;

    return kTRUE;
}

void MFTriggerPattern::RequireTriggerLvl1(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskRequiredPrescaled |= MTriggerPattern::kTriggerLvl1) : (fMaskRequiredUnprescaled |= MTriggerPattern::kTriggerLvl1);
}

void MFTriggerPattern::RequireTriggerLvl2(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskRequiredPrescaled |= MTriggerPattern::kTriggerLvl2) : (fMaskRequiredUnprescaled |= MTriggerPattern::kTriggerLvl2);
}

void MFTriggerPattern::RequireCalibration(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskRequiredPrescaled |= MTriggerPattern::kCalibration) : (fMaskRequiredUnprescaled |= MTriggerPattern::kCalibration);
}

void MFTriggerPattern::RequirePedestal(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskRequiredPrescaled |= MTriggerPattern::kPedestal)    : (fMaskRequiredUnprescaled |= MTriggerPattern::kPedestal);
}

void MFTriggerPattern::RequirePinDiode(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskRequiredPrescaled |= MTriggerPattern::kPinDiode)    : (fMaskRequiredUnprescaled |= MTriggerPattern::kPinDiode);
}

void MFTriggerPattern::RequireSumTrigger(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskRequiredPrescaled |= MTriggerPattern::kSumTrigger)  : (fMaskRequiredUnprescaled |= MTriggerPattern::kSumTrigger);
}


void MFTriggerPattern::DenyTriggerLvl1(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskDeniedPrescaled |= MTriggerPattern::kTriggerLvl1) : (fMaskDeniedUnprescaled |= MTriggerPattern::kTriggerLvl1);
}

void MFTriggerPattern::DenyTriggerLvl2(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskDeniedPrescaled |= MTriggerPattern::kTriggerLvl2) : (fMaskDeniedUnprescaled |= MTriggerPattern::kTriggerLvl2);
}

void MFTriggerPattern::DenyCalibration(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskDeniedPrescaled |= MTriggerPattern::kCalibration) : (fMaskDeniedUnprescaled |= MTriggerPattern::kCalibration);
}

void MFTriggerPattern::DenyPedestal(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskDeniedPrescaled |= MTriggerPattern::kPedestal)    : (fMaskDeniedUnprescaled |= MTriggerPattern::kPedestal);
}

void MFTriggerPattern::DenyPinDiode(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskDeniedPrescaled |= MTriggerPattern::kPinDiode)    : (fMaskDeniedUnprescaled |= MTriggerPattern::kPinDiode);
}

void MFTriggerPattern::DenySumTrigger(Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskDeniedPrescaled |= MTriggerPattern::kSumTrigger)  : (fMaskDeniedUnprescaled |= MTriggerPattern::kSumTrigger);
}

// -------------------------------------------------------------------------
//
// Low level settings. USE THESE ONLY IF YOU ARE AN EXPERT!
//
// You can concatenate bits either by using MTriggerPatter:
//   eg. MTriggerPattern::kTriggerLvl1 & MTiggerPattern::kTriggerLvl2
// of by hexadecimal values:
//   eg. 0xab
//
//  while 0xab can be decoded like:
//
//                                   8421 8421
//       0xa=10=8+2 0xb=11=8+2+1 --> 1010 1011
//
// or vice versa it is easy to get a hexadecimal number from a bit pattern,
//   eg.
//
//       8421 8421
//       0101 1101  -->  4+1=5=0x5 8+4+1=13=0xd --> 0x5d
//
void MFTriggerPattern::SetMaskRequired(const Byte_t mask, Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskRequiredPrescaled = mask) : (fMaskRequiredUnprescaled = mask);
}

// -------------------------------------------------------------------------
//
// Low level settings. USE THESE ONLY IF YOU ARE AN EXPERT!
//
// You can concatenate bits either by using MTriggerPatter:
//   eg. MTriggerPattern::kTriggerLvl1 & MTiggerPattern::kTriggerLvl2
// of by hexadecimal values:
//   eg. 0xab
//
//  while 0xab can be decoded like:
//
//                                   8421 8421
//       0xa=10=8+2 0xb=11=8+2+1 --> 1010 1011
//
// or vice versa it is easy to get a hexadecimal number from a bit pattern,
//   eg.
//
//       8421 8421
//       0101 1101  -->  4+1=5=0x5 8+4+1=13=0xd --> 0x5d
//
void MFTriggerPattern::SetMaskDenied(const Byte_t mask, Prescale_t prescaled)
{
    prescaled==kPrescaled ? (fMaskDeniedPrescaled  = mask) : (fMaskDeniedUnprescaled  = mask);
}

// -------------------------------------------------------------------------
//
// Create the mask to allow a particular (un)prescaled trigger pattern.
//
// Possible arguments are (upper/lower case is ignored):
//
//           "LT1"  : Trigger Level 1 flag
//           "CAL"  : Calibration flag
//           "LT2"  : Trigger Level 2 flag
//           "PED"  : Pedestal flag
//           "PIND" : Pin Diode flag
//           "SUMT" : Sum Trigger flag
// 
// concatenations of these strings are allowed and considered as 
// a logic "and", while trigger pattern flags not considered are
// anyway allowed. To deny a particular trigger pattern use
// the method Deny
// Example: patt = "lt1 lt2" allows events with trigger pattern flags
// {LT1,CAL,LT2} but not events with flags {LT1,CAL}.
//
void MFTriggerPattern::Require(TString patt, Prescale_t prescaled)
{
    if (patt.Contains("LT1", TString::kIgnoreCase))
        RequireTriggerLvl1(prescaled);

    if (patt.Contains("LT2", TString::kIgnoreCase))
        RequireTriggerLvl2(prescaled);

    if (patt.Contains("CAL", TString::kIgnoreCase))
        RequireCalibration(prescaled);

    if (patt.Contains("PED", TString::kIgnoreCase))
        RequirePedestal(prescaled);

    if (patt.Contains("PIND", TString::kIgnoreCase))
        RequirePinDiode(prescaled);

    if (patt.Contains("SUMT", TString::kIgnoreCase))
        RequireSumTrigger(prescaled);
}

// -------------------------------------------------------------------------
//
// Create the mask to deny a particular (un)prescaled trigger pattern.
//
// This method is there because is not possible to deny trigger patterns
// using only the Require pattern. Possible arguments are (upper/lower
// case is ignored):
//
//           "LT1"  : Trigger Level 1 flag
//           "CAL"  : Calibration flag
//           "LT2"  : Trigger Level 2 flag
//           "PED"  : Pedestal flag
//           "PIND" : Pin Diode flag
//           "SUMT" : Sum Trigger flag
// 
// concatenations of these strings are allowed and considered as 
// a logic "and", while trigger pattern flags not considered are
// anyway allowed.
//
// Example: patt = "lt1 lt2" deny events with trigger pattern flags
// {LT1,CAL,LT2} but not events with flags {LT1,CAL}.
//
void MFTriggerPattern::Deny(TString patt, Prescale_t prescaled)
{
    if (patt.Contains("LT1", TString::kIgnoreCase))
        DenyTriggerLvl1(prescaled);

    if (patt.Contains("LT2", TString::kIgnoreCase))
        DenyTriggerLvl2(prescaled);

    if (patt.Contains("CAL", TString::kIgnoreCase))
        DenyCalibration(prescaled);

    if (patt.Contains("PED", TString::kIgnoreCase))
        DenyPedestal(prescaled);

    if (patt.Contains("PIND", TString::kIgnoreCase))
        DenyPinDiode(prescaled);

    if (patt.Contains("SUMT", TString::kIgnoreCase))
        DenySumTrigger(prescaled);
}

// --------------------------------------------------------------------------
//
Bool_t MFTriggerPattern::IsExpressionTrue() const
{
    return fResult;
}
