/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MRawFile                                                                //
//                                                                          //
//  This tasks reads the raw binary file like specified in the TDAS???      //
//  and writes the data in the corresponding containers which are           //
//  either retrieved from the parameter list or created and added.          //
//                                                                          //
//  Input Containers:                                                       //
//   -/-                                                                    //
//                                                                          //
//  Output Containers:                                                      //
//   MRawRunHeader, MRawEvtHeader, MRawEvtData, MRawCrateArray, MTime       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MRawFileRead.h"

#include <fstream.h>

#include "MLog.h"
#include "MTime.h"
#include "MParList.h"
#include "MRawRunHeader.h"
#include "MRawEvtHeader.h"
#include "MRawEvtData.h"
#include "MRawCrateData.h"
#include "MRawCrateArray.h"

ClassImp(MRawFileRead);


/*/  ----------- please don't delete and don't care about (Thomas) ------------
#define kBUFSZ 1024

class bifstream : public istream, public streambuf
{
private:
    char fBuffer[kBUFSZ]; //!
    FILE *fd;

    int sync()
    {
        memset(fBuffer, 0, kBUFSZ);
        return 0; 
    }
    int underflow()
    {
        int sz=fread(fBuffer, kBUFSZ, 1, fd);
        setg(fBuffer, fBuffer, fBuffer+kBUFSZ);

        return sz==kBUFSZ ? *(unsigned char*)fBuffer : EOF;//EOF;
    }
public:
    bifstream(const char *name) : istream(this)
    {
        fd = fopen(name, "rb");
        setbuf(fBuffer, kBUFSZ);
    }
};
*/

// --------------------------------------------------------------------------
//
// Default constructor. It tries to open the given file.
//
MRawFileRead::MRawFileRead(const char *fname, const char *name, const char *title)
{
    fName  = name  ? name  : "MRawFileRead";
    fTitle = title ? title : "Read task to read DAQ binary files";

    //
    // open the input stream
    //
    fFileName = fname;
    fIn = new ifstream(fname);
}

// --------------------------------------------------------------------------
//
// Destructor. Delete input stream.
//
MRawFileRead::~MRawFileRead()
{
    delete fIn;
}

// --------------------------------------------------------------------------
//
// The PreProcess of this task checks for the following containers in the
// list:
//   MRawRunHeader <output>   if not found it is created
//   MRawEvtHeader <output>   if not found it is created
//   MRawEvtData <output>     if not found it is created
//   MRawCrateArray <output>  if not found it is created
//   MRawEvtTime <output>     if not found it is created (MTime)
//
// If all containers are found or created the run header is read from the
// binary file and printed.  If the Magic-Number (file identification)
// doesn't match we stop the eventloop.
//
// Now the EvtHeader and EvtData containers are initialized.
//
Bool_t MRawFileRead::PreProcess(MParList *pList)
{
    //
    // first of all check if opening the file in the constructor was
    // successfull
    //
    if (!(*fIn))
    {
        *fLog << "Error: Cannot open file '" << fFileName << "'" << endl;
        return kFALSE;
    }

    //
    //  check if all necessary containers exist in the Parameter list.
    //  if not create one and add them to the list
    //
    fRawRunHeader = (MRawRunHeader*)pList->FindCreateObj("MRawRunHeader");
    if (!fRawRunHeader)
        return kFALSE;

    fRawEvtHeader = (MRawEvtHeader*)pList->FindCreateObj("MRawEvtHeader");
    if (!fRawEvtHeader)
        return kFALSE;

    fRawEvtData = (MRawEvtData*)pList->FindCreateObj("MRawEvtData");
    if (!fRawEvtData)
        return kFALSE;

    fRawCrateArray = (MRawCrateArray*)pList->FindCreateObj("MRawCrateArray");
    if (!fRawCrateArray)
        return kFALSE;

    fRawEvtTime = (MTime*)pList->FindCreateObj("MTime", "MRawEvtTime");
    if (!fRawEvtTime)
        return kTRUE;

    //
    // Read RUN HEADER (see specification) from input stream
    //
    fRawRunHeader->ReadEvt(*fIn);
    fRawRunHeader->Print();

    if (fRawRunHeader->GetMagicNumber()!=kMagicNumber)
        return kFALSE;

    //
    // Give the run header information to the 'sub-classes'
    //
    fRawEvtHeader->Init(fRawRunHeader, fRawEvtTime);
    fRawEvtData  ->Init(fRawRunHeader);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// The Process reads one event from the binary file:
//  - The event header is read
//  - the run header is read
//  - all crate information is read
//  - the raw data information of one event is read
//
Bool_t MRawFileRead::Process()
{
    //
    //  Read in the next EVENT HEADER (see specification),
    // if there is no next event anymore stop eventloop
    //
    if (!fRawEvtHeader->ReadEvt(*fIn))
        return kFALSE;

    //
    // Delete arrays which stores the pixel information (time slices)
    //
    fRawEvtData->DeletePixels();

    //
    // clear the TClonesArray which stores the Crate Information
    //
    fRawCrateArray->Clear();

    //
    //  Get number of crates from the run header
    //
    const UShort_t nc = fRawRunHeader->GetNumCrates();

    //
    // read the CRATE DATA (see specification) from file
    //
    for (int i=0; i<nc; i++)
    {
        fRawCrateArray->GetEntry(i)->ReadEvt(*fIn);

        fRawEvtData->ReadEvt(*fIn);
    }

    return kTRUE;
}

