/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  07/2001 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MFillH                                                                  //
//                                                                          //
//  This is a common interface (task) to fill mars histograms. Every mars   //
//  histogram which is derived from MH can be filled with this task.        //
//                                                                          //
//  You must specifiy the parameter container with which data the histogram //
//  container should be filled, and the histogram container which has       //
//  to be filled. This can be done by either specifing the name of the      //
//  objects in the parameter list or by specifiing a pointer to the object. //
//  (s. Constructor)                                                        //
//                                                                          //
//  WARNING:                                                                //
//   Because MFillH is a generalized task to fill histograms it doesn't     //
//   know about which branches from a file are necessary to fill the        //
//   histograms. If you are reading data from a file which is directly      //
//   filled into a histogram via MFillH, please call either                 //
//   MReadTree::DisableAutoScheme() or enable the necessary branches by     //
//   yourself, using MReadTree::EnableBranch()                              //
//                                                                          //
//  Input Containers:                                                       //
//   A parameter container                                                  //
//                                                                          //
//  Output Containers:                                                      //
//   A histogram container                                                  //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
#include "MFillH.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MH.h"
#include "MParList.h"

ClassImp(MFillH);

// --------------------------------------------------------------------------
//
// Initializes name and title of the object. It is called by all
// constructors.
//
void MFillH::Init(const char *name, const char *title)
{
    fName  = name  ? name  : "MFillH";
    fTitle = title ? title : "Task to fill Mars histograms";

    fH            = NULL;
    fParContainer = NULL;
}

// --------------------------------------------------------------------------
//
// Constructor.
//
// - par is the name of the parameter container which should be filled into
//   the histogram
// - hist is the name of the histogram container (which must have been
//   derived from MH)
//
MFillH::MFillH(const char *par, const char *hist, const char *name, const char *title)
{
    Init(name, title);

    fParContainerName = par;
    fHName = hist;
}

// --------------------------------------------------------------------------
//
// Constructor.
//
// - par is a pointer to the instance of your parameter container from which
//   the data should be used to fill the histogram.
// - hist is the name of the histogram container (which must have been
//   derived from MH)
//
MFillH::MFillH(const MParContainer *par, const char *hist, const char *name, const char *title)
{
    Init(name, title);

    fParContainer = par;
    fHName = hist;
}

// --------------------------------------------------------------------------
//
// Constructor.
//
// - par is a pointer to the instance of your parameter container from which
//   the data should be used to fill the histogram.
// - hist is a pointer to the instance of your histogram container (which must
//   have been derived from MH) into which the data should flow
//
MFillH::MFillH(const char *par, MH *hist, const char *name, const char *title)
{
    Init(name, title);

    fParContainerName = par;
    fH = hist;
}

// --------------------------------------------------------------------------
//
// Constructor.
//
// - par is a pointer to the instance of your parameter container from which
//   the data should be used to fill the histogram.
// - hist is the name of the histogram container (which must have been
//   derived from MH)
//
MFillH::MFillH(const MParContainer *par, MH *hist, const char *name, const char *title)
{
    Init(name, title);

    fParContainer = par;
    fH = hist;
}

// --------------------------------------------------------------------------
//
// Checks the parameter list for the existance of the parameter container. If
// the name of it was given in the constructor. It checks also for the
// existance of the histogram container in the parameter list if a name was
// given. If it is not available it tried to create a histogram container
// with the same type as the given object name.
//
Bool_t MFillH::PreProcess(MParList *pList)
{
    if (!fParContainer)
    {
        fParContainer = (MParContainer*)pList->FindObject(fParContainerName);
        if (!fParContainer)
        {
            *fLog << dbginf << fParContainerName << " [MParContainer] not found... aborting." << endl;
            return kFALSE;
        }
    }

    if (!fH)
    {
        fH = (MH*)pList->FindCreateObj(fHName);
        if (!fH)
            return kFALSE;
    }

    if (!fH->InheritsFrom("MH"))
    {
        *fLog << dbginf << fH->GetName() << " [" << fH->ClassName();
        *fLog << "] doesn't inherit from MH - cannot be used for MFillH... aborting." << endl;
        return kFALSE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fills the data from the parameter conatiner into the histogram container
//
Bool_t MFillH::Process()
{
    fH->Fill(fParContainer);

    return kTRUE;
} 

// --------------------------------------------------------------------------
//
// Set the ReadyToSave flag of the histogram container, because now all data
// has been filled into the histogram.
//
Bool_t MFillH::PostProcess()
{
    fH->SetReadyToSave();
    return kTRUE;
}
