/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Oscar Blanch  11/2001 < mailto:blanch@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MMcPedestalNSB                                                        //
//                                                                         //
//  Input Containers:                                                      //
//   MMcFadcHeader                                                         //
//   MMcRunHeader                                                          //
//   MRawRunHeader                                                         //
//                                                                         //
//  Output Containers:                                                     //
//   MPedestalCam                                                          //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MMcPedestalNSB.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MPedestalCam.h"
#include "MRawRunHeader.h"
#include "MMcRunHeader.hxx"
#include "MMcFadcHeader.hxx"

ClassImp(MMcPedestalNSB);

MMcPedestalNSB::MMcPedestalNSB(const float difnsb,
			       const char *name, const char *title)
{
    fName  = name  ? name  : "MMcPedestalNSB";
    fTitle = title ? title : "Task to copy monte carlo pedestals into MPedestal Container";

    AddToBranchList("fPedestalMean");
    AddToBranchList("fElecNoise");

    fdnsb_pixel=difnsb;
}

Bool_t MMcPedestalNSB::PreProcess(MParList *pList)
{
    MRawRunHeader *run = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (run)
    {
	if (run->GetRunType() != kRTMonteCarlo){
	    *fLog << dbginf << "This task is only for Monte Carlo files, therefore the RunType should be "<<kRTMonteCarlo<<" and it is: "<<run->GetRunType()<<" ... aborting"<<endl;
	    return kFALSE;
	}
    }
    else
    {
        *fLog << dbginf << "MRawRunHeader not found... aborting." << endl;
	return kFALSE;
    }

    const MMcFadcHeader *mcped = (MMcFadcHeader*)pList->FindObject("MMcFadcHeader");

    if (!mcped)
    {
        *fLog << dbginf << "MMcFadcHeader not found... aborting." << endl;
        return kFALSE;
    }

    MMcRunHeader *mcrun = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (mcrun)
    {
	if (fdnsb_pixel >= 0 && fdnsb_pixel != mcrun->GetNumPheFromDNSB()){
	    *fLog<< dbginf <<endl<< "The MC file has been generated with diffuse nsb : "<<mcrun->GetNumPheFromDNSB()<<" but you set up the diffuse NSB to :"<<fdnsb_pixel<<endl;
	fdnsb_pixel = fdnsb_pixel*50.0/15.0;
	}
	else
	    fdnsb_pixel = mcrun->GetNumPheFromDNSB()*50.0/15.0;
    }
    else
    {
	if (fdnsb_pixel < 0 ){
	    *fLog << dbginf << "MMcRunHeader not found... aborting." << endl;
	    return kFALSE;
	}
	else {
	    *fLog<< dbginf <<endl<< "The MC file has been generated with diffuse nsb : "<<mcrun->GetNumPheFromDNSB()<<" but you set up the diffuse NSB to :"<<fdnsb_pixel<<endl;
	    fdnsb_pixel = fdnsb_pixel*50.0/15.0;
	}
    }

    MPedestalCam *pedcam = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!pedcam)
        return kFALSE;

    const int num = mcped->GetNumPixel();

    pedcam->InitSize(num);

    for (int i=0; i<num; i++)
    {
        MPedestalPix &pix = (*pedcam)[i];

        const Float_t pedrms = mcped->GetPedestalRms(i);

        const Float_t sigrms = pedrms/sqrt(run->GetNumSamplesHiGain()*2);

	if (i<397) // Central Pixels 
	{
	    pix.SetPedestalRms(sqrt(pedrms*pedrms+fdnsb_pixel*mcped->GetAmplitud()*mcped->GetAmplitud()), sigrms);
	}
	else
	{
	    pix.SetPedestalRms(sqrt(pedrms*pedrms+fdnsb_pixel*mcped->GetAmplitud()*mcped->GetAmplitud()*4), sigrms);
	}

    }

    return kTRUE;
}



