/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Oscar Blanch  11/2001 < mailto:blanch@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MMcPedestalNSBAdd                                                      //
//                                                                         //
//  This Task adds the contribution of the diffuse NSB to the FADC         //
//  pedestals. We assume that NSB introduces larger fluctuation but does   //
//  not change the mean value.                                             //
//  To be precise we add quadratically to the rms that is already in       //
//  MPedestalCam the NSB contribution.                                     //
//  The number of photons from the diffuse NSB follows a poisson           // 
//  distribution with expected mean value X, then its standard desviation  //
//  is sqrt(X).                                                            //
//  X is the number of phe per ns so we have to convert in FADC counts per // 
//  slice:                                                                 //
//                                                                         //
//  Y=sqrt(X*FADC_time/Number_of_slice*pixel_size)*Amp_single_phe_response //
//                                                                         //
//  Input Containers:                                                      //
//   MMcFadcHeader                                                         //
//   MMcRunHeader                                                          //
//   MRawRunHeader                                                         //
//                                                                         //
//  Output Containers:                                                     //
//   MPedestalCam                                                          //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MMcPedestalNSBAdd.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MPedestalCam.h"
#include "MRawRunHeader.h"
#include "MMcRunHeader.hxx"
#include "MMcFadcHeader.hxx"
#include "MGeomCam.h"
#include "MGeomPix.h"

ClassImp(MMcPedestalNSBAdd);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MMcPedestalNSBAdd::MMcPedestalNSBAdd(const Float_t difnsb,
                                     const char *name, const char *title)
    : fDnsbPixel(difnsb)
{
    fName  = name  ? name  : "MMcPedestalNSBAdd";
    fTitle = title ? title : "Task to copy monte carlo pedestals into MPedestal Container";

    //
    // This is not needed here using MReadMarsFile because for the
    // RunHeader tree the auto scheme is disabled by default
    //
    AddToBranchList("fPedesMean");
    AddToBranchList("fElecNoise");
}

Bool_t MMcPedestalNSBAdd::CheckRunType(MParList *pList) const
{
    MRawRunHeader *runheader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!runheader)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    return runheader->GetRunType() == kRTMonteCarlo;
}

// --------------------------------------------------------------------------
//
// The PreProcess does nothing since the process does not exist and
// therefore it does not need any pointer or files already initialysed

Bool_t MMcPedestalNSBAdd::PreProcess(MParList *pList)
{

   // FIX ME , ReInit should be called automatically. When it is 
   // implemented then this line should be removed.

    if (!CheckRunType(pList))
    {
        *fLog << warn << dbginf << "Warning - MMcPedestalNSB is for Monte Carlo files only... removing this task from list." << endl;
        return kSKIP;
    }

    fFadc = (MMcFadcHeader*)pList->FindObject("MMcFadcHeader");
    if (!fFadc)
    {
        *fLog << err << dbginf << "MMcFadcHeader not found... aborting." << endl;
	return kFALSE;
    }

    fGeom = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!fGeom)
    {
        *fLog << err << dbginf << "MGeomCam not found... aborting." << endl;
	return kFALSE;
    }

    fPedCam = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!fPedCam)
	return kFALSE;

    MMcRunHeader *mcrunheader = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!mcrunheader && fDnsbPixel < 0 )
    {
        *fLog << err << dbginf << "Using the default argument of MMcPedestalNSB::MMcPedestalNSB() ";
        *fLog << "only allowed if MMcRunHeader is available... aborting." << endl;
        return kFALSE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
//  This function is called each time MReadTree::Notify is called, which 
//  happens when it  changes the file to read from.
//  Here we add the contribution from NSB to the pedestals.
//  The ReInit searches for the following input containers:
//   - MRawRunHeader
//   - MMcRunHeader
//   - MMcFacdHeader
//   - MGeomCam
//
//   The following output containers are also searched and created if
//   they were not found:
//   - MPedestalCam
//
Float_t MMcPedestalNSBAdd::GetDnsb(MParList *pList) const
{
    MMcRunHeader *mcrunheader = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!mcrunheader && fDnsbPixel < 0 )
    {
        *fLog << err << dbginf << "Using the default argument of MMcPedestalNSB::MMcPedestalNSB() ";
        *fLog << "only allowed if MMcRunHeader is available... aborting." << endl;
        return -1;
    }

    if (!mcrunheader)
        return fDnsbPixel;

    if (fDnsbPixel >= 0 && fDnsbPixel != mcrunheader->GetNumPheFromDNSB())
    {
        *fLog << warn << dbginf << "The MC file has been generated with diffuse nsb " << mcrunheader->GetNumPheFromDNSB();
        *fLog <<" but you set up the diffuse NSB to " << fDnsbPixel << endl;

        return fDnsbPixel;
    }

    return mcrunheader->GetNumPheFromDNSB();
}

Bool_t MMcPedestalNSBAdd::ReInit(MParList *pList)
{
    if (!CheckRunType(pList))
        return kFALSE;

    Float_t dnsbpix = GetDnsb(pList) * 50.0/15.0;

    if (dnsbpix < 0)
        return kFALSE;

    const int num = fFadc->GetNumPixel();
    
    fPedCam->InitSize(num);

    const Float_t size0 = (*fGeom)[0].GetR() * (*fGeom)[0].GetR();

    for (int i=0; i<num; i++)
    {
        MPedestalPix &pix     = (*fPedCam)[i];
	MGeomPix     &pixgeom = (*fGeom)[i];

        const Float_t pedrms  = pix.GetSigma();
        const Float_t size    = pixgeom.GetR()*pixgeom.GetR()/size0;

        const Float_t ampl    = fFadc->GetAmplitud();

	pix.SetSigma(sqrt(pedrms*pedrms + dnsbpix*ampl*ampl*size));
    }

    return kTRUE;
}

