/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    1/2002 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Wolfgang Wittek 1/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHTimeDiffTime                                                          //
//                                                                          //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHTimeDiffTime.h"

#include <TCanvas.h>

#include "MTime.h"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHTimeDiffTime);


// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title only. Typically you won't
// need to change this.
//
MHTimeDiffTime::MHTimeDiffTime(const char *name, const char *title)
    : fLastTime(0), fHist()
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHTimeDiffTime";
    fTitle = title ? title : "2-D histogram in time and time difference";

    fHist.SetDirectory(NULL);

    fHist.GetXaxis()->SetTitle("\\Delta t [s]");
    fHist.GetYaxis()->SetTitle("t [s]");
}

Bool_t MHTimeDiffTime::SetupFill(const MParList *plist)
{
   fTime = (MTime*)plist->FindObject("MTime");
   if (!fTime)
   {
       *fLog << err << dbginf << "MTime not found... aborting." << endl;
       return kFALSE;
   }

   const MBinning* binsdtime = (MBinning*)plist->FindObject("BinningTimeDiff");
   const MBinning* binstime  = (MBinning*)plist->FindObject("BinningTime");
   if (!binstime || !binsdtime)
   {
       *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
       return kFALSE;      
   }

   SetBinning(&fHist, binsdtime, binstime);

   return kTRUE;
}

TObject *MHTimeDiffTime::DrawClone(Option_t *opt) const
{
    TCanvas *c = MakeDefCanvas("DiffTimeTime", "Distrib of \\Delta t, t");
    c->Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    //
    // FIXME: ProjectionX,Y is not const within root
    //
    TH1 *h;

    c->cd(1);
    h = ((TH2*)&fHist)->ProjectionX("ProX", -1, 9999, "E");
    h->Draw(opt);
    h->SetBit(kCanDelete);
    gPad->SetLogy();

    c->cd(2);
    h = ((TH2*)&fHist)->ProjectionY("ProY", -1, 9999, "E");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    c->cd(3);
    ((TH2*)&fHist)->DrawCopy(opt);

    c->Modified();
    c->Update();

    return c;
}

void MHTimeDiffTime::Draw(Option_t *opt)
{
    if (!gPad)
        MakeDefCanvas("DiffTimeTime", "Distrib of \\Delta t, t");

    gPad->Divide(2,2);

    TH1 *h;

    gPad->cd(1);
    h = fHist.ProjectionX("ProX", -1, 9999, "E");
    h->Draw(opt);
    h->SetBit(kCanDelete);
    gPad->SetLogy();

    gPad->cd(2);
    h = fHist.ProjectionY("ProY", -1, 9999, "E");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    gPad->cd(3);
    fHist.DrawCopy(opt);

    gPad->Modified();
    gPad->Update();
}

Bool_t MHTimeDiffTime::Fill(const MParContainer *par)
{
    const Int_t time = fTime->GetTimeLo();

    fHist.Fill(0.0001*(time-fLastTime), 0.0001*time);

    fLastTime = time;

    return kTRUE;
}

