/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek 3/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHThetabarTime                                                          //
//                                                                          //
//  calculates the average Theta for different bins in time                 //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHThetabarTime.h"

#include <TCanvas.h>

#include "MTime.h"
#include "MMcEvt.hxx"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHThetabarTime);


// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram.
//
MHThetabarTime::MHThetabarTime(const char *name, const char *title)
    : fHist()
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHThetabarTime";
    fTitle = title ? title : "1-D profile histogram Thetabar vs. time";

    fHist.SetDirectory(NULL);

    fHist.SetXTitle("time [s]");
    fHist.SetYTitle("\\overline{\\Theta} [\\circ]");
}

// --------------------------------------------------------------------------
//
// Set the binnings and prepare the filling of the histogram
//
Bool_t MHThetabarTime::SetupFill(const MParList *plist)
{
    fTime = (MTime*)plist->FindObject("MTime");
    if (!fTime)
    {
        *fLog << err << dbginf << "MTime not found... aborting." << endl;
        return kFALSE;
    }

    fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
    if (!fMcEvt)
    {
       *fLog << err << dbginf << "MMcEvt not found... aborting." << endl;
       return kFALSE;
    }

    const MBinning* binstime  = (MBinning*)plist->FindObject("BinningTime");
    if (!binstime )
    {
        *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
        return kFALSE;
    }

    SetBinning(&fHist, binstime);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw a copy of the histogram
//
TObject *MHThetabarTime::DrawClone(Option_t *opt) const
{
    TCanvas &c = *MakeDefCanvas("ThetabarTime", "Thetabar vs. time");

    gROOT->SetSelectedPad(NULL);

    ((TProfile*)&fHist)->DrawCopy(opt);

    c.Modified();
    c.Update();

    return &c;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
//
void MHThetabarTime::Draw(Option_t *opt)
{
    if (!gPad)
        MakeDefCanvas("ThetabarTime", "Thetabar vs. time");

    fHist.DrawCopy(opt);

    gPad->Modified();
    gPad->Update();
}

// --------------------------------------------------------------------------
//
// Fill the histogram
//
Bool_t MHThetabarTime::Fill(const MParContainer *par)
{
    const Int_t time = fTime->GetTimeLo();

    fHist.Fill(0.0001*time, fMcEvt->GetTheta()*kRad2Deg);

    return kTRUE;
}
