/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  2001 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHHillasExt
//
// This class contains histograms for every Hillas parameter
//
/////////////////////////////////////////////////////////////////////////////

#include "MHHillasExt.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"
#include "MParList.h"
#include "MHillasExt.h"

ClassImp(MHHillasExt);

// --------------------------------------------------------------------------
//
// Setup four histograms for Width, Length
//
MHHillasExt::MHHillasExt(const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHHillasExt";
    fTitle = title ? title : "Container for Hillas (ext) histograms";

    //
    // loop over all Pixels and create two histograms
    // one for the Low and one for the High gain
    // connect all the histogram with the container fHist
    //
    fHConc.SetDirectory(NULL);
    fHConc1.SetDirectory(NULL);
    fHAsym.SetDirectory(NULL);
    fHM3Long.SetDirectory(NULL);
    fHM3Trans.SetDirectory(NULL);

    fHConc.SetTitle("Ratio: Conc");
    fHConc1.SetTitle("Ratio: Conc1");
    fHAsym.SetTitle("Asymmetry");
    fHM3Long.SetTitle("3^{rd} Moment Longitudinal");
    fHM3Trans.SetTitle("3^{rd} Moment Transverse");

    fHConc.SetXTitle("Ratio");
    fHConc1.SetXTitle("Ratio");
    fHAsym.SetXTitle("Asym [mm]");
    fHM3Long.SetXTitle("3^{rd} M_{l} [mm]");
    fHM3Trans.SetXTitle("3^{rd} M_{t} [mm]");

    fHConc.SetYTitle("Counts");
    fHConc1.SetYTitle("Counts");
    fHAsym.SetYTitle("Counts");
    fHM3Long.SetYTitle("Counts");
    fHM3Trans.SetYTitle("Counts");
}

// --------------------------------------------------------------------------
//
// Delete the four histograms
//
MHHillasExt::~MHHillasExt()
{
}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning (with the names 'BinningWidth' and 'BinningLength')
// are found in the parameter list
// Use this function if you want to set the conversion factor which
// is used to convert the mm-scale in the camera plain into the deg-scale
// used for histogram presentations. The conversion factor is part of
// the camera geometry. Please create a corresponding MGeomCam container.
//
Bool_t MHHillasExt::SetupFill(const MParList *plist)
{
    const MBinning* binsc  = (MBinning*)plist->FindObject("BinningConc");
    const MBinning* binsc1 = (MBinning*)plist->FindObject("BinningConc1");
    const MBinning* binsa  = (MBinning*)plist->FindObject("BinningAsym");
    const MBinning* binsl  = (MBinning*)plist->FindObject("BinningM3Long");
    const MBinning* binst  = (MBinning*)plist->FindObject("BinningM3Trans");
    if (!binsc || !binsc1 || !binsa || !binsl || !binst)
    {
        *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
        return kFALSE;
    }

    SetBinning(&fHConc,    binsc);
    SetBinning(&fHConc1,   binsc1);
    SetBinning(&fHAsym,    binsa);
    SetBinning(&fHM3Long,  binsl);
    SetBinning(&fHM3Trans, binst);

    const MGeomCam *geom = (MGeomCam*)plist->FindObject("MGeomCam");
    if (!geom)
    {
        *fLog << warn << dbginf << "No Camera Geometry available. Using mm-scale for histograms." << endl;
        return kTRUE;
    }

    fHAsym.SetXTitle("Asym [\\circ]");
    fHM3Long.SetXTitle("3^{rd} M_{l} [\\circ]");
    fHM3Trans.SetXTitle("3^{rd} M_{t} [\\circ]");

    return MHHillas::SetupFill(plist);
}

// --------------------------------------------------------------------------
//
// Fill the four histograms with data from a MHillas-Container.
// Be careful: Only call this with an object of type MHillas
//
Bool_t MHHillasExt::Fill(const MParContainer *par)
{
    const MHillasExt &h = *(MHillasExt*)par;

    fHConc.Fill(h.GetConc());
    fHConc1.Fill(h.GetConc1());

    if (fUseMmScale)
    {
        fHAsym.Fill(h.GetAsym());
        fHM3Long.Fill(h.GetM3Long());
        fHM3Trans.Fill(h.GetM3Trans());
    }
    else
    {
        fHAsym.Fill(fMm2Deg*h.GetAsym());
        fHM3Long.Fill(fMm2Deg*h.GetM3Long());
        fHM3Trans.Fill(fMm2Deg*h.GetM3Trans());
    }

    return MHHillas::Fill(par);
}

// --------------------------------------------------------------------------
//
// With this function you can convert the histogram ('on the fly') between
// degrees and millimeters.
//
void MHHillasExt::SetMmScale(Bool_t mmscale)
{
    if (fUseMmScale == mmscale)
        return;

    if (fMm2Deg<0)
    {
        *fLog << warn << dbginf << "Warning - Sorry, no conversion factor for conversion available." << endl;
        return;
    }

    if (fUseMmScale)
    {
        fHAsym.SetXTitle("Asym [mm]");
        fHM3Long.SetXTitle("3^{rd} M_{l}[mm]");
        fHM3Trans.SetXTitle("3^{rd} M_{t} [mm]");

        fHAsym.Scale(1./fMm2Deg);
        fHM3Long.Scale(1./fMm2Deg);
        fHM3Trans.Scale(1./fMm2Deg);
    }
    else
    {
        fHAsym.SetXTitle("Asym [\\circ]");
        fHM3Long.SetXTitle("3^{rd} M_{l} [\\circ]");
        fHM3Trans.SetXTitle("3^{rd} M_{t} [\\circ]");

        fHAsym.Scale(fMm2Deg);
        fHM3Long.Scale(fMm2Deg);
        fHM3Trans.Scale(fMm2Deg);
    }

    MHHillas::SetMmScale(mmscale);
}

// --------------------------------------------------------------------------
//
// Draw clones of all four histograms. So that the object can be deleted
// and the histograms are still visible in the canvas.
// The cloned object are deleted together with the canvas if the canvas is
// destroyed. If you want to handle dostroying the canvas you can get a
// pointer to it from this function
//
TObject *MHHillasExt::DrawClone(Option_t *opt) const
{
    TCanvas &c = *MakeDefCanvas("Hillas", "Histograms of Hillas Parameters",
                               3*350, 2*250);
    c.Divide(3, 2);

    gROOT->SetSelectedPad(NULL);

    //
    // This is necessary to get the expected bahviour of DrawClone
    //
    c.cd(1);
    ((TH1F&)fHConc).DrawCopy();

    c.cd(4);
    ((TH1F&)fHConc1).DrawCopy();

    c.cd(2);
    ((TH1F&)fHAsym).DrawCopy();

    c.cd(3);
    ((TH1F&)fHM3Long).DrawCopy();

    c.cd(6);
    ((TH1F&)fHM3Trans).DrawCopy();

    c.Modified();
    c.Update();

    MHHillas::DrawClone();

    return &c;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the four histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHHillasExt::Draw(Option_t *)
{
    if (!gPad)
        MakeDefCanvas("Hillas", "Histograms of Hillas Parameters", 350, 3*250);

    gPad->Divide(3, 2);

    gPad->cd(1);
    fHConc.DrawCopy();

    gPad->cd(4);
    fHConc1.DrawCopy();

    gPad->cd(2);
    fHAsym.DrawCopy();

    gPad->cd(3);
    fHM3Long.DrawCopy();

    gPad->cd(6);
    fHM3Trans.DrawCopy();

    gPad->Modified();
    gPad->Update();

    MHHillas::DrawClone();
}
