/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 5/2002 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Rudy Bock, 5/2002 <mailto:rkb@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

// -------------------------------------------------------------------------
//
//  plot.C
//
//  This macro shows how to fill and display a histogram using Mars
//
void plot()
{
    //
    // Create a empty Parameter List and an empty Task List
    // The tasklist is identified in the eventloop by its name
    //
    MParList  plist;

    MTaskList tlist;
    plist.AddToList(&tlist);

    //
    // Now setup the tasks and tasklist:
    // ---------------------------------
    //

    // First Task: Read file with image parameters
    // (created with the star.C macro)
    MReadMarsFile  read("Events", "star.root");
    read.DisableAutoScheme();
    tlist.AddToList(&read);

    // Create a filter for Gammas
    MFParticleId fgamma("MMcEvt", '=', kGAMMA);
    tlist.AddToList(&fgamma);

    // Create a filter for Non-Gammas
    MFParticleId fhadrons("MMcEvt", '!', kGAMMA);
    tlist.AddToList(&fhadrons);

    // -------------------------------------------------------
    //
    // set the name of the variable to plot and the binning
    //
    TString var("MHillas.fSize");

    MBinning bins("BinningMH3X");
    bins.SetEdgesLog(50, 100, 20000);
    plist.AddToList(&bins);
    //
    // -------------------------------------------------------

    // Create a histogram for the data from gammas and from non-gammas
    MH3 h3g(var);
    MH3 h3h(var);

    // Add the histograms to the parameter container list
    plist.AddToList(&h3g);
    plist.AddToList(&h3h);

    // Create a task which fills one histogram with the gamma-data
    MFillH fillg(&h3g);
    fillg.SetFilter(&fgamma);
    tlist.AddToList(&fillg);

    // Create a task which fills the other histogram with the non-gamma-data
    MFillH fillh(&h3h);
    fillh.SetFilter(&fhadrons);
    tlist.AddToList(&fillh);

    //
    // Create and setup the eventloop
    //
    MEvtLoop evtloop;
    evtloop.SetParList(&plist);

    //
    // Execute your analysis
    //
    MProgressBar bar;
    evtloop.SetProgressBar(&bar);
    if (!evtloop.Eventloop())
        return;

    tlist.PrintStatistics();

    // Create a default canvas called Plot and set the
    // x-axis to logarithmic scale
    MH::MakeDefCanvas("Plot");
    gPad->SetLogx();

    // Setup some style options of the two histograms
    // and draw a copy of both
    h3h.GetHist().SetLineColor(kRed);
    h3h.GetHist().SetFillStyle(4000);
    h3g.GetHist().DrawCopy();
    h3h.GetHist().DrawCopy("same");

    // Now create a new histogram, fill it with the division of the
    // two histograms and draw also a copy of it
    TH1D h;
    MH::SetBinning(&h, &bins);
    h.Divide(&h3g.GetHist(), &h3h.GetHist());
    h.SetLineColor(kGreen);
    h.SetFillStyle(4000);
    h.DrawCopy("same");
}
