/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001
!   Author(s): Rudolf Bock     10/2001 <mailto:Rudolf.Bock@cern.ch>
!   Author(s): Wolfgang Wittek  6/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHillas
//
// Storage Container for image parameters
//
//    basic image parameters
//
// Version 1:
// ----------
// fLength   [mm]       major axis of ellipse
// fWidth    [mm]       minor axis
// fDelta    [rad]      angle of major axis with x-axis
//                      by definition the major axis is pointing into
//                      the hemisphere x>0, thus -pi/2 < delta < pi/2
// fSize     [#CerPhot] total sum of pixels
// fMeanX    [mm]       x of center of ellipse
// fMeanY    [mm]       y of center of ellipse
//
// Version 2:
// ----------
// fNumCorePixels  number of pixels called core
// fNumUsedPixels  number of pixels which survived the cleaning
//
// Version 3:
// ----------
// fNumCorePixels  moved to MNewImagePar
// fNumUsedPixels  moved to MNewImagePar
// fCosDelta       added
// fSinDelte       added
//
/////////////////////////////////////////////////////////////////////////////
#include "MHillas.h"

#include <fstream.h>

#include <TArrayF.h>
#include <TEllipse.h>

#include "MGeomPix.h"
#include "MGeomCam.h"

#include "MCerPhotPix.h"
#include "MCerPhotEvt.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHillas);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MHillas::MHillas(const char *name, const char *title) : fEllipse(NULL)
{
    fName  = name  ? name  : "MHillas";
    fTitle = title ? title : "Storage container for image parameters of one event";

    Reset();

    fEllipse = new TEllipse;
}

// --------------------------------------------------------------------------
//
// Destructor. Deletes the TEllipse if one exists.
//
MHillas::~MHillas()
{
    Clear();
}

// --------------------------------------------------------------------------
//
// Initializes the values with defaults. For the default values see the
// source code.
//
void MHillas::Reset()
{
    fLength = -1;
    fWidth  = -1;
    fDelta  =  0;

    fSize   = -1;
    fMeanX  = -1;
    fMeanY  = -1;

    Clear();
}

// --------------------------------------------------------------------------
//
// Print the hillas Parameters to *fLog
//
void MHillas::Print(Option_t *) const
{
    Double_t atg = atan2(fMeanY, fMeanX)*kRad2Deg;

    if (atg<0)
        atg += 180;

    *fLog << all;
    *fLog << "Basic Image Parameters (" << GetName() << ")" << endl;
    *fLog << " - Length      [mm]  = " << fLength << endl;
    *fLog << " - Width       [mm]  = " << fWidth  << endl;
    *fLog << " - Delta       [deg] = " << fDelta*kRad2Deg << endl;
    *fLog << " - Size        [1]   = " << fSize   << " #CherPhot"   << endl;
    *fLog << " - Meanx       [mm]  = " << fMeanX  << endl;
    *fLog << " - Meany       [mm]  = " << fMeanY  << endl;
    *fLog << " - atg(y/x)    [deg] = " << atg     << endl;
}

// --------------------------------------------------------------------------
//
// Instead of adding MHillas itself to the Pad (s. AppendPad in TObject)
// we create an ellipse, which is added to the Pad by its Draw function
// You can remove it by deleting the Ellipse Object (s. Clear())
//
void MHillas::Draw(Option_t *opt)
{
    Clear();

    if (fLength<0 || fWidth<0)
        return;

    fEllipse = new TEllipse(fMeanX, fMeanY, fLength, fWidth,
                            0, 360, fDelta*kRad2Deg+180);

    fEllipse->SetLineWidth(2);
    fEllipse->Draw();
}

// --------------------------------------------------------------------------
//
// If a TEllipse object exists it is deleted
//
void MHillas::Clear(Option_t *)
{
    if (!fEllipse)
        return;

    delete fEllipse;

    fEllipse = NULL;
}


// --------------------------------------------------------------------------
//
// Calculate the image parameters from a Cherenkov photon event
// assuming Cher.photons/pixel and pixel coordinates are given
// In case you don't call Calc from within an eventloop make sure, that
// you call the Reset member function before.
// Returns:
//   0  no error
//   1  number of pixels < 3
//   2  size==0
//   3  number of used pixel < 3
//   4  CorrXY == 0
//
Int_t MHillas::Calc(const MGeomCam &geom, const MCerPhotEvt &evt)
{
    const UInt_t npixevt = evt.GetNumPixels();

    //
    // sanity check
    //
    if (npixevt < 3)
        return 1;

    //
    // calculate mean value of pixel coordinates and fSize
    // -----------------------------------------------------
    //
    // Because this are only simple sums of roughly 600 values
    // with an accuracy less than three digits there should not
    // be any need to use double precision (Double_t) for the
    // calculation of fMeanX, fMeanY and fSize.
    //
    fMeanX = 0;
    fMeanY = 0;
    fSize  = 0;

    Int_t numused = 0;

    for (UInt_t i=0; i<npixevt; i++)
    {
        const MCerPhotPix &pix = evt[i];

        if (!pix.IsPixelUsed())
            continue;

        const MGeomPix &gpix = geom[pix.GetPixId()];

        const Float_t nphot = pix.GetNumPhotons();

        fSize  += nphot;		             // [counter]
        fMeanX += nphot * gpix.GetX();               // [mm]
        fMeanY += nphot * gpix.GetY();               // [mm]

        numused++;
    }

    //
    // sanity checks
    //
    if (fSize==0)
        return 2;

    fMeanX /= fSize;                                 // [mm]
    fMeanY /= fSize;                                 // [mm]

    if (numused<3)
        return 3;

    //
    // calculate 2nd moments
    // ---------------------
    //
    // To make sure that the more complicated sum is evaluated as
    // accurate as possible (because it is needed for more
    // complicated calculations (see below) we calculate it using
    // double prcision.
    //
    Double_t corrxx=0;                               // [m^2]
    Double_t corrxy=0;                               // [m^2]
    Double_t corryy=0;                               // [m^2]

    for (UInt_t i=0; i<npixevt; i++)
    {
        const MCerPhotPix &pix = evt[i];

        if (!pix.IsPixelUsed())
            continue;

        const MGeomPix &gpix = geom[pix.GetPixId()];

        const Float_t dx = gpix.GetX() - fMeanX;     // [mm]
        const Float_t dy = gpix.GetY() - fMeanY;     // [mm]

        const Float_t nphot = pix.GetNumPhotons();   // [#phot]

        corrxx += nphot * dx*dx;                     // [mm^2]
        corrxy += nphot * dx*dy;                     // [mm^2]
        corryy += nphot * dy*dy;                     // [mm^2]
    }

    //
    // If corrxy=0 (which should never happen, because fSize>0) we
    // cannot calculate Length and Width. The calculation failed
    // and returns kFALSE
    // In reallity it is almost impossible to have a distribution
    // of cerenkov photons in the used pixels which is exactly symmetric
    // along one of the axis.
    //
    if (corrxy==0)
        return 4;

    //
    // calculate the basic Hillas parameters: orientation and size of axes
    // -------------------------------------------------------------------
    //
    // fDelta is the angle between the major axis of the ellipse and
    //  the x axis. It is independent of the position of the ellipse
    //  in the camera it has values between -pi/2 and pi/2 degrees
    //
    const Double_t d0    = corryy - corrxx;
    const Double_t d1    = corrxy*2;
    const Double_t d2    = d0 + sqrt(d0*d0 + d1*d1);
    const Double_t tand  = d2 / d1;
    const Double_t tand2 = tand*tand;

    fDelta = atan(tand);

    const Double_t s2 = tand2+1;
    const Double_t s  = sqrt(s2);

    fCosDelta =  1.0/s;   // need these in derived classes
    fSinDelta = tand/s;   // like MHillasExt

    Double_t axis1 = (tand2*corryy + d2 + corrxx)/s2/fSize;
    Double_t axis2 = (tand2*corrxx - d2 + corryy)/s2/fSize;

    //
    // fLength^2 is the second moment along the major axis of the ellipse
    // fWidth^2  is the second moment along the minor axis of the ellipse
    //
    // From the algorithm we get: fWidth <= fLength is always true
    //
    // very small numbers can get negative by rounding
    //
    fLength = axis1<0 ? 0 : sqrt(axis1);  // [mm]
    fWidth  = axis2<0 ? 0 : sqrt(axis2);  // [mm]

    SetReadyToSave();

    return 0;
}

// --------------------------------------------------------------------------
//
// This function is ment for special usage, please never try to set
// values via this function
//
void MHillas::Set(const TArrayF &arr)
{
    if (arr.GetSize() != 6)
        return;

    fLength = arr.At(0);  // [mm]        major axis of ellipse
    fWidth  = arr.At(1);  // [mm]        minor axis of ellipse
    fDelta  = arr.At(2);  // [rad]       angle of major axis with x-axis
    fSize   = arr.At(3);  // [#CerPhot]  sum of content of all pixels (number of Cherenkov photons)
    fMeanX  = arr.At(4);  // [mm]        x-coordinate of center of ellipse
    fMeanY  = arr.At(5);  // [mm]        y-coordinate of center of ellipse
}
