/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Robert Wagner 10/2002 <mailto:magicsoft@rwagner.de>
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHSigmabarTheta                                                         //
//                                                                          //
//  2D-Histogram in Sigmabar and Theta                                      //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHSigmabarTheta.h"

#include <TCanvas.h>

#include <math.h>

#include "MMcEvt.hxx"
#include "MSigmabar.h"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHSigmabarTheta);

// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram. 
//
MHSigmabarTheta::MHSigmabarTheta(const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHSigmabarTheta";
    fTitle = title ? title : "3-D histogram in sigmabar and theta";

    fHist.SetDirectory(NULL);

    fHist.SetTitle("3D-plot of sigmabar and theta");
    fHist.SetXTitle("\\theta [\\circ]");
    fHist.SetYTitle("\\bar{\\sigma}");
    fHist.SetZTitle("N");
}

// --------------------------------------------------------------------------
//
// Set binnings and prepare filling of the histogram
// 
Bool_t MHSigmabarTheta::SetupFill(const MParList *plist)
{
   fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
   if (!fMcEvt)
   {
       *fLog << err << "MMcEvt not found... aborting." << endl;
       return kFALSE;
   }

   fSigmabar = (MSigmabar*)plist->FindObject("MSigmabar");
   if (!fSigmabar)
   {
       *fLog << err << "MSigmabar not found... aborting." << endl;
       return kFALSE;
   }

   MBinning* binstheta  = (MBinning*)plist->FindObject("BinningTheta");
   if (!binstheta)
   {
       *fLog << err << "BinningTheta [MBinning] not found... aborting." << endl;
       return kFALSE;      
   }
   MBinning* binssigmabar = (MBinning*)plist->FindObject("BinningSigmabar");
   if (!binssigmabar)
   {
       *fLog << err << "BinningSigmabar [MBinning] not found... aborting." << endl;
       return kFALSE;      
   }

   SetBinning(&fHist, binstheta, binssigmabar);

   fHist.Sumw2(); 

   return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the histogram
// 
Bool_t MHSigmabarTheta::Fill(const MParContainer *par, Double_t w)
{
    fHist.Fill(fMcEvt->GetTheta()*kRad2Deg, fSigmabar->GetSigmabar());  
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
// 
void MHSigmabarTheta::Draw(Option_t *opt)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    fHist.Draw(opt);
    
    pad->Modified();
    pad->Update();
}

