/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek 4/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHMcEnergyMigration                                                     //
//                                                                          //
//  calculates the migration matrix E-est vs. E-true                        //
//  for different bins in Theta                                             //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHMcEnergyMigration.h"

#include <TCanvas.h>

#include "MMcEvt.hxx"

#include "MEnergyEst.h"
#include "MBinning.h"
#include "MHillasSrc.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHMcEnergyMigration);


// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram.
//
MHMcEnergyMigration::MHMcEnergyMigration(const char *name, const char *title)
    : fHist()
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHMcEnergyMigration";
    fTitle = title ? title : "3-D histogram   E-true E-est Theta";

    fHist.SetDirectory(NULL);

    fHist.SetTitle("3D-plot   E-true E-est Theta");
    fHist.SetXTitle("E_{true} [GeV]");
    fHist.SetYTitle("E_{est} [GeV]");
    fHist.SetZTitle("\\Theta [\\circ]");
}

// --------------------------------------------------------------------------
//
// Set the binnings and prepare the filling of the histograms
//
Bool_t MHMcEnergyMigration::SetupFill(const MParList *plist)
{
    fEnergy = (MEnergyEst*)plist->FindObject("MEnergyEst");
    if (!fEnergy)
    {
        *fLog << err << dbginf << "MEnergyEst not found... aborting." << endl;
        return kFALSE;
    }

    fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << err << dbginf << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    const MBinning* binsenergy = (MBinning*)plist->FindObject("BinningE");
    const MBinning* binstheta  = (MBinning*)plist->FindObject("BinningTheta");
    if (!binsenergy || !binstheta)
    {
        *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
        return kFALSE;
    }

    SetBinning(&fHist, binsenergy, binsenergy, binstheta);

    fHist.Sumw2();

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
//
void MHMcEnergyMigration::Draw(Option_t *opt)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->Divide(2,2);
    
    TH1 *h;

    pad->cd(1);
    gPad->SetBorderMode(0);
    gPad->SetLogx();
    h = fHist.Project3D("ex_pro");
    h->SetTitle("Distribution of E_{true}");
    h->SetXTitle("E_{true} [GeV]");
    h->SetBit(kCanDelete);
    h->Draw(opt);

    pad->cd(2);
    gPad->SetBorderMode(0);
    gPad->SetLogx();
    h = fHist.Project3D("ey_pro");
    h->SetTitle("Distribution of E_{est}");
    h->SetXTitle("E_{est} [GeV]");
    h->SetBit(kCanDelete);
    h->Draw(opt);

    pad->cd(3);
    gPad->SetBorderMode(0);
    h = fHist.Project3D("ez_pro");
    h->SetTitle("Distribution of \\Theta");
    h->SetXTitle("\\Theta [\\circ]");
    h->SetBit(kCanDelete);
    h->Draw(opt);

    pad->cd(4);
    gPad->SetBorderMode(0);
    fHist.Draw(opt);

    pad->Modified();
    pad->Update();
}

// --------------------------------------------------------------------------
//
// Fill the histogram
//
Bool_t MHMcEnergyMigration::Fill(const MParContainer *par, Double_t w)
{
    //    MHillasSrc &hil = *(MHillasSrc*)par;
    //    fHist.Fill(fMcEvt->GetEnergy(), hil.GetSize());

    // get E-true from fMcEvt and E-est from fEnergy
    fHist.Fill(fMcEvt->GetEnergy(), fEnergy->GetEnergy(), fMcEvt->GetTheta()*kRad2Deg);

    return kTRUE;
}
