/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek, 03/2003 <mailto:wittek@mppmu.mpg.de>
!   Author(s): Thomas Bretz, 04/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHNewImagePar
//
////////////////////////////////////////////////////////////////////////////
#include "MHNewImagePar.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"
#include "MBinning.h"
#include "MParList.h"

#include "MHillas.h"
#include "MNewImagePar.h"

ClassImp(MHNewImagePar);

// --------------------------------------------------------------------------
//
// Setup histograms 
//
MHNewImagePar::MHNewImagePar(const char *name, const char *title)
{
    fName  = name  ? name  : "MHNewImagePar";
    fTitle = title ? title : "Histograms of new image parameters";

    fHistLeakage1.SetName("Leakage1");
    fHistLeakage1.SetTitle("Leakage_{1}");
    fHistLeakage1.SetXTitle("Leakage");
    fHistLeakage1.SetYTitle("Counts");
    fHistLeakage1.SetDirectory(NULL);
    fHistLeakage1.SetFillStyle(4000);

    fHistLeakage2.SetName("Leakage2");
    fHistLeakage2.SetTitle("Leakage_{2}");
    fHistLeakage2.SetXTitle("Leakage");
    fHistLeakage2.SetYTitle("Counts");
    fHistLeakage2.SetDirectory(NULL);
    fHistLeakage2.SetLineColor(kBlue);
    fHistLeakage2.SetFillStyle(4000);
 
    fHistUsedPix.SetName("UsedPix");
    fHistUsedPix.SetTitle("Number of used pixels");
    fHistUsedPix.SetXTitle("Number of Pixels");
    fHistUsedPix.SetYTitle("Counts");
    fHistUsedPix.SetDirectory(NULL);
    fHistUsedPix.SetLineColor(kGreen);
    fHistUsedPix.SetFillStyle(4000);

    fHistCorePix.SetName("CorePix");
    fHistCorePix.SetTitle("Number of core pixels");
    fHistCorePix.SetXTitle("Number of Pixels");
    fHistCorePix.SetYTitle("Counts");
    fHistCorePix.SetDirectory(NULL);
    fHistCorePix.SetLineColor(kRed);
    fHistCorePix.SetFillStyle(4000);

    fHistConc.SetDirectory(NULL);
    fHistConc1.SetDirectory(NULL);
    fHistConc.SetName("Conc2");
    fHistConc1.SetName("Conc1");
    fHistConc.SetTitle("Ratio: Conc");
    fHistConc1.SetTitle("Ratio: Conc1");
    fHistConc.SetXTitle("Ratio");
    fHistConc1.SetXTitle("Ratio");
    fHistConc.SetYTitle("Counts");
    fHistConc1.SetYTitle("Counts");
    fHistConc.SetFillStyle(4000);
    fHistConc1.SetFillStyle(4000);
    fHistConc1.SetLineColor(kBlue);
    fHistConc.SetFillStyle(0);


    MBinning bins;

    bins.SetEdges(100, 0, 1);
    bins.Apply(fHistLeakage1);
    bins.Apply(fHistLeakage2);
    bins.Apply(fHistConc);
    bins.Apply(fHistConc1);

    bins.SetEdges(150, 0, 150);
    bins.Apply(fHistUsedPix);
    bins.Apply(fHistCorePix);
}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning
//
Bool_t MHNewImagePar::SetupFill(const MParList *plist)
{
    ApplyBinning(*plist, "Leakage", &fHistLeakage1);
    ApplyBinning(*plist, "Leakage", &fHistLeakage2);

    ApplyBinning(*plist, "Pixels", &fHistUsedPix);
    ApplyBinning(*plist, "Pixels", &fHistCorePix);

    ApplyBinning(*plist, "Conc",   &fHistConc);
    ApplyBinning(*plist, "Conc1",  &fHistConc1);

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Fill the histograms with data from a MNewImagePar container.
//
Bool_t MHNewImagePar::Fill(const MParContainer *par, Double_t w)
{
    const MNewImagePar &h = *(MNewImagePar*)par;

    fHistLeakage1.Fill(h.GetLeakage1(), w);
    fHistLeakage2.Fill(h.GetLeakage2(), w);

    fHistUsedPix.Fill(h.GetNumUsedPixels(), w);
    fHistCorePix.Fill(h.GetNumCorePixels(), w);

    fHistConc.Fill(h.GetConc(), w);
    fHistConc1.Fill(h.GetConc1(), w);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the two histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHNewImagePar::Draw(Option_t *)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->Divide(2,2);

    pad->cd(1);
    gPad->SetBorderMode(0);
    TAxis &x = *fHistLeakage1.GetXaxis();
    x.SetRangeUser(0.01, x.GetXmax());
    MH::Draw(fHistLeakage1, fHistLeakage2, "Leakage1 and Leakage2");

    pad->cd(2);
    gPad->SetBorderMode(0);
    MH::Draw(fHistCorePix, fHistUsedPix, "Number of core/used Pixels");

    pad->cd(3);
    gPad->SetBorderMode(0);
    MH::Draw(fHistConc1, fHistConc, "Concentrations");

    pad->cd(4);
    gPad->SetBorderMode(0);

    pad->Modified();
    pad->Update();
}

TH1 *MHNewImagePar::GetHistByName(const TString name)
{
    if (name.Contains("Leakage1", TString::kIgnoreCase))
        return &fHistLeakage1;
    if (name.Contains("Leakage2", TString::kIgnoreCase))
        return &fHistLeakage2;
    if (name.Contains("Conc", TString::kIgnoreCase))
        return &fHistConc;
    if (name.Contains("Conc1", TString::kIgnoreCase))
        return &fHistConc1;
    if (name.Contains("UsedPix", TString::kIgnoreCase))
        return &fHistUsedPix;
    if (name.Contains("CorePix", TString::kIgnoreCase))
        return &fHistCorePix;

    return NULL;
}
