/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and time saving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek 6/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHGamma                                                                 //
//                                                                          //
//  manipulation of alpha distributions                                     //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHGamma.h"

#include <TCanvas.h>
#include <TPad.h>

#include <math.h>

#include <TH2.h>
#include <TH3.h>

#include "MHAlphaEnergyTheta.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHGamma);


// --------------------------------------------------------------------------
//
// Default Constructor. 
//
MHGamma::MHGamma(const TString &name, const TString &title)
    : fHist(NULL), fProject(NULL)
{
    fName  = name.IsNull()  ? (TString)"MHGamma" : name;
    fTitle = title.IsNull() ? (TString)"3D-histogram of Alpha, E_est, Theta (Gamma sample)" : title;
}

TH3D *MHGamma::Subtract(const MHAlphaEnergyTheta &h1, const MHAlphaEnergyTheta &h2)
{
    return Subtract(h1.GetHist(), h2.GetHist());
}

TObject *MHGamma::DrawClone(Option_t *opt) const
{
    DrawClone1();
    DrawClone2();

    return NULL;
}

void MHGamma::DrawClone1() const
{
    if (!fHist)
        return;

    //
    // ------------- Part 1 ---------------------
    //
    TString titlex = fHist->GetXaxis()->GetTitle();
    TString titley = fHist->GetYaxis()->GetTitle();
    TString titlez = fHist->GetYaxis()->GetTitle();

    TString canvtitle = "3D-plot "; //of ";
    /*
     canvtitle += titlex;
     canvtitle += ", ";
     canvtitle += titley;
     canvtitle += ", ";
     canvtitle += titlez+" ";
     */
    canvtitle += "for 'gamma' sample";

    TCanvas &c = *MakeDefCanvas("Alpha", canvtitle);

    c.Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    TH1 *h;

    c.cd(1);
    h = ((TH3D*)(fHist))->Project3D(fName+"_ex");

    TString title= "Source-Antisource: ";
    h->SetTitle(title + titlex);
    h->SetXTitle(titlex);
    h->SetYTitle("Counts");

    h->Draw();
    h->SetBit(kCanDelete);

    c.cd(2);
    h = ((TH3D*)(fHist))->Project3D(fName+"_ey");

    h->SetTitle(title + titley);
    h->SetXTitle(titley);
    h->SetYTitle("Counts");

    h->Draw();
    h->SetBit(kCanDelete);
    gPad->SetLogx();

    c.cd(3);
    h = ((TH3D*)(fHist))->Project3D(fName+"_ez");

    h->SetTitle(title + titlez);
    h->SetXTitle(titlez);
    h->SetYTitle("Counts");

    h->Draw();
    h->SetBit(kCanDelete);

    c.cd(4);
    ((TH3D*)fHist)->DrawCopy();

    c.Modified();
    c.Update();
}


// --------------------------------------------------------------------------
//
// Calculate the histogram as the difference of two histograms :
//          fHist(gamma) = h1(source) - h2(antisource)
// 
TH3D *MHGamma::Subtract(const TH3D *h1, const TH3D *h2)
{
    if (fHist)
        delete fHist;

    fHist = new TH3D;
    fHist->SetName(fName);
    fHist->SetTitle(fTitle);
    fHist->SetDirectory(NULL);

    SetBinning((TH1*)fHist, (TH1*)h1);

    fHist->SetXTitle((((TH1*)h1)->GetXaxis())->GetTitle());
    fHist->SetYTitle((((TH1*)h1)->GetYaxis())->GetTitle());
    fHist->SetZTitle((((TH1*)h1)->GetZaxis())->GetTitle());

    fHist->Add((TH1*)h1, (TH1*)h2, 1, -1); // ROOT: FIXME!

    return fHist;
}

// --------------------------------------------------------------------------
//
// Integrate fHist(gamma) in the alpha range (lo, up)
// 
TH2D *MHGamma::GetAlphaProjection(Axis_t lo, Axis_t up)
{
    if (up < lo)
    {
        *fLog << err << fName << "MHGamma : Alpha projection not possible: lo=" << lo << " up=" << up << endl;
        return NULL;
    }

    TAxis &axe = *fHist->GetXaxis();

    Int_t ilo = axe.FindFixBin(lo);
    Int_t iup = axe.FindFixBin(up);

    const Double_t epslo1 = lo-axe.GetBinLowEdge(ilo);
    const Double_t epslo2 = axe.GetBinUpEdge(ilo)-lo;

    const Double_t epsup1 = up-axe.GetBinLowEdge(iup);
    const Double_t epsup2 = axe.GetBinUpEdge(iup)-up;

    const Double_t epslo = epslo1<epslo2 ? epslo1 : epslo2;
    const Double_t epsup = epsup1<epsup2 ? epsup1 : epsup2;

    if (epslo1>epslo2)
        ilo++;

    if (epsup1<epsup2)
        iup--;

    if (epslo>0.01*axe.GetBinWidth(ilo) || epsup>0.01*axe.GetBinWidth(iup))
    {
        *fLog << err << fName << "MHGamma : binning is not adequate for the requested projection:" << endl;
        *fLog << "Please specify a lower or upper limit which is not more than 1% away from a bin edge" << endl;
        *fLog << " epslo = " << epslo << endl;
        *fLog << " epsup = " << epsup << endl;
        *fLog << " dwl   = " << axe.GetBinWidth(ilo) << endl;
        *fLog << " dwu   = " << axe.GetBinWidth(iup) << endl;
        return NULL;
    }

    axe.SetRange(ilo, iup);

    fLo = lo;
    fHi = up;

    if (fProject)
        delete fProject;
    fProject = (TH2D*)fHist->Project3D(fName+"_ezy");

    const TString title = "2D-plot of ";
    const TString titley = fHist->GetYaxis()->GetTitle();
    const TString titlez = fHist->GetZaxis()->GetTitle();

    fProject->SetTitle(title + titley + " vs. " + titlez);
    fProject->SetXTitle(titley);
    fProject->SetYTitle(titlez);

    return fProject;
}

void MHGamma::DrawClone2() const
{
    if (!fProject)
        return;

    const TString titley = fHist->GetYaxis()->GetTitle();
    const TString titlez = fHist->GetZaxis()->GetTitle();

    TString canvtitle = "No.of Gammas ";//vs. ";
    /*
     canvtitle += titley;
     canvtitle += " and ";
     canvtitle += titlez;
     */
    canvtitle += Form("(%.1f < alpha < %.1f deg)", fLo, fHi);

    TCanvas &c = *MakeDefCanvas("Gamma", canvtitle);

    c.Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    TH1 *h;

    c.cd(1);
    h = fProject->ProjectionX(fName+"_xpro", -1, 9999, "E");
    TString title = "No.of gammas: ";
    h->SetTitle(title+titley);
    h->SetXTitle(titley);
    h->SetYTitle("Counts");

    h->Draw();
    h->SetBit(kCanDelete);
    gPad->SetLogx();

    c.cd(2);
    h = fProject->ProjectionY(fName+"_ypro", -1, 9999, "E");
    h->SetTitle(title+titlez);
    h->SetXTitle(titlez);
    h->SetYTitle("Counts");

    h->Draw();
    h->SetBit(kCanDelete);

    c.cd(3);

    fProject->DrawCopy();
    gPad->SetLogx();

    c.Modified();
    c.Update();
}
