/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Rudolf Bock     10/2001 <mailto:Rudolf.Bock@cern.ch>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
// MHillasSrcCalc
//
// Task to calculate the source dependant part of the hillas parameters
//
//////////////////////////////////////////////////////////////////////////////
#include "MHillasSrcCalc.h"

#include <fstream.h>

#include "MParList.h"

#include "MSrcPosCam.h"
#include "MHillasSrc.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHillasSrcCalc);

static const TString gsDefName  = "MHillasSrcCalc";
static const TString gsDefTitle = "Calculate position dependant image parameters";

// -------------------------------------------------------------------------
//
// Default constructor. The first argument is the name of a container
// containing the source position in the camera plain (MScrPosCam).
// The default is "MSrcPosCam". hil is the name of a container
// of type MHillasSrc (or derived) in which the parameters are stored
// The default is "MHillasSrc"
//
MHillasSrcCalc::MHillasSrcCalc(const char *src, const char *hil,
                               const char *name, const char *title)
    : fHillas(NULL), fSrcPos(NULL), fHillasSrc(NULL)
{
    fName  = name  ? name  : gsDefName.Data();
    fTitle = title ? title : gsDefTitle.Data();

    fSrcName     = src;
    fHillasName  = hil;
    fHillasInput = "MHillas";
}

// -------------------------------------------------------------------------
//
Bool_t MHillasSrcCalc::PreProcess(MParList *pList)
{
    fHillas = (MHillas*)pList->FindObject(fHillasInput, "MHillas");
    if (!fHillas)
    {
        *fLog << err << dbginf << "MHillas not found... aborting." << endl;
        return kFALSE;
    }

    fSrcPos = (MSrcPosCam*)pList->FindObject(fSrcName, "MSrcPosCam");
    if (!fSrcPos)
    {
        *fLog << err << dbginf << fSrcName << " [MSrcPosCam] not found... aborting." << endl;
        return kFALSE;
    }

    fHillasSrc = (MHillasSrc*)pList->FindCreateObj("MHillasSrc", fHillasName);
    if (!fHillasSrc)
        return kFALSE;

    fHillasSrc->SetSrcPos(fSrcPos);

    fErrors = 0;

    return kTRUE;
}

// -------------------------------------------------------------------------
//
Bool_t MHillasSrcCalc::Process()
{

    if (!fHillasSrc->Calc(fHillas))
    {
        fErrors++;
        return kCONTINUE;

    }
    return kTRUE;
}

// --------------------------------------------------------------------------
//
//  Prints some statistics about the hillas calculation. The percentage
//  is calculated with respect to the number of executions of this task.
//
Bool_t MHillasSrcCalc::PostProcess()
{
    if (GetNumExecutions()==0)
        return kTRUE;

    *fLog << inf << endl;
    *fLog << GetDescriptor() << " execution statistics:" << endl;
    *fLog << dec << setfill(' ');
    *fLog << " " << fErrors << " (" << (int)(fErrors*100/GetNumExecutions()) << "%) Evts skipped due to: Dist==0" << endl;
    *fLog << endl;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Implementation of SavePrimitive. Used to write the call to a constructor
// to a macro. In the original root implementation it is used to write
// gui elements to a macro-file.
//
void MHillasSrcCalc::StreamPrimitive(ofstream &out) const
{
    if (fHillas)
        fHillas->SavePrimitive(out);

    if (fSrcPos)
        fSrcPos->SavePrimitive(out);

    if (fHillasSrc)
        fHillasSrc->SavePrimitive(out);

    out << "   MHillasSrcCalc " << GetUniqueName() << "(";

    if (fSrcPos)
        out << "&" << fSrcPos->GetUniqueName();
    else
        out << "\"" << fSrcName << "\"";

    out << ", ";

    if (fHillasSrc)
        out << "&" << fHillasSrc->GetUniqueName();
    else
        out << "\"" << fHillasName << "\"";

    if (fName!=gsDefName || fTitle!=gsDefTitle)
    {
        out << ", \"" << fName << "\"";
        if (fTitle!=gsDefTitle)
            out << ", \"" << fTitle << "\"";
    }
    out << ");" << endl;
}
