/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    1/2002 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Wolfgang Wittek 1/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHAlphaEnergyTime                                                       //
//                                                                          //
//  3D-histogram in alpha, E-est and time                                   //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHAlphaEnergyTime.h"

#include <TCanvas.h>

#include <math.h>

#include "MHillasSrc.h"
#include "MEnergyEst.h"
#include "MTime.h"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHAlphaEnergyTime);

using namespace std;

// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram. 
//
MHAlphaEnergyTime::MHAlphaEnergyTime(const char *name, const char *title) 
  : fHist()
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHAlphaEnergyTime";
    fTitle = title ? title : "3-D histogram in alpha, energy and time";

    fHist.SetDirectory(NULL);

    fHist.SetTitle("3D-plot of alpha, E-est, time");
    fHist.SetXTitle("\\alpha [\\circ]");
    fHist.SetYTitle("E-est [GeV]            ");
    fHist.SetZTitle("time [s]");
}

// --------------------------------------------------------------------------
//
// Set binnings and prepare filling of the histogram
// 
Bool_t MHAlphaEnergyTime::SetupFill(const MParList *plist)
{
   fEnergy = (MEnergyEst*)plist->FindObject("MEnergyEst");
   if (!fEnergy)
   {
       *fLog << err << dbginf << "MHAlphaEnergyTime : MEnergyEst not found... aborting." << endl;
       return kFALSE;
   }

   fTime = (MTime*)plist->FindObject("MTime");
   if (!fTime)
   {
       *fLog << err << dbginf << "MHAlphaEnergyTime : MTime not found... aborting." << endl;
       return kFALSE;
   }

   MBinning* binsenergy = (MBinning*)plist->FindObject("BinningE");
   MBinning* binsalphaflux  = (MBinning*)plist->FindObject("BinningAlphaFlux");
   MBinning* binstime   = (MBinning*)plist->FindObject("BinningTime");
   if (!binsenergy || !binsalphaflux || !binstime)
   {
       *fLog << err << dbginf << "MHAlphaEnergyTime : At least one MBinning not found... aborting." << endl;
       return kFALSE;      
   }

   SetBinning(&fHist, binsalphaflux, binsenergy, binstime);

   fHist.Sumw2();

   return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the histogram
// 
Bool_t MHAlphaEnergyTime::Fill(const MParContainer *par, const Stat_t w)
{
    MHillasSrc &hil = *(MHillasSrc*)par;

    fHist.Fill(fabs(hil.GetAlpha()), fEnergy->GetEnergy(), 0.0001*fTime->GetTimeLo(), w);
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
// 
void MHAlphaEnergyTime::Draw(Option_t *opt)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    pad->Divide(2,2);

    TH1 *h;

    pad->cd(1);
    gPad->SetBorderMode(0);
    h = fHist.Project3D("ex");
    h->SetTitle("Distribution of \\alpha [\\circ]");
    h->SetXTitle("\\alpha [\\circ]");
    h->SetYTitle("Counts");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    pad->cd(2);
    gPad->SetBorderMode(0);
    gPad->SetLogx();
    h = fHist.Project3D("ey");
    h->SetTitle("Distribution of E-est [GeV]");
    h->SetXTitle("E-est [GeV]            ");
    h->SetYTitle("Counts");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    pad->cd(3);
    gPad->SetBorderMode(0);
    h = fHist.Project3D("ez");
    h->SetTitle("Distribution of time [s]");
    h->SetXTitle("time [s]");
    h->SetYTitle("Counts");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    pad->cd(4);
    gPad->SetBorderMode(0);
    fHist.Draw(opt);

    pad->Modified();
    pad->Update();

}

// --------------------------------------------------------------------------
//
// Integrate fHist     (Alpha,E-est,Time) over the Time to get
//           fAlphaEest(Alpha,E-est)
// 
TH2D *MHAlphaEnergyTime::IntegrateTime(const char *title, Bool_t draw)
{
    Int_t nzbins = fHist.GetNbinsZ();
    TAxis &axez  = *fHist.GetZaxis();
    axez.SetRange(1,nzbins);

    TH2D &fAlphaEest = *(TH2D *)fHist.Project3D("exy");

    fAlphaEest.SetTitle(title);
    fAlphaEest.SetXTitle("E-est [GeV]            ");
    fAlphaEest.SetYTitle("\\alpha  [  \\circ]");

    if (draw == kTRUE)
    {
      TCanvas &c = *MakeDefCanvas(title, title);

      gROOT->SetSelectedPad(NULL);

      fAlphaEest.DrawCopy();
      gPad->SetLogx();

      c.Modified();
      c.Update();
    }

    return &fAlphaEest;
}

// --------------------------------------------------------------------------
//
// Integrate fHist     (Alpha,E-est,Time) over E-est to get
//           fAlphaTime(Alpha,Time)
// 
TH2D *MHAlphaEnergyTime::IntegrateEest(const char *title, Bool_t draw)
{
    Int_t nybins = fHist.GetNbinsY();
    TAxis &axey  = *fHist.GetYaxis();
    axey.SetRange(1,nybins);

    TH2D &fAlphaTime = *(TH2D *)fHist.Project3D("exz");

    fAlphaTime.SetTitle(title);
    fAlphaTime.SetXTitle("Time [s]");
    fAlphaTime.SetYTitle("\\alpha  [  \\circ]");

    if (draw == kTRUE)
    {
      TCanvas &c = *MakeDefCanvas(title, title);

      gROOT->SetSelectedPad(NULL);

      fAlphaTime.DrawCopy();

      c.Modified();
      c.Update();
    }

    return &fAlphaTime;
}

// --------------------------------------------------------------------------
//
// Integrate fHist (Alpha,E-est,Time) over Eest and Time to get
//           fAlpha(Alpha)
// 
TH1D *MHAlphaEnergyTime::IntegrateEestTime(const char *title, Bool_t draw)
{
    Int_t nybins = fHist.GetNbinsY();
    TAxis &axey  = *fHist.GetYaxis();
    axey.SetRange(1,nybins);

    Int_t nzbins = fHist.GetNbinsZ();
    TAxis &axez  = *fHist.GetZaxis();
    axez.SetRange(1,nzbins);

    TH1D &fAlpha = *(TH1D *)fHist.Project3D("ex");

    fAlpha.SetTitle(title);
    fAlpha.SetXTitle("\\alpha  [  \\circ]");
    fAlpha.SetYTitle("Counts");

    if (draw == kTRUE)
    {
      TCanvas &c = *MakeDefCanvas(title, title);

      gROOT->SetSelectedPad(NULL);

      fAlpha.DrawCopy();

      c.Modified();
      c.Update();
    }

    return &fAlpha;
}







