/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MRawEvtHeader 
//    
// One Event is a sample of FADC measurements of different Pixels 
// (Photomultipliers) from the Camera of MAGIC. So all data (FADC) of the 
// interesting pixels are the modules of an event. To describe pixels the 
// class MRawPixel is used and the class MRawCrate to describe Crates.
// To define a single events some other data members are needed 
// (Time of the events, tirgger pattern of event..)
// 
// To describe one event on the level of FADC values the Class MRawEvtHeader is
// created. It has the following data members: 
//
// UInt_t    fDAQEvtNumber
// -----------------------
// This it the number of the Event in one 
// data run. The first event in this run get
// the number zero. The next one is one bigger.
//
// Assuming that one run takes 1 hour and a
// triggerrate of 1kHz the number must be able
// to reach 3.6e6 Events. To reach this number
// you need at least 22 bits. This is the reason
// why we use an integer (of root type UInt_t)
// with a range to 4.2e9. 
//
// MTime   fRawEvtTime
// -------------------
// Time of the event. 
// The start point of the time determination can be
// the millenium. From that start point the time is
// measured in 200ns-count. One day for example
// contains 432.e9 counts. An array of two unsigned Int is able to 
// contain 1.8e19 200ns-counts. This corresponds to 41.e6
// days. This should be more than the livetime of MAGIC.
// Private member of MTime.h
//
// UInt_t  fNumTrigLvl1
// --------------------
//
// Number of first level trigger
// This member counts the number of First Level Trigger
// between the last and this event. May be that due to 
// dead time of the DAQ this number is different from 1.
// If the DAQ is fast enough, this value should be 1. 
// This may be usefull in GammaRayBursts and if we 
// apply a data reduction in the DAQ-chain, which selects
// only good events. 
//
// UInt_t  fNumTrigLvl2
// ------------------ -
//
// Number of second level trigger
// This member counts the number of Second Level Trigger
// between the last and this event. 
//
// UInt_t  fTrigPattern[2]
// -----------------------
// Trigger Pattern used for this event
// Each event triggers for a particular configuration and each  
// configuration shoul have an ID (which is not fixed yet).
//
// UShort_t fAllLowGainOn
// ----------------------
// Type of Trigger. 
// This is a Byte (8 bit) to indicated if any of the pixels 
// have a non-negligible low gain (1) or not (0) 
//
/////////////////////////////////////////////////////////////////////////////
#include "MRawEvtHeader.h"

#include <iomanip>
#include <fstream>

#include "MLog.h"
#include "MLogManip.h"

#include "MTime.h"
#include "MArrayB.h"
#include "MRawRunHeader.h"

ClassImp(MRawEvtHeader);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. Create the array to store the data.
//
MRawEvtHeader::MRawEvtHeader(const char *name, const char *title)
{
    fName  = name  ? name  : "MRawEvtHeader";
    fTitle = title ? title : "Raw Event Header Information";

    //
    //   set all member to zero, init the pointer to ClonesArray,
    //
    fPixLoGainOn = new MArrayB;

    Clear();
}


// --------------------------------------------------------------------------
//
// Destructor. Deletes the array to store pixlogainon
//
MRawEvtHeader::~MRawEvtHeader()
{
    delete fPixLoGainOn;
}

// --------------------------------------------------------------------------
//
// you have to init the conatainer before you can read from
// a raw binary file
//
void MRawEvtHeader::Init(MRawRunHeader *rh, MTime *t)
{
    //
    // this is the number of entries in the array like specification
    //
    UInt_t fN = (rh->GetNumCrates() * rh->GetNumPixInCrate() + 7) / 8;

    //
    // initialize the array
    //
    fPixLoGainOn->Set(fN);

    //
    // this is the conatiner where we have to store the time of the event we
    // read from the input stream
    //
    fTime = t;
}

// --------------------------------------------------------------------------
//
//   Implementation of the Clear function
//
//   Resets all members to zero, clear the list of Pixels
//
void MRawEvtHeader::Clear(Option_t *)
{
    fDAQEvtNumber   = 0;
    fNumTrigLvl1    = 0;
    fNumTrigLvl2    = 0;
    fTrigPattern[0] = 0;
    fTrigPattern[1] = 0;
    fTrigType       = 0;
    fNumLoGainOn    = 0;
}

// --------------------------------------------------------------------------
//
//  This member function prints all Data of one Event to *fLog.
//
void MRawEvtHeader::Print(Option_t *o) const
{
    *fLog << all;
    *fLog << "DAQEvtNr: " << dec << fDAQEvtNumber << "  (";
    *fLog << "Trigger: ";
    *fLog << "NumLvl1=" << fNumTrigLvl1 << " ";
    *fLog << "NumLvl2=" << fNumTrigLvl2 << " ";
    *fLog << "Pattern=" << hex << setfill('0');
    *fLog << setw(2) << fTrigPattern[0];
    *fLog << setw(2) << fTrigPattern[1] << " " << dec;

    *fLog << "Type=";
    switch (fTrigType)
    {
    case 0:
        *fLog << "Trigger";
        break;
    case 1:
        *fLog << "Pedestal";
        break;
    case 2:
        *fLog << "Calibration";
        break;
    }
    *fLog << ")" << endl;
    *fLog << "Number of Lo Gains On: " << fNumLoGainOn << endl;

    for (unsigned int i=0; i<fPixLoGainOn->GetSize(); i++)
    {
        for (int j=0; j<8; j++)
        {
            const UInt_t on = (*fPixLoGainOn)[i]&(1<<j) ? 1 : 0;
            *fLog << on;
        }
    }
    *fLog << endl;
}

// --------------------------------------------------------------------------
//
// used to set the header information (eg. from MC)
//
void MRawEvtHeader::FillHeader(UInt_t uiN, Float_t ulTP)
{
    fDAQEvtNumber = uiN;
    fTrigPattern[0] = (UInt_t) (ulTP/4294967296.0) ;
    fTrigPattern[1] = (UInt_t) (ulTP-fTrigPattern[0]*4294967296.0);
}

// --------------------------------------------------------------------------
//
// read the EVENT HEADER information from the input stream
// return FALSE if there is now header anymore, else TRUE
//
int MRawEvtHeader::ReadEvt(istream &fin)
{
    fin.read((char*)&fDAQEvtNumber, 4);

    UInt_t fAbsTime[2];
    fin.read((char*)fAbsTime,       8);

    //
    // store the time of the event in the corresponding container
    //
    fTime->SetTime(fAbsTime[0], fAbsTime[1]);

    Byte_t dummy[4];

    fin.read((char*)&fNumTrigLvl1,  4);
    fin.read((char*)&fNumTrigLvl2,  4);
    fin.read((char*)fTrigPattern,   8);
    fin.read((char*)&fTrigType,     2);
    fin.read((char*)dummy,          2); // was fAllLoGainOn
    fin.read((char*)fPixLoGainOn->GetArray(), fPixLoGainOn->GetSize());

    fNumLoGainOn = 0;
    for (unsigned int i=0; i<fPixLoGainOn->GetSize(); i++)
        for (int j=0; j<8; j++)
            if ((*fPixLoGainOn)[i] & (1<<j))
                fNumLoGainOn++;

    fin.read((char*)&dummy, 4);

    return !fin.eof();
}
