/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek  2003 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////
//
// MHCT1Supercuts
//
// This class contains histograms for the supercuts
//
// the histograms are filled during the optimization of the supercuts
//
///////////////////////////////////////////////////////////////////////
#include "MHCT1Supercuts.h"

#include <math.h>

#include <TH1.h>
#include <TH2.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MParList.h"
#include "MHFindSignificance.h"

#include "MLog.h"
#include "MLogManip.h"



ClassImp(MHCT1Supercuts);

using namespace std;

// --------------------------------------------------------------------------
//
// Setup four histograms for Alpha, and Dist
//
MHCT1Supercuts::MHCT1Supercuts(const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHCT1Supercuts";
    fTitle = title ? title : "Container for supercuts";


    fDegree = new TH1F("Degree", "Degree of polynomial",   5, -0.5,  4.5);
    fProb   = new TH1F("Prob",   "chi2 probability",      40,    0,  1.0);
    fNdf    = new TH1F("NDF",    "NDF of polynomial fit", 60, -0.5, 59.5);
    fGamma  = new TH1F("gamma",  "gamma",                 40,  0.0,  8.0);
    fNexNon = new TH1F("NexNon", "Nex / Non",             50,  0.0,  1.0);
    fSigLiMa= new TH1F("Significance", "significance of gamma signal",   
                                                          60,  0.0, 120.0);
    fSigtoBackg= new TH2F("SigtoBackg", "Significance vs signal/backg ratio",
                          50,  0.0,  10.0, 60, 0.0, 120.0);
    fSigDegree = new TH2F("SigDegree", "Significance vs Degree of polynomial",
                           5, -0.5,   4.5, 60, 0.0, 120.0);
    fSigNbins  = new TH2F("SigNbins", "Significance vs number of bins",
                           40, -0.5, 79.5, 60, 0.0, 120.0);

    fDegree->SetDirectory(NULL);
    fProb->SetDirectory(NULL);
    fNdf->SetDirectory(NULL);
    fGamma->SetDirectory(NULL);
    fNexNon->SetDirectory(NULL);
    fSigLiMa->SetDirectory(NULL);
    fSigtoBackg->SetDirectory(NULL);
    fSigDegree->SetDirectory(NULL);
    fSigNbins->SetDirectory(NULL);

    fDegree->SetXTitle("order of polynomial");
    fProb->SetXTitle("chi2 probability of polynomial fit");
    fNdf->SetXTitle("NDF of polynomial fit");
    fGamma->SetXTitle("gamma");
    fNexNon->SetXTitle("Nex / Non");
    fSigLiMa->SetXTitle("significance");

    fSigtoBackg->SetXTitle("signa./background ratio");
    fSigtoBackg->SetYTitle("significance");

    fSigDegree->SetXTitle("order of polynomial");
    fSigDegree->SetYTitle("significance");

    fSigNbins->SetXTitle("number of bins");
    fSigNbins->SetYTitle("significance");

    fDegree->SetYTitle("Counts");
    fProb->SetYTitle("Counts");
    fNdf->SetYTitle("Counts");
    fGamma->SetYTitle("Counts");
    fNexNon->SetYTitle("Counts");
    fSigLiMa->SetYTitle("Counts");

    fSigtoBackg->SetZTitle("Counts");
    fSigDegree->SetZTitle("Counts");
    fSigNbins->SetZTitle("Counts");
}

// --------------------------------------------------------------------------
//
// Delete the histograms
//
MHCT1Supercuts::~MHCT1Supercuts()
{
    delete fDegree;
    delete fProb;
    delete fNdf;
    delete fGamma;
    delete fNexNon;
    delete fSigLiMa;

    delete fSigtoBackg;
    delete fSigDegree;
    delete fSigNbins;
}

// --------------------------------------------------------------------------
//
// Set the pointers (if there are any)
//

Bool_t MHCT1Supercuts::SetupFill(const MParList *plist)
{
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the histograms from the 'MHFindSignificance' container
//

Bool_t MHCT1Supercuts::Fill(const MParContainer *par, const Stat_t w)
{
    if (!par)
    {
        *fLog << err << "MHCT1Supercuts::Fill: Pointer (!=NULL) expected." << endl;
        return kFALSE;
    }

    MHFindSignificance &h = *(MHFindSignificance*)par;

    fDegree    ->Fill(h.GetDegree( ), w);
    fProb      ->Fill(h.GetProb(),    w);
    fNdf       ->Fill(h.GetNdf(),     w);
    fGamma     ->Fill(h.GetGamma(),   w);

    Double_t ratio = h.GetNon()>0.0 ? h.GetNex()/h.GetNon() : 0.0;
    fNexNon    ->Fill(ratio,          w);

    fSigLiMa   ->Fill(h.GetSigLiMa(), w);

    Double_t sigtobackg = h.GetNbg()!=0.0 ? h.GetNex() / h.GetNbg() : 0.0;
    fSigtoBackg->Fill(sigtobackg,    h.GetSigLiMa(), w);

    fSigDegree ->Fill(h.GetDegree(), h.GetSigLiMa(), w);
    fSigNbins  ->Fill(h.GetMbins(),  h.GetSigLiMa(), w);

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the two histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHCT1Supercuts::Draw(Option_t *)
{
  //TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
  //pad->SetBorderMode(0);
  //AppendPad("");

    TCanvas *pad = new TCanvas("Supercuts", "Supercut plots", 900, 900);
    gROOT->SetSelectedPad(NULL);

    pad->Divide(3, 3);

    pad->cd(1);
    gPad->SetBorderMode(0);
    fDegree->Draw();

    pad->cd(2);
    gPad->SetBorderMode(0);
    fProb->Draw();

    pad->cd(3);
    gPad->SetBorderMode(0);
    fNdf->Draw();

    pad->cd(4);
    gPad->SetBorderMode(0);
    fGamma->Draw();

    pad->cd(5);
    gPad->SetBorderMode(0);
    fNexNon->Draw();

    pad->cd(6);
    gPad->SetBorderMode(0);
    fSigLiMa->Draw();

    pad->cd(7);
    gPad->SetBorderMode(0);
    fSigtoBackg->Draw();

    pad->cd(8);
    gPad->SetBorderMode(0);
    fSigDegree->Draw();

    pad->cd(9);
    gPad->SetBorderMode(0);
    fSigNbins->Draw();

    pad->Modified();
    pad->Update();
}
//=========================================================================









