/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 12/2000 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Harald Kornmayer, 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//            
// MCerPhotEvt
//            
// Class Version 1:
//  - first version
//
// Class Version 2:
//  - added fLut to accelerate GetPixById a lot
//
/////////////////////////////////////////////////////////////////////////////
#include "MCerPhotEvt.h"

#include <math.h>
#include <limits.h>
#include <fstream>

#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"

ClassImp(MCerPhotEvt);

using namespace std;

// --------------------------------------------------------------------------
//
// Creates a MCerPhotPix object for each pixel in the event
//
MCerPhotEvt::MCerPhotEvt(const char *name, const char *title) : fNumPixels(0)
{
    fName  = name  ? name  : "MCerPhotEvt";
    fTitle = title ? title : "(Number of Photon)-Event Information";

    fPixels = new TClonesArray("MCerPhotPix", 0);
}

// --------------------------------------------------------------------------
//
// This is not yet implemented like it should.
//
/*
void MCerPhotEvt::Draw(Option_t* option) 
{
    //
    //   FIXME!!! Here the Draw function of the CamDisplay
    //   should be called to add the CamDisplay to the Pad.
    //   The drawing should be done in MCamDisplay::Paint
    //

    //    MGeomCam *geom = fType ? new MGeomCamMagic : new MGeomCamCT1;
    //    MCamDisplay *disp = new MCamDisplay(geom);
    //    delete geom;
    //    disp->DrawPhotNum(this);
}
*/

// --------------------------------------------------------------------------
//
// reset counter and delete netries in list.
//
void MCerPhotEvt::Reset()
{
    fNumPixels = 0;
    fLut.Set(0);
    // fPixels->Delete();
}

void MCerPhotEvt::FixSize()
{
    if (fPixels->GetEntriesFast() == (Int_t)fNumPixels)
        return;

    fPixels->ExpandCreateFast(fNumPixels);
}

// --------------------------------------------------------------------------
//
//  Dump the cerenkov photon event to *fLog
//
void MCerPhotEvt::Print(Option_t *) const
{
    const Int_t entries = fPixels->GetEntries();

    *fLog << GetDescriptor() << dec << endl;
    *fLog << " Number of Pixels: " << fNumPixels << "(" << entries << ")" << endl;

    for (Int_t i=0; i<entries; i++ )
        (*this)[i].Print();
}

// --------------------------------------------------------------------------
//
// Checks if in the pixel list is an entry with pixel id
//
Bool_t MCerPhotEvt::IsPixelExisting(Int_t id) const
{
    const MCerPhotPix *pix = GetPixById(id);
    return pix ? kTRUE : kFALSE;
} 

// --------------------------------------------------------------------------
//
//   Checks if in the pixel list is an entry with pixel id
//
Bool_t MCerPhotEvt::IsPixelUsed(Int_t id) const
{
    const MCerPhotPix *pix = GetPixById(id);
    return pix ? pix->IsPixelUsed() : kFALSE;
} 

// --------------------------------------------------------------------------
//
//   Checks if in the pixel list is an entry with pixel id
//
Bool_t MCerPhotEvt::IsPixelCore(Int_t id) const
{
    const MCerPhotPix *pix = GetPixById(id);
    return pix ? pix->IsPixelCore() : kFALSE;
} 

// --------------------------------------------------------------------------
//
// get the minimum number of photons  of all valid pixels in the list
// If you specify a geometry the number of photons is weighted with the
// area of the pixel
//
Float_t MCerPhotEvt::GetNumPhotonsMin(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return -5.;

    const UInt_t n = geom->GetNumPixels();

    Float_t minval = FLT_MAX;

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];
        if (!pix.IsPixelUsed())
            continue;

        const UInt_t id = pix.GetPixId();
        if (id>=n)
            continue;

        Float_t testval = pix.GetNumPhotons();

        if (geom)
            testval *= geom->GetPixRatio(id);

        if (testval < minval)
            minval = testval;
    }

    return minval;
}

// --------------------------------------------------------------------------
//
// get the maximum number of photons of all valid pixels in the list
// If you specify a geometry the number of photons is weighted with the
// area of the pixel
//
Float_t MCerPhotEvt::GetNumPhotonsMax(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return 50.;

    const UInt_t n = geom->GetNumPixels();

    Float_t maxval = -FLT_MAX;

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];
        if (!pix.IsPixelUsed())
            continue;

        const UInt_t id = pix.GetPixId();
        if (id>=n)
            continue;

        Float_t testval = pix.GetNumPhotons();
        if (geom)
            testval *= geom->GetPixRatio(id);

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

// --------------------------------------------------------------------------
//
// get the minimum ratio of photons/error
//
Float_t MCerPhotEvt::GetRatioMin(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return -5.;

    Float_t minval = (*this)[0].GetNumPhotons()/(*this)[0].GetErrorPhot();

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];
        if (!pix.IsPixelUsed())
            continue;

        Float_t testval = pix.GetNumPhotons()/pix.GetErrorPhot();
        if (geom)
            testval *= TMath::Sqrt(geom->GetPixRatio(pix.GetPixId()));

        if (testval < minval)
            minval = testval;
    }

    return minval;
}

// --------------------------------------------------------------------------
//
// get the maximum ratio of photons/error
//
Float_t MCerPhotEvt::GetRatioMax(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return -5.;

    Float_t maxval = -FLT_MAX;

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];
        if (!pix.IsPixelUsed())
            continue;

        Float_t testval = pix.GetNumPhotons()/pix.GetErrorPhot();
        if (geom)
            testval *= TMath::Sqrt(geom->GetPixRatio(pix.GetPixId()));

        if (testval > maxval)
            maxval = testval;
    }

    return maxval;
}

// --------------------------------------------------------------------------
//
// get the minimum of error
// If you specify a geometry the number of photons is weighted with the
// area of the pixel
//
Float_t MCerPhotEvt::GetErrorPhotMin(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return 50.;

    Float_t minval = FLT_MAX;

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];
        if (!pix.IsPixelUsed())
            continue;

        Float_t testval = pix.GetErrorPhot();

        if (geom)
            testval *= geom->GetPixRatio(pix.GetPixId());

        if (testval < minval)
            minval = testval;
    }
    return minval;
}

// --------------------------------------------------------------------------
//
// get the maximum ratio of photons/error
// If you specify a geometry the number of photons is weighted with the
// area of the pixel
//
Float_t MCerPhotEvt::GetErrorPhotMax(const MGeomCam *geom) const
{
    if (fNumPixels <= 0)
        return 50.;

    Float_t maxval = -FLT_MAX;

    for (UInt_t i=1; i<fNumPixels; i++)
    {
        const MCerPhotPix &pix = (*this)[i];
        if (!pix.IsPixelUsed())
            continue;

        Float_t testval = pix.GetErrorPhot();

        if (geom)
            testval *= geom->GetPixRatio(pix.GetPixId());

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

void MCerPhotEvt::RemoveUnusedPixels()
{
    TIter Next(fPixels);
    MCerPhotPix *pix = NULL;

    while ((pix=(MCerPhotPix*)Next()))
        if (!pix->IsPixelUsed())
            fPixels->Remove(pix);

    fPixels->Compress();
    fNumPixels=fPixels->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Return a pointer to the pixel with the requested idx. NULL if it doesn't
// exist. The Look-up-table fLut is used. If its size is zero (according
// to Rene this will happen if an old class object is loaded) we still
// try to search in the array.
//
MCerPhotPix *MCerPhotEvt::GetPixById(int idx) const
{
    if (fLut.GetSize()>0)
        return idx>=0 && idx<fLut.GetSize() ? (MCerPhotPix*)(fPixels->UncheckedAt(fLut[idx])) : 0;

    TIter Next(fPixels);
    MCerPhotPix *pix = NULL;

    while ((pix=(MCerPhotPix*)Next()))
        if (pix->GetPixId()==idx)
            return pix;

    return NULL;
}

// --------------------------------------------------------------------------
//
// Returns, depending on the type flag:
//
//  0: Number of Photons*PixRatio
//  1: Error*sqrt(PixRatio)
//  2: Cleaning level = Num Photons*sqrt(PixRatio)/Error
//  3: Number of Photons
//  4: Error
//
Bool_t MCerPhotEvt::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{
    MCerPhotPix *pix = GetPixById(idx);
    if (!pix || !pix->IsPixelUsed())
        return kFALSE;

    const Double_t ratio = cam.GetPixRatio(idx);

    switch (type)
    {
    case 1:
        val = pix->GetErrorPhot()*TMath::Sqrt(ratio);
        return kTRUE;
    case 2:
        if (pix->GetErrorPhot()<=0)
            return kFALSE;
        val = pix->GetNumPhotons()*TMath::Sqrt(ratio)/pix->GetErrorPhot();
        return kTRUE;
    case 3:
        val = pix->GetNumPhotons();
        break;
    case 4:
        val = pix->GetErrorPhot();
        break;
    default:
        val = pix->GetNumPhotons()*ratio;
        return kTRUE;
    }
    return kTRUE;
}

void MCerPhotEvt::DrawPixelContent(Int_t num) const
{
    *fLog << warn << "MCerPhotEvt::DrawPixelContent - not available." << endl;
}
