/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 12/2000 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MMcPedestalCopy
//
//  This task looks for the nformation about FADC pedestals in
//  MMcFadcHeader and translates it to the pedestal values in
//  MPedestalCam
//
//  Input Containers:
//   MMcFadcHeader
//
//  Output Containers:
//   MPedestalCam
//
/////////////////////////////////////////////////////////////////////////////
#include "MMcPedestalCopy.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MPedestalPix.h"
#include "MPedestalCam.h"

#include "MRawRunHeader.h"
#include "MMcRunHeader.hxx"
#include "MMcFadcHeader.hxx"

ClassImp(MMcPedestalCopy);

using namespace std;

MMcPedestalCopy::MMcPedestalCopy(const char *name, const char *title)
{
    fName  = name  ? name  : "MMcPedestalCopy";
    fTitle = title ? title : "Copy MC pedestals into MPedestal Container";

    //
    // This is not needed here using MReadMarsFile because for the
    // RunHeader tree the auto scheme is disabled by default
    //
    AddToBranchList("MMcFadcHeader.fPedesMean");
    AddToBranchList("MMcFadcHeader.fElecNoise");
}

// --------------------------------------------------------------------------
//
// Check for the run type. Return kTRUE if it is a MC run or if there
// is no MC run header (old camera files) kFALSE in case of a different
// run type
//
Bool_t MMcPedestalCopy::CheckRunType(MParList *pList) const
{
    const MRawRunHeader *run = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!run)
    {
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;
        return kTRUE;
    }

    return run->GetRunType() == kRTMonteCarlo;
}

// --------------------------------------------------------------------------
//
// Search for MPedestalCam and MMcFadcHeader.
//
Int_t MMcPedestalCopy::PreProcess(MParList *pList)
{
    fMcRun = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!fMcRun)
        *fLog << warn << dbginf << "MMcRunHeader not found... assuming camera<0.7" << endl;

    fPedCam = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!fPedCam)
        return kFALSE;

    fMcPed = (MMcFadcHeader*)pList->FindObject("MMcFadcHeader");
    if (!fMcPed)
    {
        *fLog << err << dbginf << "MMcFadcHeader not found... aborting." << endl;
        return kFALSE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for the runtype.
// Initialize the size of MPedestalCam to the number of pixels from
// MMcFadcHeader.
//
Bool_t MMcPedestalCopy::ReInit(MParList *pList)
{
    if (!CheckRunType(pList))
        return kFALSE;

    const int num = fPedCam->GetSize();

    const Bool_t camver70 = fMcRun && fMcRun->GetCamVersion()>=70;

    for (int i=0; i<num; i++)
    {
        MPedestalPix &pix = (*fPedCam)[i];

        // Here one should compute the Pedestal taking into account how
        // the MC makes the transformation analogic-digital for the FADC.

        const Float_t pedest = fMcPed->GetPedestal(i);
        const Float_t sigma  = camver70 ? fMcPed->GetPedestalRmsHigh(i) : fMcPed->GetElecNoise(i);

        pix.Set(pedest, sigma);
    }

    if (camver70)
        fPedCam->SetReadyToSave();

    return kTRUE;
}
