#include <TSystem.h>

#include "TFile.h"
#include "TTree.h"
#include "TBranch.h"

#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MArgs.h"
#include "MTime.h"
#include "MRawRunHeader.h"
#include "MRawEvtHeader.h"
#include "MRawEvtData.h"
#include "MRawCrateArray.h"
#include "MInputStreamID.h"

#include "MMcEvt.hxx"
#include "MMcTrig.hxx"

using namespace std;

/////////////////////////////////////////////////////////////////////////////
//
// This is an demonstration how to read in a merpped root file
// This is a demonstration how to use root, not how you should
// read a merpped file!
//
/////////////////////////////////////////////////////////////////////////////

static void StartUpMessage()
{
    gLog << all << endl;

    //                1         2         3         4         5
    //       12345678901234567890123456789012345678901234567890
    gLog << "==================================================" << endl;
    gLog << "              ReadRaw - MARS V" << MARSVER          << endl;
    gLog << "       MARS - Read and print raw data files"        << endl;
    gLog << "            Compiled on <" << __DATE__ << ">"       << endl;
    gLog << "               Using ROOT v" << ROOTVER             << endl;
    gLog << "==================================================" << endl;
    gLog << endl;
}

static void Usage()
{
    gLog << all << endl;
    gLog << "Sorry the usage is:" << endl;
    gLog << "   readraw [-vn] [-dec] [-a0] inputfile[.root]" << endl << endl;
    gLog << "     input file:   Magic DAQ binary file." << endl;
    gLog << "     -a0: Do not use Ansii codes." << endl;
    gLog << "     -vn: Verbosity level n [default=2]" << endl;
    gLog << "     -d1: print data in decimal values" << endl;
}

void EnableBranch(TTree *t, TString name, void *ptr)
{
    if (!t->GetBranch(name+"."))
        return;

    t->GetBranch(name+".")->SetAddress(ptr);
    gLog << " Found '" << name << "'" << endl;
}

int main(int argc, const char **argv)
{
    StartUpMessage();

    //
    // Evaluate arguments
    //
    MArgs arg(argc, argv);

    //
    // Set verbosity to highest level.
    //
    gLog.SetDebugLevel(arg.HasOption("-v") ? arg.GetIntAndRemove("-v") : 2);

    if (arg.HasOption("-a") && arg.GetIntAndRemove("-a")==0)
        gLog.SetNoColors();

    const bool kDecimal = arg.HasOption("-d") && arg.GetIntAndRemove("-d")==1;

    //
    // check for the right usage of the program
    //
    if (arg.GetNumArguments()!=1)
    {
        Usage();
        return -1;
    }

    //
    // This is to make argv[i] more readable insidethe code
    //
    TString kNamein = arg.GetArgumentStr(0);
  
    if (!kNamein.EndsWith(".root"))
        kNamein += ".root";

    //
    // Initialize Non-GUI (batch) mode
    //
    gROOT->SetBatch();

    //
    // check whether the given files are OK.
    //
    if (gSystem->AccessPathName(kNamein, kFileExists))
    {
        gLog << err << "Sorry, the input file '" << kNamein << "' doesn't exist." << endl;
        return -1;
    }

    //
    //  open the file
    //
    gLog << " Open the file '" << kNamein << "'" << endl;
    TFile input(kNamein, "READ");

    //
    // open the Run Header and read in
    //
    gLog << " Check for Tree 'RunHeaders'" << endl;
    TTree *runtree = (TTree*)input.Get("RunHeaders");
    if (!runtree)
        gLog << " WARNING: This file has NO Tree 'RunHeaders'" << endl << endl;
    else
    {
        gLog << " Entries in Tree RunHeaders: " << dec << runtree->GetEntries() << endl;

        MRawRunHeader *runheader = NULL;
        runtree->GetBranch("MRawRunHeader")->SetAddress(&runheader);
        runtree->GetEvent(0);
        runheader->Print();
    }

    //
    // open the DataTree and read in 
    //
    gLog << " Check the Tree 'Events'" << endl ;
    TTree *evttree = (TTree*)input.Get("Events") ;
    if (!evttree)
    {
        gLog << "Tree 'Events' not found in file... exit!" << endl;
        return -1;
    }

    //
    //  check the branches in the Tree 
    //
    gLog << " Check all the Branches in the Tree." << endl;
    gLog << endl;

    MRawEvtHeader  *evtheader = NULL;
    MTime          *evttime   = NULL;
    MRawEvtData    *evtdata   = NULL;
    MRawCrateArray *evtcrate  = NULL;
    MMcEvt         *evtmc     = NULL;
    MMcTrig        *trigmc    = NULL;

    EnableBranch(evttree, "MRawEvtHeader",  &evtheader);
    EnableBranch(evttree, "MTime",          &evttime);
    EnableBranch(evttree, "MRawEvtData",    &evtdata);
    EnableBranch(evttree, "MRawCrateArray", &evtcrate);
    EnableBranch(evttree, "MMcEvt",         &evtmc);
    EnableBranch(evttree, "MMcTrig",        &trigmc);

    //
    // loop over all entries 
    //
    const Int_t nent = (Int_t)evttree->GetEntries();

    gLog << endl;
    gLog << " Entries in Tree Data: " << dec << nent << endl;
    gLog << endl;

    for (Int_t i = 0; i<nent; i++)
    {
        gLog << "Entry: " << i << endl;

        //
        // readin event with the selected branches
        //
        evttree->GetEvent(i);

        if (evtmc)
            evtmc->Print();
        if (trigmc)
            trigmc->Print();
        if (evtheader)
            evtheader->Print();
        if (evttime)
            evttime->Print();
        if (evtcrate)
            evtcrate->Print();
        if (evtdata)
            evtdata->Print(kDecimal?"dec":"hex");

        gLog << endl;
    } 
    
    // end of small readin program

    return 0;
}
