/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MPedestalCam                                                            //
//                                                                         //
// Hold the Pedestal information for all pixels in the camera              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"

ClassImp(MPedestalCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. Creates a MPedestalPix object for each pixel
//
MPedestalCam::MPedestalCam(const char *name, const char *title)
{
    fName  = name  ? name  : "MPedestalCam";
    fTitle = title ? title : "Storage container for all Pedestal Information in the camera";

    fArray = new TClonesArray("MPedestalPix", 1);

//    for (int i=0; i<577; i++)
//        new ((*fArray)[i]) MPedestalPix;
}

// --------------------------------------------------------------------------
//
// Delete the array conatining the pixel pedest information
//
MPedestalCam::~MPedestalCam()
{
    delete fArray;
}

// --------------------------------------------------------------------------
//
// Set the size of the camera
//
void MPedestalCam::InitSize(const UInt_t i)
{
    fArray->ExpandCreate(i);
}

// --------------------------------------------------------------------------
//
// Get the size of the MPedestalCam
//
Int_t MPedestalCam::GetSize() const
{
    return fArray->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MPedestalPix &MPedestalCam::operator[](Int_t i)
{
    return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MPedestalPix &MPedestalCam::operator[](Int_t i) const
{
    return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Check if position i is inside bounds
//
Bool_t MPedestalCam::CheckBounds(Int_t i)
{
    return i < fArray->GetEntriesFast();
} 

void MPedestalCam::Clear(Option_t *o)
{
    fArray->ForEach(TObject, Clear)();
}

void MPedestalCam::Print(Option_t *o) const
{
    *fLog << all << GetDescriptor() << ":" << endl;
    int id = 0;

    TIter Next(fArray);
    MPedestalPix *pix;
    while ((pix=(MPedestalPix*)Next()))
    {
        id++;

        if (!pix->IsValid())
            continue;

        *fLog << id-1 << ": ";
        *fLog << pix->GetPedestal() << " " << pix->GetPedestalRms() << endl;
    }
}

Float_t MPedestalCam::GetPedestalMin(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t minval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval < minval)
            minval = testval;
    }
    return minval;
}

Float_t MPedestalCam::GetPedestalMax(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t maxval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

Bool_t MPedestalCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{
    switch (type)
    {
    case 0:
        val = (*this)[idx].GetPedestal();
        break;
    case 1:
        val = (*this)[idx].GetPedestalRms();
        break;
    default:
	return kFALSE;
    }
    return val>=0;
}

void MPedestalCam::DrawPixelContent(Int_t num) const
{
    *fLog << warn << "MPedestalCam::DrawPixelContent - not available." << endl;
}
