/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 11/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MReport
//
//  This is a base class for all reports comming from subsystems stored in
//  a report file.
//
//  Be carefull: The class name of all classes derived from this class
//               should start with 'MReport', see SetupReading
//
//////////////////////////////////////////////////////////////////////////////
#include "MReport.h"

#include "MLogManip.h"

#include "MTime.h"
#include "MParList.h"

ClassImp(MReport);

using namespace std;

// --------------------------------------------------------------------------
//
// Interpretes the beginning of a line which starts like:
//   status hour minute second millisec skip skip skip skip skip
// The identifier is assumed to be removed.
//
// While skip are numbers which won't enter the analysis
//
Bool_t MReport::InterpreteHeader(TString &str)
{
    int len, state;
    int hor, min, sec, ms;

    int n = sscanf(str.Data(),
                   fHasReportTime ?
                   " %d %*d %*d %*d %d %d %d %d "
                   "%*d %*d %*d %*d %*d %*d %*d %*d %n" :
                   " %d %*d %*d %*d %d %d %d %d ",
                   &state, &hor, &min, &sec, &ms, &len);
    if (n!=5)
    {
        *fLog << err << "ERROR - Cannot interprete Body of " << fIdentifier << endl;
        return kFALSE;
    }

    fState=state;
    if (fTime)
        fTime->SetTime(hor, min, sec, ms*1000000);

    str.Remove(0, len);
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Report Body must be overwritten. It will get the line idetified to belong
// to fIdentifier without the leading status and time infomration as an
// argument.
//
Bool_t MReport::InterpreteBody(TString &str)
{
    *fLog << warn << "No interpreter existing for: " << fIdentifier << endl;
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Interprets Header and Body of a report file line. Calls SetReadyToSave()
// in case the interpretation was successfull. And fTime->SetReadyToSave()
// when a corresponding time container exists.
//
Bool_t MReport::Interprete(TString &str)
{
    if (!InterpreteHeader(str))
        return kFALSE;

    if (!InterpreteBody(str))
        return kFALSE;

    SetReadyToSave();
    if (fTime)
        fTime->SetReadyToSave();

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for the existance of a corresponding MTime in the given parameter
// list. If it is not found a new one will be created. The name of the
// MTime object is created by taking the ClassName() of the derived MReport
// class and stripping the leading MReport
//
Bool_t MReport::SetupReading(MParList &plist)
{
    fTime = NULL;

    TString id(ClassName());
    if (!id.BeginsWith("MReport"))
    {
        *fLog << warn << " WARNING - Class name '" << id << "' ";
        *fLog << " doesn't begin with 'MReport'... no MTime assigned." << endl;
        return kFALSE;
    }

    id.Remove(0, 7);
    if (id.IsNull())
    {
        *fLog << warn << " WARNING - No postfix existing... no MTime assigned." << endl;
        return kFALSE;
    }

    id.Prepend("MTime");

    fTime = (MTime*)plist.FindCreateObj("MTime", id);
    if (!fTime)
        return kFALSE;

    return kTRUE;
}
