/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MRawFileRead                                                            //
//                                                                          //
//  This tasks reads the raw binary file like specified in the TDAS???      //
//  and writes the data in the corresponding containers which are           //
//  either retrieved from the parameter list or created and added.          //
//                                                                          //
//  Input Containers:                                                       //
//   -/-                                                                    //
//                                                                          //
//  Output Containers:                                                      //
//   MRawRunHeader, MRawEvtHeader, MRawEvtData, MRawCrateArray, MRawEvtTime //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MRawFileRead.h"

#include <fstream>

#include "MLog.h"
#include "MLogManip.h"

#include "MTime.h"
#include "MParList.h"
#include "MRawRunHeader.h"
#include "MRawEvtHeader.h"
#include "MRawEvtData.h"
#include "MRawCrateData.h"
#include "MRawCrateArray.h"

ClassImp(MRawFileRead);

using namespace std;

/*  ----------- please don't delete and don't care about (Thomas) ------------
#define kBUFSZ 64 //1024*1024*64
#include <iomanip.h>
class bifstream : public istream, public streambuf
{
private:
    char fBuffer[kBUFSZ]; //!
    FILE *fd;

    int sync()
    {
        memset(fBuffer, 0, kBUFSZ);
        return 0; 
    }
    int underflow()
    {
        int sz=fread(fBuffer, kBUFSZ, 1, fd);
        //int sz=fread(fBuffer, 1, kBUFSZ, fd);
        setg(fBuffer, fBuffer, fBuffer+kBUFSZ);

        return sz==1 ? *(unsigned char*)fBuffer : EOF;//EOF;
        //return sz==kBUFSZ ? *(unsigned char*)fBuffer : EOF;//EOF;
    }
public:
    bifstream(const char *name) : istream(this)
    {
        fd = fopen(name, "rb");
        setbuf(fBuffer, kBUFSZ);
    }
};
*/

// --------------------------------------------------------------------------
//
// Default constructor. It tries to open the given file.
//
MRawFileRead::MRawFileRead(const char *fname, const char *name, const char *title)
    : fFileName(fname), fIn(NULL)
{
    fName  = name  ? name  : "MRawFileRead";
    fTitle = title ? title : "Read task to read DAQ binary files";

    fIn = new ifstream;
}

// --------------------------------------------------------------------------
//
// Destructor. Delete input stream.
//
MRawFileRead::~MRawFileRead()
{
    delete fIn;
}

// --------------------------------------------------------------------------
//
// The PreProcess of this task checks for the following containers in the
// list:
//   MRawRunHeader <output>   if not found it is created
//   MRawEvtHeader <output>   if not found it is created
//   MRawEvtData <output>     if not found it is created
//   MRawCrateArray <output>  if not found it is created
//   MRawEvtTime <output>     if not found it is created (MTime)
//
// If all containers are found or created the run header is read from the
// binary file and printed.  If the Magic-Number (file identification)
// doesn't match we stop the eventloop.
//
// Now the EvtHeader and EvtData containers are initialized.
//
Int_t MRawFileRead::PreProcess(MParList *pList)
{
    //
    // open the input stream
    // first of all check if opening the file in the constructor was
    // successfull
    //
    fIn->open(fFileName);
    if (!(*fIn))
    {
        *fLog << err << "Error: Cannot open file '" << fFileName << "'" << endl;
        return kFALSE;
    }

    //
    //  check if all necessary containers exist in the Parameter list.
    //  if not create one and add them to the list
    //
    fRawRunHeader = (MRawRunHeader*)pList->FindCreateObj("MRawRunHeader");
    if (!fRawRunHeader)
        return kFALSE;

    fRawEvtHeader = (MRawEvtHeader*)pList->FindCreateObj("MRawEvtHeader");
    if (!fRawEvtHeader)
        return kFALSE;

    fRawEvtData = (MRawEvtData*)pList->FindCreateObj("MRawEvtData");
    if (!fRawEvtData)
        return kFALSE;

    fRawCrateArray = (MRawCrateArray*)pList->FindCreateObj("MRawCrateArray");
    if (!fRawCrateArray)
        return kFALSE;

    fRawEvtTime = (MTime*)pList->FindCreateObj("MTime", "MRawEvtTime");
    if (!fRawEvtTime)
        return kTRUE;

    //
    // Read RUN HEADER (see specification) from input stream
    //
    fRawRunHeader->ReadEvt(*fIn);
    fRawRunHeader->Print();

    if (fRawRunHeader->GetMagicNumber()!=kMagicNumber)
        return kFALSE;

    //
    // Give the run header information to the 'sub-classes'
    //
    fRawEvtHeader->Init(fRawRunHeader, fRawEvtTime);
    fRawEvtData  ->Init(fRawRunHeader);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// The Process reads one event from the binary file:
//  - The event header is read
//  - the run header is read
//  - all crate information is read
//  - the raw data information of one event is read
//
Int_t MRawFileRead::Process()
{
    //
    //  Read in the next EVENT HEADER (see specification),
    // if there is no next event anymore stop eventloop
    //
    if (!fRawEvtHeader->ReadEvt(*fIn))
        return kFALSE;

    //
    //  Get number of crates from the run header
    //
    const UShort_t nc = fRawRunHeader->GetNumCrates();

    //
    // Delete arrays which stores the pixel information (time slices)
    //
    fRawEvtData->ResetPixels();

    //
    // clear the TClonesArray which stores the Crate Information
    // and create a new array of the correct size
    //
    fRawCrateArray->SetSize(nc);

    //
    // read the CRATE DATA (see specification) from file
    //
    for (int i=0; i<nc; i++)
    {
        fRawCrateArray->GetEntry(i)->ReadEvt(*fIn);
        if (!*fIn)
            return kFALSE;

        fRawEvtData->ReadEvt(*fIn);
        if (!*fIn)
            return kFALSE;
    }

    {
        // FIXME This is a stupid workaround for the missing time stamp!
        //       Might be used depending on the run number in the future
        static Double_t tm = 0; // Range of roughly 8min
        const UInt_t ct = (*fRawCrateArray)[0]->GetFADCClockTick();

        tm = ct<tm ? fmod(tm, (UInt_t)(-1))+(UInt_t)(-1)+ct : ct;

        const Double_t mhz = 9.375;                        // [1e6 ticks/s]
        const Double_t t   = tm/mhz;                       // [us]
        cout << (ULong_t)t << endl;
        const UInt_t ns    = (UInt_t)fmod(t*1e3, 1e6);
        const UShort_t ms  = (UShort_t)fmod(t/1e3, 1e3);
        const Byte_t s     = (Byte_t)fmod(t/1e6, 60);

        // Create an artificial time stamp!
        UInt_t m     = (Byte_t)fmod(t/60e6, 60);
        //const Byte_t h     = (Byte_t)(t/3600e6);
        m += fRawRunHeader->GetRunNumber()*10;
        m %= 360; // 6h

        fRawEvtTime->Set(fRawRunHeader->GetDateYear(),
                         fRawRunHeader->GetDateMonth(),
                         fRawRunHeader->GetDateDay(),
                         m/60, m%60, s, ms, ns);

        //cout << tm << ": ";
        //fRawEvtTime->Print();
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
//  Close the file. Check whether the number of read events differs from
//  the number the file should containe (MRawRunHeader). Prints a warning
//  if it doesn't match.
//
Int_t MRawFileRead::PostProcess()
{
    fIn->close();

    //
    // Sanity check for the number of events
    //
    if (fRawRunHeader->GetNumEvents() == GetNumExecutions()-1)
        return kTRUE;

    *fLog << warn << "Warning - number of read events (" << GetNumExecutions()-1;
    *fLog << ") doesn't match number in run header (";
    *fLog << fRawRunHeader->GetNumEvents() << ")." << endl;

    return kTRUE;
}
