/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJExtractSignal
//
/////////////////////////////////////////////////////////////////////////////
#include "MJExtractSignal.h"

#include <TFile.h>
#include <TStyle.h>
#include <TCanvas.h>
#include <TSystem.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MRunIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MHCamera.h"

#include "MPedestalCam.h"
#include "MCalibrationCam.h"
#include "MHCamEvent.h"

#include "MReadMarsFile.h"
#include "MGeomApply.h"
#include "MBadPixelsMerge.h"
#include "MExtractSignal.h"
#include "MFillH.h"
#include "MCalibrate.h"
#include "MPedPhotCalc.h"
#include "MWriteRootFile.h"

#include "MJExtractSignal.h"
#include "MStatusDisplay.h"


ClassImp(MJExtractSignal);

using namespace std;

MJExtractSignal::MJExtractSignal(const char *name, const char *title) : fRuns(0)
{
    fName  = name  ? name  : "MJExtractSignal";
    fTitle = title ? title : "Tool to create a pedestal file (MPedestalCam)";
}
/*
void MJExtractSignal::CamDraw(TCanvas &c, MHCamera &cam1, MCamEvent &note, Int_t i)
{
    c.cd(i);
    gPad->SetBorderMode(0);
    MHCamera *obj1=(MHCamera*)cam1.DrawCopy("hist");
    obj1->AddNotify(note);

    c.cd(i+2);
    gPad->SetBorderMode(0);
    obj1->Draw();

    c.cd(i+4);
    gPad->SetBorderMode(0);
    TH1D *obj2 = (TH1D*)obj1->Projection();
    obj2->Draw();
    obj2->SetBit(kCanDelete);
    obj2->Fit("gaus","Q");
    obj2->GetFunction("gaus")->SetLineColor(kYellow);
}

void MJExtractSignal::CamDraw(TCanvas &c, MHCamera &cam1, MHCamera &cam2, MCamEvent &note)
{
    c.Divide(2, 3);

    CamDraw(c, cam1, note, 1);
    CamDraw(c, cam2, note, 2);
}

void MJExtractSignal::DisplayResult(MParList &plist)
{
    if (!fDisplay)
        return;

    //
    // Update display
    //
    TString title = fDisplay->GetTitle();
    title += "--  Calibration";
    title += fRunNumber<0 ? Form("File %s", (const char*)fFileName) : Form("Run #%d", fRunNumber);
    title += "  --";
    fDisplay->SetTitle(title);

    //
    // Get container from list
    //
    MGeomCam &geomcam = *(MGeomCam*)plist.FindObject("MGeomCam");

    // Create histograms to display
    MHCamera disp1 (geomcam, "Cal;Charge",        "Fitted Mean Charges");
    MHCamera disp3 (geomcam, "Cal;SigmaCharge",   "Sigma of Fitted Charges");
    MHCamera disp5 (geomcam, "Cal;ChargeProb",    "Probability of Fit");
    MHCamera disp6 (geomcam, "Cal;Time",          "Arrival Times");
    MHCamera disp7 (geomcam, "Cal;SigmaTime",     "Sigma of Arrival Times");
    MHCamera disp8 (geomcam, "Cal;TimeChiSquare", "Chi Square of Time Fit");
//    MHCamera disp9 (geomcam, "Cal;Ped",           "Pedestals");
//    MHCamera disp10(geomcam, "Cal;PedRms",        "Pedestal RMS");
    MHCamera disp11(geomcam, "Cal;RSigma",        "Reduced Sigmas");
    MHCamera disp12(geomcam, "Cal;PheFFactorMethod",               "Nr. of Phe's (F-Factor Method)");
    MHCamera disp13(geomcam, "Cal;MeanConversionFFactorMethod",    "Conversion Factor (F-Factor Method)");
    MHCamera disp14(geomcam, "Cal;MeanPhotInsidePlexiglass",       "Nr. of Photons (Blind Pixel Method)");
    MHCamera disp15(geomcam, "Cal;MeanConversionBlindPixelMethod", "Conversion Factor (Blind Pixel Method)");
    MHCamera disp16(geomcam, "Cal;RSigma/Charge",                  "Reduced Sigma per Charge");

    disp1.SetCamContent(fCalibrationCam, 0);
    disp1.SetCamError(fCalibrationCam, 1);

    disp3.SetCamContent(fCalibrationCam, 2);
    disp3.SetCamError(fCalibrationCam, 3);

    disp5.SetCamContent(fCalibrationCam, 4);

    disp6.SetCamContent(fCalibrationCam, 5);
    disp6.SetCamError(fCalibrationCam, 6);
    disp7.SetCamContent(fCalibrationCam, 6);
    disp8.SetCamContent(fCalibrationCam, 7);
    disp11.SetCamContent(fCalibrationCam, 10);

    disp12.SetCamContent(fCalibrationCam, 11);
    disp12.SetCamError(fCalibrationCam, 12);

    disp13.SetCamContent(fCalibrationCam, 13);
    disp13.SetCamError(fCalibrationCam, 14);

    disp14.SetCamContent(fCalibrationCam, 15);
    disp15.SetCamContent(fCalibrationCam, 16);
    disp16.SetCamContent(fCalibrationCam, 17);

    disp1.SetYTitle("Charge [FADC counts]");
    disp3.SetYTitle("\\sigma_{Charge} [FADC counts]");
    disp5.SetYTitle("P_{Charge} [1]");
    disp6.SetYTitle("Arr. Time [Time Slice Nr.]");
    disp7.SetYTitle("\\sigma_{Time} [Time Slices]");
    disp8.SetYTitle("\\chi^{2}_{Time} [1]");
//    disp9.SetYTitle("Ped [FADC Counts ]");
//    disp10.SetYTitle("RMS_{Ped} [FADC Counts ]");
    disp11.SetYTitle("\\sqrt{\\sigma^{2}_{Charge} - RMS^{2}_{Ped}} [FADC Counts]");
    disp12.SetYTitle("Nr. Photo-Electrons [1]");
    disp13.SetYTitle("Conversion Factor [PhE/FADC Count]");
    disp14.SetYTitle("Nr. Photons [1]");
    disp15.SetYTitle("Conversion Factor [Phot/FADC Count]");
    disp16.SetYTitle("Reduced Sigma / Charge [1]");

    gStyle->SetOptStat(1111);
    gStyle->SetOptFit();

    // Display Histograms: Charges
    CamDraw(fDisplay->AddTab("FitdCharge"), disp1, disp3, fCalibrationCam);

    // Fit Probability
    TCanvas &c12 = fDisplay->AddTab("Fit Prob.");
    c12.Divide(1, 3);

    c12.cd(1);
    gPad->SetBorderMode(0);
    MHCamera *obj1=(MHCamera*)disp5.DrawCopy("hist");
    obj1->AddNotify(fCalibrationCam);

    c12.cd(2);
    gPad->SetBorderMode(0);
    obj1->Draw();
    
    c12.cd(3);
    gPad->SetBorderMode(0);
    TH1D *obj2 = (TH1D*)obj1->Projection();
    obj2->Draw();
    obj2->SetBit(kCanDelete);
    obj2->Fit("pol0","Q");
    obj2->GetFunction("pol0")->SetLineColor(kYellow);

    // Times
    CamDraw(fDisplay->AddTab("FitdTimes"), disp6, disp8, fCalibrationCam);

    // Pedestals
    //CamDraw(fDisplay->AddTab("Pedestals"), disp9, disp10, calcam);

    // Reduced Sigmas
    CamDraw(fDisplay->AddTab("RedSigma"), disp11, disp16, fCalibrationCam);

    // F-Factor Method
    CamDraw(fDisplay->AddTab("F-Factor"), disp12, disp13, fCalibrationCam);

}
*/
Bool_t MJExtractSignal::WriteResult()
{
    if (fOutputPath.IsNull())
        return kTRUE;

    const TString oname = GetOutputFileP();

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "UPDATE");

    if (fDisplay && fDisplay->Write()<=0)
    {
        *fLog << err << "Unable to write MStatusDisplay to " << oname << endl;
        return kFALSE;
    }

    if (fPedPhotCam.Write()<=0)
    {
        *fLog << err << "Unable to write MPedPhotCam to " << oname << endl;
        return kFALSE;
    }

    if (fBadPixels.Write()<=0)
    {
        *fLog << err << "Unable to write MBadPixelsCam to " << oname << endl;
        return kFALSE;
    }

    return kTRUE;

}

void MJExtractSignal::SetOutputPath(const char *path)
{
    fOutputPath = path;
    if (fOutputPath.EndsWith("/"))
        fOutputPath = fOutputPath(0, fOutputPath.Length()-1);
}

TString MJExtractSignal::GetOutputFileD() const
{
    if (!fRuns)
        return "";

    return Form("%s/%s-F3.root", (const char*)fOutputPath, (const char*)fRuns->GetRunsAsFileName());
}
TString MJExtractSignal::GetOutputFileP() const
{
    if (!fRuns)
        return "";

    return Form("%s/%s-F2.root", (const char*)fOutputPath, (const char*)fRuns->GetRunsAsFileName());
}

Bool_t MJExtractSignal::ProcessD(MPedestalCam &pedcam)
{
    const TString fname = GetOutputFileD();

    if (gSystem->AccessPathName(fname, kFileExists))
        return ProcessFileD(pedcam);

    return kTRUE;
}

Bool_t MJExtractSignal::ProcessFileD(MPedestalCam &pedcam)
{
    if (!fRuns)
    {
        *fLog << err << "No Runs choosen... abort." << endl;
        return kFALSE;
    }
    if (fRuns->GetNumRuns() != fRuns->GetNumEntries())
    {
        *fLog << err << "Number of files found doesn't metch number of runs... abort." << endl;
        return kFALSE;
    }

    *fLog << inf;
    fLog->Separator(GetDescriptor());
    *fLog << "Calculate MExtractedSignalCam from Runs " << fRuns->GetRunsAsString() << endl;
    *fLog << endl;

    MReadMarsFile read("Events");
    read.DisableAutoScheme();
    static_cast<MRead&>(read).AddFiles(*fRuns);

    // Setup Tasklist
    MParList plist;
    plist.AddToList(&pedcam);

    MTaskList tlist;
    plist.AddToList(&tlist);

    MGeomApply      apply; // Only necessary to craete geometry
    MBadPixelsMerge merge(&fBadPixels);
    MExtractSignal  extract;

    MHCamEvent evt("ExtSignal");
    evt.SetType(0);
    MFillH fill(&evt, "MExtractedSignalCam");

    MWriteRootFile write(GetOutputFileD(), "RECREATE", fRuns->GetRunsAsString(), 2);
    write.AddContainer("MExtractedSignalCam", "Events");
    write.AddContainer("MTime",               "Events");
    write.AddContainer("MRawRunHeader",       "RunHeaders");
    write.AddContainer("MPedestalCam",        "RunHeaders");
    write.AddContainer("MBadPixelsCam",       "RunHeaders");

    tlist.AddToList(&read);
    tlist.AddToList(&apply);
    tlist.AddToList(&merge);
    tlist.AddToList(&extract);
    if (TestBit(kEnableGraphicalOutput))
        tlist.AddToList(&fill);
    tlist.AddToList(&write);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);

    // Execute first analysis
    if (!evtloop.Eventloop())
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    //DisplayResult(plist);

    //if (!WriteResult())
    //    return kFALSE;

    *fLog << inf << GetDescriptor() << ": Done." << endl;

    return kTRUE;
}

Bool_t MJExtractSignal::ReadPedPhotCam()
{
    const TString fname = GetOutputFileP();

    if (gSystem->AccessPathName(fname, kFileExists))
    {
        *fLog << err << "Input file " << fname << " doesn't exist." << endl;
        return kFALSE;
    }

    *fLog << inf << "Reading from file: " << fname << endl;

    TFile file(fname, "READ");
    if (fPedPhotCam.Read()<=0)
    {
        *fLog << "Unable to read MPedPhotCam from " << fname << endl;
        return kFALSE;
    }

    if (file.FindKey("MBadPixelsCam"))
    {
        MBadPixelsCam bad;
        if (bad.Read()<=0)
        {
            *fLog << "Unable to read MBadPixelsCam from " << fname << endl;
            return kFALSE;
        }
        fBadPixels.Merge(bad);
    }

    if (fDisplay /*&& !fDisplay->GetCanvas("Pedestals")*/) // FIXME!
        fDisplay->Read();

    return kTRUE;
}

Bool_t MJExtractSignal::ProcessP(MPedestalCam &pedcam, MCalibrationCam &calcam)
{
    if (!ReadPedPhotCam())
        return ProcessFileP(pedcam, calcam);

    return kTRUE;
}

Bool_t MJExtractSignal::ProcessFileP(MPedestalCam &pedcam, MCalibrationCam &calcam)
{
    if (!fRuns)
    {
        *fLog << err << "No Runs choosen... abort." << endl;
        return kFALSE;
    }
    if (fRuns->GetNumRuns() != fRuns->GetNumEntries())
    {
        *fLog << err << "Number of files found doesn't metch number of runs... abort." << endl;
        return kFALSE;
    }

    *fLog << inf;
    fLog->Separator(GetDescriptor());
    *fLog << "Calculate MExtractedSignalCam from Runs " << fRuns->GetRunsAsString() << endl;
    *fLog << endl;

    MReadMarsFile read("Events");
    read.DisableAutoScheme();
    static_cast<MRead&>(read).AddFiles(*fRuns);

    // Setup Tasklist
    MParList plist;
    plist.AddToList(&pedcam);
    plist.AddToList(&calcam);
    plist.AddToList(&fPedPhotCam);
    plist.AddToList(&fBadPixels);

    MTaskList tlist;
    plist.AddToList(&tlist);

    MGeomApply      apply; // Only necessary to craete geometry
    MBadPixelsMerge merge(&fBadPixels);
    MExtractSignal  extract;
    MCalibrate      calib;
    MPedPhotCalc    calc;

    MHCamEvent evt1("ExtOffset");
    MHCamEvent evt2("CalOffset");
    evt1.SetType(0);
    evt2.SetType(0);
    MFillH fill1(&evt1, "MExtractedSignalCam", "FillExtractedSignal");
    MFillH fill2(&evt2, "MCerPhotEvt",         "FillCerPhotEvt");

    tlist.AddToList(&read);
    tlist.AddToList(&apply);
    tlist.AddToList(&merge);
    tlist.AddToList(&extract);
    if (TestBit(kEnableGraphicalOutput))
        tlist.AddToList(&fill1);
    tlist.AddToList(&calib);
    tlist.AddToList(&calc);
    if (TestBit(kEnableGraphicalOutput))
        tlist.AddToList(&fill2);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);

    // Execute first analysis
    if (!evtloop.Eventloop())
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    //DisplayResult(plist);

    if (!WriteResult())
        return kFALSE;

    *fLog << inf << GetDescriptor() << ": Done." << endl;

    return kTRUE;
}

/*
Bool_t MJExtractSignal::ProcessFile(MPedestalCam *pedcam, MCalibrationCam *calcam)
{
    if (!fRuns)
    {
        *fLog << err << "No Runs choosen... abort." << endl;
        return kFALSE;
    }
    if (fRuns->GetNumRuns() != fRuns->GetNumEntries())
    {
        *fLog << err << "Number of files found doesn't metch number of runs... abort." << endl;
        return kFALSE;
    }

    Int_t type = 0;
    if (pedcam &&  calcam) type = 2;
    if (pedcam && !calcam) type = 3;

    *fLog << inf;
    fLog->Separator(GetDescriptor());
    *fLog << "Calculating from Runs " << fRuns->GetRunsAsString() << endl;
    *fLog << endl;

    MReadMarsFile read("Events");
    read.DisableAutoScheme();
    static_cast<MRead&>(read).AddFiles(*fRuns);

    // Setup Tasklist
    MParList plist;
    switch (type)
    {
    case 2:
        plist.AddToList(calcam);
        plist.AddToList(&fPedPhotCam);
    case 3:
        plist.AddToList(pedcam);
    }

    MTaskList tlist;
    plist.AddToList(&tlist);

    MGeomApply     apply; // Only necessary to craete geometry
    MExtractSignal extract;
    MCalibrate     calib;
    MPedPhotCalc   calc;


    MHCamEvent evt1("ExtOffset");
    MHCamEvent evt2("CalOffset");
    evt1.SetType(0);
    evt2.SetType(0);
    MFillH fill1(&evt1, "MExtractedSignalCam", "FillExtractedSignal");
    MFillH fill2(&evt2, "MCerPhotEvt",         "FillCerPhotEvt");

    tlist.AddToList(&read);
    tlist.AddToList(&apply);
    tlist.AddToList(&extract);
    if (TestBit(kEnableGraphicalOutput))
        tlist.AddToList(&fill1);
    tlist.AddToList(&calib);
    tlist.AddToList(&calc);
    if (TestBit(kEnableGraphicalOutput))
        tlist.AddToList(&fill2);


    MHCamEvent evt("ExtSignal");
    evt.SetType(0);
    MFillH fill(&evt, "MExtractedSignalCam");

    MWriteRootFile write(GetOutputFileD(), "RECREATE", fRuns->GetRunsAsString(), 2);
    write.AddContainer("MExtractedSignalCam", "Events");
    write.AddContainer("MTime",               "Events");
    write.AddContainer("MRawRunHeader",       "RunHeaders");
    write.AddContainer("MPedestalCam",        "RunHeaders");

    tlist.AddToList(&read);
    tlist.AddToList(&apply);
    tlist.AddToList(&extract);
    if (TestBit(kEnableGraphicalOutput))
        tlist.AddToList(&fill);
    tlist.AddToList(&write);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);

    // Execute first analysis
    if (!evtloop.Eventloop())
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    //DisplayResult(plist);

    //if (!WriteResult())
    //    return kFALSE;

    *fLog << inf << GetDescriptor() << ": Done." << endl;

    return kTRUE;


    // ------------------------------------------------------

    MGeomApply     apply; // Only necessary to craete geometry
    MExtractSignal extract;
    MCalibrate     calib;
    MPedPhotCalc   calc;

    MHCamEvent evt1("ExtOffset");
    MHCamEvent evt2("CalOffset");
    evt1.SetType(0);
    evt2.SetType(0);
    MFillH fill1(&evt1, "MExtractedSignalCam", "FillExtractedSignal");
    MFillH fill2(&evt2, "MCerPhotEvt",         "FillCerPhotEvt");

    tlist.AddToList(&read);
    tlist.AddToList(&apply);
    tlist.AddToList(&extract);
    if (TestBit(kEnableGraphicalOutput))
        tlist.AddToList(&fill1);
    tlist.AddToList(&calib);
    tlist.AddToList(&calc);
    if (TestBit(kEnableGraphicalOutput))
        tlist.AddToList(&fill2);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);

    // Execute first analysis
    if (!evtloop.Eventloop())
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    //DisplayResult(plist);

    if (!WriteResult())
        return kFALSE;

    *fLog << inf << GetDescriptor() << ": Done." << endl;

    return kTRUE;
    }
    */
