/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Raquel de los Reyes, 01/2004 <mailto:reyes@gae.ucm.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== *//////////////////////////////////////////////////////////////////////////////
//
// This macro made the check of the central control files (.rep files).
// It displays 8 tabs:
//   - Drive system
//   - Camera status
//   - HV power supplies
//   - LV power supplies
//   - Cooling system
//   - Trigger system
//   - Trigger macrocells
//   - Weather station
//
////////////////////////////////////////////////////////////////////////////

void CCDataCheck(const TString filename="CC_2003_11_30_20_29_20.root", const TString directory="../rootfiles/2003_12_01/")
{
  //
  // Check of the raw files
  //
  MStatusDisplay *d = new MStatusDisplay;
  d->SetTitle(Form("- %s -",filename));
  d->SetLogStream(&gLog, kTRUE);            // Disables output to stdout

  //
  // Create a empty Parameter List and an empty Task List
  // The tasklist is identified in the eventloop by its name
  //
  MParList  plist;
  
  MTaskList tlist;
  plist.AddToList(&tlist);

  //
  // Now setup the tasks and tasklist:
  // ---------------------------------
  //

  // Set all the MHVsTime classes:
  // Drive system
  MHVsTime hZd("MReportDrive.fNominalZd");
  hZd.SetName("Zd");
  MHVsTime hAz("MReportDrive.fNominalAz");
  hAz.SetName("Az");
  MHVsTime hDState("MReportDrive.fState");
  hDState.SetName("State");
  // HV power supplies
  MHVsTime hHVA("MCameraHV.fVoltageA");
  hHVA.SetName("HVA");
  MHVsTime hHVB("MCameraHV.fVoltageB");
  hHVB.SetName("HVB");
  MHVsTime hCA("MCameraHV.fCurrentA");
  hCA.SetName("CA");
  MHVsTime hCB("MCameraHV.fCurrentB");
  hCB.SetName("CB");
  // LV power supplies
  MHVsTime hLVTemp("MCameraLV.fTemp");
  hLVTemp.SetName("LVTemp");
  MHVsTime hLVHum("MCameraLV.fHumidity");
  hLVHum.SetName("LVHum");
  MHVsTime hLVStatus("MCameraHV.fStatus");
  hLVStatus.SetName("LVStatus");
  MHVsTime hLVRPS("MCameraLV.fRequestPowerSupply");
  hLVRPS.SetName("LVRPS");
  // Cooling system
  MHVsTime hTCenter("MCameraCooling.fTempCenter");
  hTCenter.SetName("TCenter");
  MHVsTime hTWall("MCameraCooling.fTempWall");
  hTWall.SetName("TWall");
  MHVsTime hTWater("MCameraCooling.fTempWater");
  hTWater.SetName("TWater");
  MHVsTime hTOptLink("MCameraCooling.fTempOptLink");
  hTOptLink.SetName("TOptLink");
  MHVsTime hHWall("MCameraCooling.fHumWall");
  hHWall.SetName("HWall");
  MHVsTime hHCenter("MCameraCooling.fHumCenter");
  hHCenter.SetName("HCenter");
  // Weather station
  MHVsTime hCCHum("MReportCC.fHumidity");
  hCCHum.SetName("CCHum");
  MHVsTime hCCTemp("MReportCC.fTemperature");
  hCCTemp.SetName("CCTemp");
  MHVsTime hCCWS("MReportCC.fWindSpeed");
  hCCWS.SetName("CCWS");
  MHVsTime hCCSR("MReportCC.fSolarRadiation");
  hCCSR.SetName("CCSR");
  // Trigger system
  MHVsTime hTrigBL2("MReportTrigger.fL2BeforePrescaler");
  hTrigBL2.SetName("TrigBL2");
  MHVsTime hTrigAL2("MReportTrigger.fL2AfterPrescaler");
  hTrigAL2.SetName("TrigAL2");
  MHVsTime hTrigStatus("MReportTrigger.fState");
  hTrigStatus.SetName("TrigStatus");
  // Camera status
  MHVsTime hCamStatus("MReportCamera.fState");
  hCamStatus.SetName("CamStatus");
  MHVsTime hCamSentStatus("MReportCamera.fStatus");
  hCamSentStatus.SetName("CamSentStatus");
  MHVsTime hCamStatusDC("MReportCamera.fStatusDC");
  hCamStatusDC.SetName("CamDCStatus");
  MHVsTime hHVStatus("MCameraHV.fStatus");
  hHVStatus.SetName("HVStatus");
  MHVsTime hLidsStatus("MCameraLids.fStatus");
  hLidsStatus.SetName("LidsStatus");
  MHVsTime hCoolStatus("MCameraCooling.fStatus");
  hCoolStatus.SetName("CoolingStatus");

  // Reads the trees of the root file and the analysed branches
  MReadReports read;
  read.AddTree("Drive");
  read.AddTree("Camera");
  read.AddTree("CC");
  read.AddTree("Trigger");
  read.AddFile(directory+filename);     // after the reading of the trees!!!
  read.AddToBranchList("MReportDrive.*");
  read.AddToBranchList("MCameraHV.*");
  read.AddToBranchList("MCameraLV.*");
  read.AddToBranchList("MCameraCooling.*");
  read.AddToBranchList("MReportCC.*");
  read.AddToBranchList("MReportTrigger.*");
  read.AddToBranchList("MReportCamera.*");
  read.AddToBranchList("MCameraLids.*");

  MGeomApply geomapl;
  tlist.AddToList(&geomapl);

  // Set of MHCamEvents classes
//    MHCamEvent HVdisplay("CamHV","Camera HV supplies");
//    HVdisplay.SetBit(MHCamera::kVariance);
//    plist.AddToList(&HVdisplay);

  // Set of MHPixVsTime classes (Trigger macroscells)
  MHPixVsTime htrigmc1(0,"TrigMacrocell1");
  htrigmc1.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc1);
  MHPixVsTime htrigmc2(1,"TrigMacrocell2");
  htrigmc2.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc2);
  MHPixVsTime htrigmc3(2,"TrigMacrocell3");
  htrigmc3.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc3);
  MHPixVsTime htrigmc4(3,"TrigMacrocell4");
  htrigmc4.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc4);
  MHPixVsTime htrigmc5(4,"TrigMacrocell5");
  htrigmc5.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc5);
  MHPixVsTime htrigmc6(5,"TrigMacrocell6");
  htrigmc6.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc6);
  MHPixVsTime htrigmc7(6,"TrigMacrocell7");
  htrigmc7.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc7);
  MHPixVsTime htrigmc8(7,"TrigMacrocell8");
  htrigmc8.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc8);
  MHPixVsTime htrigmc9(8,"TrigMacrocell9");
  htrigmc9.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc9);
  MHPixVsTime htrigmc10(9,"TrigMacrocell10");
  htrigmc10.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc10);
  MHPixVsTime htrigmc11(10,"TrigMacrocell11");
  htrigmc11.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc11);
  MHPixVsTime htrigmc12(11,"TrigMacrocell12");
  htrigmc12.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc12);
  MHPixVsTime htrigmc13(12,"TrigMacrocell13");
  htrigmc13.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc13);
  MHPixVsTime htrigmc14(13,"TrigMacrocell14");
  htrigmc14.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc14);
  MHPixVsTime htrigmc15(14,"TrigMacrocell15");
  htrigmc15.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc15);
  MHPixVsTime htrigmc16(15,"TrigMacrocell16");
  htrigmc16.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc16);
  MHPixVsTime htrigmc17(16,"TrigMacrocell17");
  htrigmc17.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc17);
  MHPixVsTime htrigmc18(17,"TrigMacrocell18");
  htrigmc18.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc18);
  MHPixVsTime htrigmc19(18,"TrigMacrocell19");
  htrigmc19.SetNameTime("MTimeTrigger");
  plist.AddToList(&htrigmc19);

  // Set of MH3 classes
  MH3 hError("MReportDrive.GetAbsError*60");
  hError.SetName("DeltaH");
  MBinning bins("BinningDeltaH");
  bins.SetEdges(18, 0, 3.6);
  plist.AddToList(&bins);
  MH3 hError2("MReportDrive.fNominalZd","MReportDrive.GetAbsError*60");
  hError2.SetName("DeltaHvsZd");
  MBinning bins2("BinningDeltaHvsZdX");
  MBinning bins3("BinningDeltaHvsZdY");
  bins2.SetEdges(90, 0, 90);
  bins3.SetEdges(18, 0, 3.6);
  plist.AddToList(&bins2);
  plist.AddToList(&bins3);
  MH3 hTempOptLink("MCameraCooling.fTempOptLink");
  hTempOptLink.SetName("TOptLink");
  MBinning bins4("BinningTOptLinkX");
  bins4.SetEdges(50, 0, 50);
  plist.AddToList(&bins4);
  MH3 hTrigZd("MReportDrive.fNominalZd","MReportTrigger.fL2AfterPrescaler");
  hTrigZd.SetName("TrigZd");
  MBinning bins5("BinningTrigZdX");
  bins5.SetEdges(90, 0, 90);
  plist.AddToList(&bins5);
  MBinning bins6("BinningTrigZdY");
  bins6.SetEdges(100, 0, 1500);
  plist.AddToList(&bins6);


  // Fill all the MH classes defined before
  MFillH fillZd(&hZd,             "MTimeDrive");
  MFillH fillAz(&hAz,             "MTimeDrive");
  MFillH fillError(&hError);
  MFillH fillDState(&hDState,     "MTimeDrive");
  MFillH fillError2(&hError2);
  MFillH fillHVA(&hHVA,           "MTimeCamera");
  MFillH fillHVB(&hHVB,           "MTimeCamera");
  MFillH fillCA(&hCA,             "MTimeCamera");
  MFillH fillCB(&hCB,             "MTimeCamera");
  MFillH fillLVTemp(&hLVTemp,     "MTimeCamera");
  MFillH fillLVHum(&hLVHum,       "MTimeCamera");
  MFillH fillLVStatus(&hLVStatus, "MTimeCamera");
  MFillH fillLVRPS(&hLVRPS,       "MTimeCamera");
  MFillH fillTCenter(&hTCenter,   "MTimeCamera");
  MFillH fillTWall(&hTWall,       "MTimeCamera");
  MFillH fillTWater(&hTWater,     "MTimeCamera");
  MFillH fillTOptLink(&hTOptLink, "MTimeCamera");
  MFillH fillTempOptLink(&hTempOptLink);
  MFillH fillHWall(&hHWall,       "MTimeCamera");
  MFillH fillHCenter(&hHCenter,   "MTimeCamera");
  MFillH fillCCHum(&hCCHum,       "MTimeCC");
  MFillH fillCCTemp(&hCCTemp,     "MTimeCC");
  MFillH fillCCWS(&hCCWS,         "MTimeCC");
  MFillH fillCCSR(&hCCSR,         "MTimeCC");
  MFillH fillTrigBL2(&hTrigBL2,   "MTimeTrigger");
  MFillH fillTrigAL2(&hTrigAL2,   "MTimeTrigger");
  MFillH fillTrigStatus(&hTrigStatus,   "MTimeTrigger");
  MFillH fillTrigZd(&hTrigZd);
  MFillH fillCamStatus(&hCamStatus,"MTimeCamera");
  MFillH fillCamSentStatus(&hCamSentStatus,"MTimeCamera");
  MFillH fillCamStatusDC(&hCamStatusDC, "MTimeCamera");
  MFillH fillHVStatus(&hHVStatus, "MTimeCamera");
  MFillH fillLidsStatus(&hLidsStatus, "MTimeCamera");
  MFillH fillCoolStatus(&hCoolStatus, "MTimeCamera");
  MFillH filltrigmc1("TrigMacrocell1","MReportTrigger");
  MFillH filltrigmc2("TrigMacrocell2","MReportTrigger");
  MFillH filltrigmc3("TrigMacrocell3","MReportTrigger");
  MFillH filltrigmc4("TrigMacrocell4","MReportTrigger");
  MFillH filltrigmc5("TrigMacrocell5","MReportTrigger");
  MFillH filltrigmc6("TrigMacrocell6","MReportTrigger");
  MFillH filltrigmc7("TrigMacrocell7","MReportTrigger");
  MFillH filltrigmc8("TrigMacrocell8","MReportTrigger");
  MFillH filltrigmc9("TrigMacrocell9","MReportTrigger");
  MFillH filltrigmc10("TrigMacrocell10","MReportTrigger");
  MFillH filltrigmc11("TrigMacrocell11","MReportTrigger");
  MFillH filltrigmc12("TrigMacrocell12","MReportTrigger");
  MFillH filltrigmc13("TrigMacrocell13","MReportTrigger");
  MFillH filltrigmc14("TrigMacrocell14","MReportTrigger");
  MFillH filltrigmc15("TrigMacrocell15","MReportTrigger");
  MFillH filltrigmc16("TrigMacrocell16","MReportTrigger");
  MFillH filltrigmc17("TrigMacrocell17","MReportTrigger");
  MFillH filltrigmc18("TrigMacrocell18","MReportTrigger");
  MFillH filltrigmc19("TrigMacrocell19","MReportTrigger");
//    MFillH fillHVD("CamHV","MCameraHV");

  // Do not display the result given by the default draw function
  fillZd.SetBit(MFillH::kDoNotDisplay);
  fillAz.SetBit(MFillH::kDoNotDisplay);
  fillError.SetBit(MFillH::kDoNotDisplay);
  fillDState.SetBit(MFillH::kDoNotDisplay);
  fillError2.SetBit(MFillH::kDoNotDisplay);
  fillHVA.SetBit(MFillH::kDoNotDisplay);
  fillHVB.SetBit(MFillH::kDoNotDisplay);
  fillCA.SetBit(MFillH::kDoNotDisplay);
  fillCB.SetBit(MFillH::kDoNotDisplay);
  fillLVTemp.SetBit(MFillH::kDoNotDisplay);
  fillLVHum.SetBit(MFillH::kDoNotDisplay);
  fillLVStatus.SetBit(MFillH::kDoNotDisplay);
  fillLVRPS.SetBit(MFillH::kDoNotDisplay);
  fillTCenter.SetBit(MFillH::kDoNotDisplay);
  fillTWall.SetBit(MFillH::kDoNotDisplay);
  fillTWater.SetBit(MFillH::kDoNotDisplay);
  fillTOptLink.SetBit(MFillH::kDoNotDisplay);
  fillTempOptLink.SetBit(MFillH::kDoNotDisplay);
  fillHWall.SetBit(MFillH::kDoNotDisplay);
  fillHCenter.SetBit(MFillH::kDoNotDisplay);
  fillCCHum.SetBit(MFillH::kDoNotDisplay);
  fillCCTemp.SetBit(MFillH::kDoNotDisplay);
  fillCCWS.SetBit(MFillH::kDoNotDisplay);
  fillCCSR.SetBit(MFillH::kDoNotDisplay);
  fillTrigBL2.SetBit(MFillH::kDoNotDisplay);
  fillTrigAL2.SetBit(MFillH::kDoNotDisplay);
  fillTrigStatus.SetBit(MFillH::kDoNotDisplay);
  fillTrigZd.SetBit(MFillH::kDoNotDisplay);
  fillCamStatus.SetBit(MFillH::kDoNotDisplay);
  fillCamSentStatus.SetBit(MFillH::kDoNotDisplay);
  fillCamStatusDC.SetBit(MFillH::kDoNotDisplay);
  fillHVStatus.SetBit(MFillH::kDoNotDisplay);
  fillLidsStatus.SetBit(MFillH::kDoNotDisplay);
  fillCoolStatus.SetBit(MFillH::kDoNotDisplay);
  filltrigmc1.SetBit(MFillH::kDoNotDisplay);
  filltrigmc2.SetBit(MFillH::kDoNotDisplay);
  filltrigmc3.SetBit(MFillH::kDoNotDisplay);
  filltrigmc4.SetBit(MFillH::kDoNotDisplay);
  filltrigmc5.SetBit(MFillH::kDoNotDisplay);
  filltrigmc6.SetBit(MFillH::kDoNotDisplay);
  filltrigmc7.SetBit(MFillH::kDoNotDisplay);
  filltrigmc8.SetBit(MFillH::kDoNotDisplay);
  filltrigmc9.SetBit(MFillH::kDoNotDisplay);
  filltrigmc10.SetBit(MFillH::kDoNotDisplay);
  filltrigmc11.SetBit(MFillH::kDoNotDisplay);
  filltrigmc12.SetBit(MFillH::kDoNotDisplay);
  filltrigmc13.SetBit(MFillH::kDoNotDisplay);
  filltrigmc14.SetBit(MFillH::kDoNotDisplay);
  filltrigmc15.SetBit(MFillH::kDoNotDisplay);
  filltrigmc16.SetBit(MFillH::kDoNotDisplay);
  filltrigmc17.SetBit(MFillH::kDoNotDisplay);
  filltrigmc18.SetBit(MFillH::kDoNotDisplay);
  filltrigmc19.SetBit(MFillH::kDoNotDisplay);
//    fillHVD.SetBit(MFillH::kDoNotDisplay);

  // Add all the task to the task list
  tlist.AddToList(&read);
  tlist.AddToList(&fillZd,       "Drive");
  tlist.AddToList(&fillAz,       "Drive");
  tlist.AddToList(&fillError,    "Drive");
  tlist.AddToList(&fillDState,   "Drive");
  tlist.AddToList(&fillError2,   "Drive");
  tlist.AddToList(&fillHVA,      "Camera");
  tlist.AddToList(&fillHVB,      "Camera");
  tlist.AddToList(&fillCA,       "Camera");
  tlist.AddToList(&fillCB,       "Camera");
  tlist.AddToList(&fillLVTemp,   "Camera");
  tlist.AddToList(&fillLVHum,    "Camera");
  tlist.AddToList(&fillLVStatus, "Camera");
  tlist.AddToList(&fillLVRPS,    "Camera");
  tlist.AddToList(&fillTCenter,  "Camera");
  tlist.AddToList(&fillTWall,    "Camera");
  tlist.AddToList(&fillTWater,   "Camera");
  tlist.AddToList(&fillTOptLink, "Camera");
  tlist.AddToList(&fillTempOptLink, "Camera");
  tlist.AddToList(&fillHWall,    "Camera");
  tlist.AddToList(&fillHCenter,  "Camera");
  tlist.AddToList(&fillCCHum,    "CC");
  tlist.AddToList(&fillCCTemp,   "CC");
  tlist.AddToList(&fillCCWS,     "CC");
  tlist.AddToList(&fillCCSR,     "CC");
  tlist.AddToList(&fillTrigBL2,  "Trigger");
  tlist.AddToList(&fillTrigAL2,  "Trigger");
  tlist.AddToList(&fillTrigStatus,  "Trigger");
  tlist.AddToList(&fillTrigZd);
  tlist.AddToList(&fillCamStatus,  "Camera");
  tlist.AddToList(&fillCamSentStatus,  "Camera");
  tlist.AddToList(&fillCamStatusDC,"Camera");
  tlist.AddToList(&fillHVStatus,"Camera");
  tlist.AddToList(&fillLidsStatus,"Camera");
  tlist.AddToList(&fillCoolStatus,"Camera");
  tlist.AddToList(&filltrigmc1, "Trigger");
  tlist.AddToList(&filltrigmc2, "Trigger");
  tlist.AddToList(&filltrigmc3, "Trigger");
  tlist.AddToList(&filltrigmc4, "Trigger");
  tlist.AddToList(&filltrigmc5, "Trigger");
  tlist.AddToList(&filltrigmc6, "Trigger");
  tlist.AddToList(&filltrigmc7, "Trigger");
  tlist.AddToList(&filltrigmc8, "Trigger");
  tlist.AddToList(&filltrigmc9, "Trigger");
  tlist.AddToList(&filltrigmc10, "Trigger");
  tlist.AddToList(&filltrigmc11, "Trigger");
  tlist.AddToList(&filltrigmc12, "Trigger");
  tlist.AddToList(&filltrigmc13, "Trigger");
  tlist.AddToList(&filltrigmc14, "Trigger");
  tlist.AddToList(&filltrigmc15, "Trigger");
  tlist.AddToList(&filltrigmc16, "Trigger");
  tlist.AddToList(&filltrigmc17, "Trigger");
  tlist.AddToList(&filltrigmc18, "Trigger");
  tlist.AddToList(&filltrigmc19, "Trigger");
//    tlist.AddToList(&fillHVD);

  //
  // Create and setup the eventloop
  //
  MEvtLoop evtloop;
  evtloop.SetParList(&plist);
  evtloop.SetDisplay(d);
    
  //
  // Execute your analysis
  //
  if (!evtloop.Eventloop())
    return;
 
  tlist.PrintStatistics();

  gStyle->SetPadGridX(kTRUE);
  gStyle->SetPadGridY(kTRUE);

  //
  // Drive report (pointing.C from T. Bretz)
  //
  TCanvas &c1 = d.AddTab("DRIVE SYSTEM");
  // ************* Plot the telescope tracking positions on the sky ***********
  TGraph *g1 = hZd.GetGraph();
  TGraph *g2 = hAz.GetGraph();
  TPad *p = new TPad("", "",0,0.05,0.6,0.95);
  p->Draw();
  p->cd();
  gPad->SetTheta(-90);
  gPad->SetPhi(90);
  gPad->SetBorderMode(0);
  gStyle->SetOptStat(0);
  TH2F h("pol", "Telescope Tracking Positions on the Sky", 16, 0, 1, 9, 0, 1);
  h.DrawClone("surf1pol");
  gPad->Modified();
  gPad->Update();
  TView *view = gPad->GetView();
  if (!view)
    {
      cout << "No View!" << endl;
      return;
    }
  Double_t *zd=g1->GetY();
  Double_t *az=g2->GetY();
  Double_t old[2] = {0,0};
  for (int i=0; i<g1->GetN(); i++)
    {
      az[i] += 180;
      az[i] *= TMath::Pi()/180;
      Double_t x[3] = { zd[i]*cos(az[i])/90, zd[i]*sin(az[i])/90, 0};
      Double_t y[3];
      view->WCtoNDC(x, y);
      if (old[0]!=0 && old[1]!=1)
        {
	  TLine *l = new TLine(y[0], y[1], old[0], old[1]);
	  l->SetLineColor(kBlue);
	  l->Draw();
        }
      TMarker *m = new TMarker(y[0], y[1], kFullDotMedium);
      m->SetMarkerColor(i==g1->GetN()-1 ? kGreen : kRed);
      m->Draw();
      old[0] = y[0];
      old[1] = y[1];
    }
  // ---------------------- Control deviation of the motors -------------------
  c1.cd();
  p = new TPad("", "", 0.6, 0, 1, 0.29);
  p->Draw();
  p->cd();
  gStyle->SetOptStat(1110);
  gStyle->SetStatFormat(".2g");
  MH3 *mh3 = (MH3*)hError.DrawClone("nonew");
  mh3->GetHist()->SetXTitle("\\Delta [arcmin]");
  mh3->GetHist()->SetYTitle("");
  mh3->GetHist()->SetTitle("Control deviation of the motors");
  mh3->GetHist()->SetStats(1);
  TLine ln;
  ln.SetLineColor(kGreen);
  ln.DrawLine(0.5*360*60/16384., 0, 0.5*360*60/16384., hError.GetHist()->GetMaximum());
  ln.SetLineColor(kYellow);
  ln.DrawLine(1.0*360*60/16384., 0, 1.0*360*60/16384., hError.GetHist()->GetMaximum());
  ln.SetLineColor(kRed);
  ln.DrawLine(2.0*360*60/16384., 0, 2.0*360*60/16384., hError.GetHist()->GetMaximum());
  // ---------------------- Plot the drive status vs time ---------------------
  c1.cd();
  p = new TPad("", "", 0.6, 0.29, 1, 0.42);
  p->Draw();
  p->cd();
  hvt = (MHVsTime*)hDState.DrawClone("nonew");
  hvt->GetGraph()->SetMarkerStyle(kFullDotSmall);
  TH1 *hist = hvt->GetGraph()->GetHistogram();
  TAxis *axey = hist->GetYaxis();
  TAxis *axex = hist->GetXaxis();
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  hist->SetTitle("");
  hist->SetStats(0);
  hist->SetMinimum(-0.5);
  hist->SetMaximum(4.5);
  axey->Set(5, -0.5, 4.5);
  axey->SetBinLabel(axey->FindFixBin(0), "Error");
  axey->SetBinLabel(axey->FindFixBin(1), "Stopped");
  axey->SetBinLabel(axey->FindFixBin(3), "Moving");
  axey->SetBinLabel(axey->FindFixBin(4), "Tracking");
  axey->SetLabelSize(0.15);
  axex->SetLabelSize(0.08);
  axex->SetTitleSize(0.09);
  axex->SetTitleOffset(0.45);
  // --------------- Control deviation of the motors vs zenith angle ----------
  c1.cd();
  p = new TPad("", "", 0.6, 0.71, 1, 1);
  p->Draw();
  p->cd();
  gStyle->SetOptStat(1110);
  gStyle->SetStatFormat(".2g");
  mh3 = (MH3*)hError2.DrawClone("nonew");
  mh3->GetHist()->SetXTitle("Zd [\\circ]");
  mh3->GetHist()->SetYTitle("\\Delta [arcmin]");
  mh3->GetHist()->SetTitle("Control deviation of the motors");
  mh3->GetHist()->SetStats(1);
  mh3->GetHist()->Draw("box");
  // -------------------------- Zenith angle vs time --------------------------
  c1.cd();
  p = new TPad("", "", 0.6, 0.42, 1, 0.71);
  p->Draw();
  p->cd();
  gPad->SetBorderMode(0);
  hvt = (MHVsTime*)hZd.DrawClone("nonew");
  hvt->GetGraph()->SetMarkerStyle(kFullDotSmall);
  if (hvt->GetGraph()->GetN())
    {
      hvt->GetGraph()->GetHistogram()->SetXTitle("Time");
      hvt->GetGraph()->GetHistogram()->SetYTitle("Zd [\\circ]");
      hvt->GetGraph()->GetHistogram()->SetTitle("Zd vs. Time");
      hvt->GetGraph()->GetHistogram()->SetStats(0);
    }
  
  
  //
  // Camera report 
  // 
  // *********************** Camera status ************************************
  TCanvas &c2 = d.AddTab("CAMERA STATUS");
  c2->Divide(2,3);
  c2->cd(1);
  TGraph *g = hCamStatus.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetTitle("Camera status");
  MHVsTime *clone1 = (MHVsTime*)hCamStatus.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  axey = hist->GetYaxis();
  hist->SetMinimum(-0.5);
  hist->SetMaximum(9.5);
  axey->Set(10,-0.5,9.5);
  axey->SetBinLabel(axey->FindFixBin(0),"EROR");
  axey->SetBinLabel(axey->FindFixBin(1),"ALARM");
  axey->SetBinLabel(axey->FindFixBin(2),"BLOCKED!");
  axey->SetBinLabel(axey->FindFixBin(3),"WARM!");
  axey->SetBinLabel(axey->FindFixBin(4),"HOT");
  axey->SetBinLabel(axey->FindFixBin(5),"HVRAMPING");
  axey->SetBinLabel(axey->FindFixBin(6),"OK");
  axey->SetBinLabel(axey->FindFixBin(7),"INIT");
  axey->SetBinLabel(axey->FindFixBin(8),"SHUTD");
  axey->SetBinLabel(axey->FindFixBin(9),"N/A");
  axey->SetLabelSize(0.07);
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  c2->cd(2);
  TGraph *g = hCamStatusDC.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetTitle("Camera DC status");
  MHVsTime *clone1 = (MHVsTime*)hCamStatusDC.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  axey = hist->GetYaxis();
  hist->SetMinimum(-0.5);
  hist->SetMaximum(9.5);
  axey->Set(10,-0.5,9.5);
  axey->SetBinLabel(axey->FindFixBin(0),"Error");
  axey->SetBinLabel(axey->FindFixBin(1),"Alarm");
  axey->SetBinLabel(axey->FindFixBin(3),"Hot");
  axey->SetBinLabel(axey->FindFixBin(5),"Ok");
  axey->SetBinLabel(axey->FindFixBin(6),"Warm");
  axey->SetBinLabel(axey->FindFixBin(9),"n/a");
  axey->SetLabelSize(0.07);
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  c2->cd(3);
  TGraph *g = hHVStatus.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetTitle("Camera HV status");
  MHVsTime *clone1 = (MHVsTime*)hHVStatus.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  axey = hist->GetYaxis();
  hist->SetMinimum(-0.5);
  hist->SetMaximum(9.5);
  axey->Set(10,-0.5,9.5);
  axey->SetBinLabel(axey->FindFixBin(0),"Error");
  axey->SetBinLabel(axey->FindFixBin(1),"Mismatch");
  axey->SetBinLabel(axey->FindFixBin(2),"Trip");
  axey->SetBinLabel(axey->FindFixBin(3),"Ramping");
  axey->SetBinLabel(axey->FindFixBin(4),"Off");
  axey->SetBinLabel(axey->FindFixBin(5),"Nominal");
  axey->SetBinLabel(axey->FindFixBin(6),"LimCurrentWarning");
  axey->SetBinLabel(axey->FindFixBin(9),"n/a");
  axey->SetLabelSize(0.07);
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  c2->cd(4);
  TGraph *g = hLidsStatus.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetTitle("Camera Lids status");
  MHVsTime *clone1 = (MHVsTime*)hLidsStatus.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  axey = hist->GetYaxis();
  hist->SetMinimum(-0.5);
  hist->SetMaximum(9.5);
  axey->Set(10,-0.5,9.5);
  axey->SetBinLabel(axey->FindFixBin(0),"Error");
  axey->SetBinLabel(axey->FindFixBin(1),"SafetyLimit");
  axey->SetBinLabel(axey->FindFixBin(4),"Closed");
  axey->SetBinLabel(axey->FindFixBin(5),"Opened");
  axey->SetBinLabel(axey->FindFixBin(6),"Moving");
  axey->SetBinLabel(axey->FindFixBin(7),"Stopped");
  axey->SetBinLabel(axey->FindFixBin(9),"n/a");
  axey->SetLabelSize(0.07);
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  c2->cd(5);
  TGraph *g = hCoolStatus.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetTitle("Camera cooling status");
  MHVsTime *clone1 = (MHVsTime*)hCoolStatus.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  axey = hist->GetYaxis();
  hist->SetMinimum(-0.5);
  hist->SetMaximum(9.5);
  axey->Set(10,-0.5,9.5);
  axey->SetBinLabel(axey->FindFixBin(0),"Error");
  axey->SetBinLabel(axey->FindFixBin(1),"Alarm");
  axey->SetBinLabel(axey->FindFixBin(4),"Off");
  axey->SetBinLabel(axey->FindFixBin(5),"Ok");
  axey->SetBinLabel(axey->FindFixBin(6),"Temp.Warning");
  axey->SetBinLabel(axey->FindFixBin(7),"Cond.Warning");
  axey->SetBinLabel(axey->FindFixBin(9),"n/a");
  axey->SetLabelSize(0.07);
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  c2->cd(6);
  TGraph *g = hCamSentStatus.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetTitle("Camera sentinel status");
  MHVsTime *clone1 = (MHVsTime*)hCamSentStatus.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  axey = hist->GetYaxis();
  hist->SetMinimum(-0.5);
  hist->SetMaximum(9.5);
  axey->Set(10,-0.5,9.5);
  axey->SetBinLabel(axey->FindFixBin(0),"ERROR");
  axey->SetBinLabel(axey->FindFixBin(1),"ALL IS OK");
  axey->SetBinLabel(axey->FindFixBin(2),"SUN IS PRESENT");
  axey->SetBinLabel(axey->FindFixBin(3),"BAD ATM. COND.");
  axey->SetBinLabel(axey->FindFixBin(4),"BAD T/RH CAMERA");
  axey->SetBinLabel(axey->FindFixBin(5),"HOT CAMERA");
  axey->SetBinLabel(axey->FindFixBin(6),"FADC FANS PROB.");
  axey->SetBinLabel(axey->FindFixBin(7),"CECO TIMEOUT");
  axey->SetBinLabel(axey->FindFixBin(9),"N/A");
  axey->SetLabelSize(0.07);
  hist->SetXTitle("Time");
  hist->SetYTitle("");

  // ******************** HV and currents of power supplies *******************
  TCanvas &c3 = d.AddTab("CAMERA HV");
  c3->Divide(1,2);
  c3->cd(1);
  // ---------------------------- High voltages -------------------------------
  TLegend *legHV = new TLegend(0.85,0.75,0.99,0.99);
  TGraph *g = hHVA.GetGraph();
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  g->SetTitle("Voltages of power supplies");
  legHV->AddEntry(g,"Power supply A (hvps1)","l");
  g = hHVB.GetGraph();
  g->SetMarkerColor(3);
  g->SetLineColor(3);
  legHV->AddEntry(g,"Power supply B (hvps2)","l");
  MHVsTime *clone1 = (MHVsTime*)hHVA.DrawClone("nonew");
  MHVsTime *clone2 = (MHVsTime*)hHVB.DrawClone("nonewsame");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("High voltage [V]");
  hist->SetMinimum(0);
  legHV->DrawClone();
  c3->cd(2);
  // ----------------------------- Currents ----------------------------------
  TLegend *legC = new TLegend(0.85,0.75,0.99,0.99);
  TGraph *g = hCA.GetGraph();
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  g->SetTitle("Currents of power supplies");
  legC->AddEntry(g,"Power supply A (curr1)","l");
  g = hCB.GetGraph();
  g->SetMarkerColor(3);
  g->SetLineColor(3);
  legC->AddEntry(g,"Power supply B (curr2)","l");
  MHVsTime *clone1 = (MHVsTime*)hCA.DrawClone("nonew");
  MHVsTime *clone2 = (MHVsTime*)hCB.DrawClone("nonewsame");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Current [mA]");
  hist->SetMinimum(0);
  legC->DrawClone();

  // ***************************** LV power supplies **************************
  TCanvas &c4 = d.AddTab("CAMERA LV");
  c4->Divide(2,2);
  c4->cd(1);
  TGraph *g = hLVTemp.GetGraph();
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  g->SetTitle("LV temperature");
  MHVsTime *clone1 = (MHVsTime*)hLVTemp.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Temperature [\\circ]");
  c4->cd(2);
  TGraph *g = hLVHum.GetGraph();
  g->SetMarkerColor(4);
  g->SetLineColor(4);
  g->SetTitle("LV humidity");
  MHVsTime *clone1 = (MHVsTime*)hLVHum.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Humidity [%]");
  c4->cd(3);
  TGraph *g = hLVStatus.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetTitle("LV status");
  MHVsTime *clone1 = (MHVsTime*)hLVStatus.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  axey = hist->GetYaxis();
  hist->SetMinimum(-0.5);
  hist->SetMaximum(9.5);
  axey->Set(10,-0.5,9.5);
  axey->SetBinLabel(axey->FindFixBin(0),"Error");
  axey->SetBinLabel(axey->FindFixBin(1),"Alarm");
  axey->SetBinLabel(axey->FindFixBin(2),"Trip");
  axey->SetBinLabel(axey->FindFixBin(4),"Off");
  axey->SetBinLabel(axey->FindFixBin(5),"On");
  axey->SetBinLabel(axey->FindFixBin(9),"n/a");
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  c4->cd(4);
  TGraph *g = hLVRPS.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetTitle("LV request power supply");
  MHVsTime *clone1 = (MHVsTime*)hLVRPS.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  axey = hist->GetYaxis();
  hist->SetMinimum(-0.5);
  hist->SetMaximum(1.5);
  axey->Set(2, -0.5, 1.5);
  axey->SetBinLabel(axey->FindFixBin(0), "OFF");
  axey->SetBinLabel(axey->FindFixBin(1), "ON");
  // ****************************** Cooling ***********************************
  TCanvas &c5 = d.AddTab("COOLING SYSTEM");
  gStyle->SetPadGridX(kTRUE);
  gStyle->SetPadGridY(kTRUE);
  c5->Divide(1,2);
  c5->cd(1);
  // -------------------------- Camera temperatures ---------------------------
  TLegend *legtemp = new TLegend(0.85,0.75,0.99,0.99);
  TGraph *g = hTCenter.GetGraph();
  g->SetMarkerColor(8);
  g->SetLineColor(8);
  g->SetTitle("Camera temperature");
  legtemp->AddEntry(g,"T at camera center","l");
  g = hTWall.GetGraph();
  g->SetMarkerColor(12);
  g->SetLineColor(12);
  legtemp->AddEntry(g,"T at camera wall","l");
  g = hTWater.GetGraph();
  g->SetMarkerColor(4);
  g->SetLineColor(4);
  legtemp->AddEntry(g,"T at water deposit","l");
  g = hTOptLink.GetGraph();
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  legtemp->AddEntry(g,"T at optical links","l");
  MHVsTime *clone1 = (MHVsTime*)hTCenter.DrawClone("nonew");
  MHVsTime *clone2 = (MHVsTime*)hTWall.DrawClone("nonewsame");
  MHVsTime *clone3 = (MHVsTime*)hTWater.DrawClone("nonewsame");
  MHVsTime *clone4 = (MHVsTime*)hTOptLink.DrawClone("nonewsame");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Temperature [\\circ C]");
  hist->SetMinimum(0);
  legtemp->DrawClone();
  // ------------------------- Camera relative humidity -----------------------
  c5->cd(2);
  gPad->SetBorderMode(0);
  gPad->Divide(2, 1);
  TVirtualPad *c = gPad;
  c->cd(1);
  TLegend *leghum = new TLegend(0.75,0.75,0.99,0.99);
  TGraph *g = hHCenter.GetGraph();
  g->SetMarkerColor(8);
  g->SetLineColor(8);
  g->SetTitle("Camera relative humidity");
  leghum->AddEntry(g,"RH at camera center","l");
  g = hHWall.GetGraph();
  g->SetMarkerColor(12);
  g->SetLineColor(12);
  leghum->AddEntry(g,"RH at camera wall","l");
  clone1 = (MHVsTime*)hHCenter.DrawClone("nonew");
  clone2 = (MHVsTime*)hHWall.DrawClone("nonewsame");
  hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Relative humidity [%]");
  hist->SetMaximum(50);
  hist->SetMinimum(0);
  leghum->DrawClone();
  // ---------------------- Temperature distribution --------------------------
  c->cd(2);
  TH1F *h3 = hTempOptLink.GetHistByName("TOptLink");
  h3->SetXTitle("Temperature [\\circ C]");
  h3->SetYTitle("");
  h3->SetLineColor(2);
  h3->SetTitle("Distribution of opt. links temperature");
  h3->SetStats(0);
  MH3 *mh3 = (MH3*)hTempOptLink.DrawClone("nonew");

  //
  // **************************** Trigger report ******************************
  //
  TCanvas &c6 = d.AddTab("TRIGGER SYSTEM");
  c6->Divide(1,3);
  c6->cd(1);
  gStyle->SetPadGridX(kTRUE);
  gStyle->SetPadGridY(kTRUE);
  TLegend *legtrig = new TLegend(0.85,0.85,0.99,0.99);
  TGraph *g = hTrigBL2.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  g->SetTitle("L2 trigger rate");
  legtrig->AddEntry(g,"Before prescaler","l");
  g = hTrigAL2.GetGraph();
  g->SetMarkerColor(4);
  g->SetLineColor(4);
  legtrig->AddEntry(g,"After prescaler","l");
  MHVsTime *clone1 = (MHVsTime*)hTrigBL2.DrawClone("nonew");
  MHVsTime *clone2 = (MHVsTime*)hTrigAL2.DrawClone("nonewsame");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("L2 trigger rate [Hz]"); 
  legtrig->DrawClone();
   c6->cd(2);
   TH1 *h4 = hTrigZd.GetHistByName("TrigZd");
   h4->SetXTitle("Zenith angle[\\circ]");
   h4->SetYTitle("Trigger rate [Hz]");
   h4->SetMarkerColor(4);
   h4->SetTitle("L2 trigger rate after prescaler");
   h4->SetStats(0);
   mh3 = (MH3*)hTrigZd.DrawClone("nonew");
   c6->cd(3);
  TGraph *g = hTrigStatus.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetTitle("Trigger status");
  MHVsTime *clone1 = (MHVsTime*)hTrigStatus.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  axey = hist->GetYaxis();
  hist->SetMinimum(-0.5);
  hist->SetMaximum(5.5);
  axey->Set(6,-0.5,5.5);
  axey->SetBinLabel(axey->FindFixBin(0),"Error");
  axey->SetBinLabel(axey->FindFixBin(1),"Idle");
  axey->SetBinLabel(axey->FindFixBin(2),"Loading");
  axey->SetBinLabel(axey->FindFixBin(3),"Ready");
  axey->SetBinLabel(axey->FindFixBin(4),"Active");
  axey->SetBinLabel(axey->FindFixBin(5),"Stopped");
  axey->SetLabelSize(0.07);
  hist->SetXTitle("Time");
  hist->SetYTitle("");
  // ***************************** Trigger of macrocells **********************
  TCanvas &c7 = d.AddTab("TRIG. MACROCELLs");
  c7->Divide(2,2);
  c7->cd(1);
  TLegend *legtrig = new TLegend(0.80,0.70,0.99,0.99);
  TGraph *g = htrigmc1.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(2);
  legtrig->AddEntry(g,Form("Macrocell 1: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  g->SetTitle("Trigger rate of mc 1 : (maximum)");
  MHPixVsTime *pixclone1 = (MHPixVsTime*)htrigmc1.DrawClone("nonew");
  TH1 *hist = pixclone1->GetGraph()->GetHistogram();
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  hist->SetXTitle("Time");
  hist->SetYTitle("Trigger [Hz]");
  legtrig->DrawClone();
  c7->cd(2);
  TLegend *legtrig = new TLegend(0.80,0.70,0.99,0.99);
  TGraph *g = htrigmc2.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(2);
  legtrig->AddEntry(g,Form("Macrocell 2: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  g->SetTitle("Trigger rate of mc 2-7 : (maximum)");
  MHPixVsTime *pixclone1 = (MHPixVsTime*)htrigmc2.DrawClone("nonew");
  TH1 *hist = pixclone1->GetGraph()->GetHistogram();
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  hist->SetXTitle("Time");
  hist->SetYTitle("Trigger [Hz]");
  g=htrigmc3.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(3);
  legtrig->AddEntry(g,Form("Macrocell 3: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc3.DrawClone("nonewsame");
  g=htrigmc4.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(4);
  legtrig->AddEntry(g,Form("Macrocell 4: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc4.DrawClone("nonewsame");
  g=htrigmc5.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(5);
  legtrig->AddEntry(g,Form("Macrocell 5: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc5.DrawClone("nonewsame");
  g=htrigmc6.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(6);
  legtrig->AddEntry(g,Form("Macrocell 6: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc6.DrawClone("nonewsame");
  g=htrigmc7.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(7);
  legtrig->AddEntry(g,Form("Macrocell 7: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc7.DrawClone("nonewsame");
  legtrig->DrawClone();
  c7->cd(3);
  TLegend *legtrig = new TLegend(0.80,0.70,0.99,0.99);
  TGraph *g = htrigmc8.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(2);
  legtrig->AddEntry(g,Form("Macrocell 8: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  g->SetTitle("Trigger rate of mc 8-13 : (maximum)");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  MHPixVsTime *pixclone1 = (MHPixVsTime*)htrigmc8.DrawClone("nonew");
  TH1 *hist = pixclone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Trigger [Hz]");
  g=htrigmc9.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(3);
  legtrig->AddEntry(g,Form("Macrocell 9: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc9.DrawClone("nonewsame");
  g=htrigmc10.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(4);
  legtrig->AddEntry(g,Form("Macrocell 10: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc10.DrawClone("nonewsame");
  g=htrigmc11.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(5);
  legtrig->AddEntry(g,Form("Macrocell 11: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc11.DrawClone("nonewsame");
  g=htrigmc12.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(6);
  legtrig->AddEntry(g,Form("Macrocell 12: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  pixclone1 = (MHPixVsTime*)htrigmc12.DrawClone("nonewsame");
  g=htrigmc13.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(7);
  legtrig->AddEntry(g,Form("Macrocell 13: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc13.DrawClone("nonewsame");
  legtrig->DrawClone();
  c7->cd(4);
  TLegend *legtrig = new TLegend(0.80,0.70,0.99,0.99);
  TGraph *g = htrigmc14.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(2);
  legtrig->AddEntry(g,Form("Macrocell 14: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  g->SetTitle("Trigger rate of mc 14-19 : (maximum)");
  MHPixVsTime *pixclone1 = (MHPixVsTime*)htrigmc14.DrawClone("nonew");
  TH1 *hist = pixclone1->GetGraph()->GetHistogram();
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  hist->SetXTitle("Time");
  hist->SetYTitle("Trigger [Hz]");
  g=htrigmc15.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(3);
  legtrig->AddEntry(g,Form("Macrocell 15: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc15.DrawClone("nonewsame");
  g=htrigmc16.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(4);
  legtrig->AddEntry(g,Form("Macrocell 16: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc16.DrawClone("nonewsame");
  g=htrigmc17.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(5);
  legtrig->AddEntry(g,Form("Macrocell 17: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc17.DrawClone("nonewsame");
  g=htrigmc18.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(6);
  legtrig->AddEntry(g,Form("Macrocell 18: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc18.DrawClone("nonewsame");
  g=htrigmc19.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(7);
  legtrig->AddEntry(g,Form("Macrocell 19: %3.0f",g->GetHistogram()->GetMaximum()),"p");
  if(hist->GetMaximum()<g->GetHistogram()->GetMaximum())
    hist->SetMaximum(g->GetHistogram()->GetMaximum());
  pixclone1 = (MHPixVsTime*)htrigmc19.DrawClone("nonewsame");
  legtrig->DrawClone();

  //
  // *************************** Weather station ******************************
  //
  TCanvas &c8 = d.AddTab("WEATHER STATION");
  c8->Divide(2,2);
  // ----------------------- Relative humidity --------------------------------
  c8->cd(1);
  TGraph *g = hCCHum.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(4);
  g->SetLineColor(4);
  g->SetTitle("Relative humidity");
  MHVsTime *clone1 = (MHVsTime*)hCCHum.DrawClone("nonew");
  TH1 *hist = clone1->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Humidity [%]"); 
  // -------------------------- Temperature -----------------------------------
  c8->cd(2);
  TGraph *g = hCCTemp.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(2);
  g->SetLineColor(2);
  g->SetTitle("Temperature");
  MHVsTime *clone2 = (MHVsTime*)hCCTemp.DrawClone("nonew");
  TH1 *hist = clone2->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Temperature [\\circ C]"); 
  // --------------------------- Wind speed -----------------------------------
  c8->cd(3);
  TGraph *g = hCCWS.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(3);
  g->SetLineColor(3);
  g->SetTitle("Wind speed");
  MHVsTime *clone3 = (MHVsTime*)hCCWS.DrawClone("nonew");
  TH1 *hist = clone3->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Wind speed [km/h]"); 
  // -------------------------- Solar radiation -------------------------------
  c8->cd(4);
  TGraph *g = hCCSR.GetGraph();
  g->SetMarkerStyle(kFullDotSmall);
  g->SetMarkerColor(9);
  g->SetLineColor(9);
  g->SetTitle("Solar radiation");
  MHVsTime *clone4 = (MHVsTime*)hCCSR.DrawClone("nonew");
  TH1 *hist = clone4->GetGraph()->GetHistogram();
  hist->SetXTitle("Time");
  hist->SetYTitle("Solar radiation [W/m^2]"); 

  //
  // Make sure the display hasn't been deleted by the user while the
  // eventloop was running.
  //
  if ((d = evtloop.GetDisplay()))
    {
      TString file;
      if (filename.Last('.')>0)
	file = filename(0, filename.Last('.')); 

      // Save data in a postscriptfile (status.ps)
      d->SaveAsPS(-1,Form("%s.ps",file));
    }
}














