/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug 11/2003 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MHCalibrationPINDiode
//
//  Performs all the necessary fits to extract the mean number of photons
//              out of the derived light flux
//
//////////////////////////////////////////////////////////////////////////////
#include "MHCalibrationPINDiode.h"
#include "MHCalibrationConfig.h"

#include <TH1.h>
#include <TF1.h>
#include <TPaveText.h>

ClassImp(MHCalibrationPINDiode);

using namespace std;

// --------------------------------------------------------------------------
//
// Default Constructor. 
//
MHCalibrationPINDiode::MHCalibrationPINDiode(const char *name, const char *title)
    : fChargeNbins(1000),
      fChargevsNbins(10000)
{

    fName  = name  ? name  : "MHCalibrationPINDiode";
    fTitle = title ? title : "Fill the accumulated charges and times all PINDiode events and perform fits";

    // Create a large number of bins, later we will rebin
    fChargeFirstHiGain = -100.5;
    fChargeLastHiGain  = 1999.5;

    fHPINDiodeCharge   = new TH1F("HPINDiodeCharge","Distribution of Summed FADC Slices PINDiode",
			          fChargeNbins,fChargeFirstHiGain,fChargeLastHiGain);
    fHPINDiodeCharge->SetXTitle("Sum FADC Slices");
    fHPINDiodeCharge->SetYTitle("Nr. of events");
    fHPINDiodeCharge->Sumw2();
    fHPINDiodeCharge->SetDirectory(NULL);

    fHPINDiodeChargevsN = new TH1I("HPINDiodeChargevsN","Sum of Hi Gain Charges vs. Event Number Pixel ",
				 fChargevsNbins,-0.5,(Axis_t)fChargevsNbins - 0.5);
    fHPINDiodeChargevsN->SetXTitle("Event Nr.");
    fHPINDiodeChargevsN->SetYTitle("Sum of Hi Gain FADC slices");
    fHPINDiodeChargevsN->SetDirectory(NULL);

    Clear();
}

MHCalibrationPINDiode::~MHCalibrationPINDiode()
{

  delete fHPINDiodeCharge;
  delete fHPINDiodeChargevsN;

  if (fChargeGausFit)
    delete fChargeGausFit;
  if (fFitLegend)
    delete fFitLegend;
}

void MHCalibrationPINDiode::Clear(Option_t *o)
{
  
  fTotalEntries            = 0;

  fChargeFirstHiGain       = -100.5;
  fChargeLastHiGain        = 1999.5;

  fChargeChisquare         = -1.;
  fChargeProb              = -1.;
  fChargeNdf               = -1;

  if (fChargeGausFit)
    delete fChargeGausFit;
  if (fFitLegend)
    delete fFitLegend;

  return;

}

void MHCalibrationPINDiode::Reset()
{

  Clear();

  fHPINDiodeCharge->Reset();
  fHPINDiodeChargevsN->Reset();
}


Bool_t MHCalibrationPINDiode::FillCharge(Float_t q)
{
    return (fHPINDiodeCharge->Fill(q) > -1);
}

Bool_t MHCalibrationPINDiode::FillChargevsN(Float_t q, Int_t n)
{
    return (fHPINDiodeChargevsN->Fill(n,q) > -1);
}

void MHCalibrationPINDiode::CutAllEdges()
{

  Int_t nbins = 20;

  StripZeros(fHPINDiodeCharge,nbins);

  fChargeFirstHiGain = fHPINDiodeCharge->GetBinLowEdge(fHPINDiodeCharge->GetXaxis()->GetFirst());
  fChargeLastHiGain  = fHPINDiodeCharge->GetBinLowEdge(fHPINDiodeCharge->GetXaxis()->GetLast())
                      +fHPINDiodeCharge->GetBinWidth(0);
  StripZeros(fHPINDiodeChargevsN,0);

}

