/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MBadPixelsPix
//
/////////////////////////////////////////////////////////////////////////////
#include "MBadPixelsPix.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MBadPixelsPix);

using namespace std;

const Char_t MBadPixelsPix::fgRunMask=MBadPixelsPix::kUnsuitableRun;

// ------------------------------------------------------------------------
//
// Initialize Pixel to be Ok.
//
MBadPixelsPix::MBadPixelsPix(const char* name, const char* title)
    : fInfo(0)
{
    fName  = name  ? name  : "MBadPixelsPix";
    fTitle = title ? title : "Container storeing bad pixel information for a single pixel";
}

// ------------------------------------------------------------------------
//
// Invalidate all bits which are not run-wise. This will be called for
// all entries in the parameter list, just before each time the task-list
// is executed.
//
void MBadPixelsPix::Reset()
{
    fInfo[0] &= fgRunMask;
}

// ------------------------------------------------------------------------
//
// Invalidate values (set=0 which mean Pixel OK)
//
void MBadPixelsPix::Clear(Option_t *o)
{
    fInfo.Reset(0);
}

// ------------------------------------------------------------------------
//
// Merge (bitwise or) the information in pix into this pixel.
//
void MBadPixelsPix::Merge(const MBadPixelsPix &pix)
{
    const Int_t n = pix.fInfo.GetSize();
    if (n>fInfo.GetSize())
        fInfo.Set(n);

    for (int i=0; i<n; i++)
        fInfo[i] |= pix.fInfo[n];
}

/****************************************************************************
           This is a collection of possible defects for later use
 ****************************************************************************/

/*
 1  PMT defective.
 2  Preamplifier defective.
 3  Optical link defective.
 4  HV cannot be set.
 7  HV unstable.
 5  HV readout defective.
 8  DC unstable.
 6  DC readout defective.
 9  Discriminator threshold cannot be set.
 10  Trigger delay cannot be set.
 11  Discriminator gives no output.
 <-? 12  Pixel out of L1T.
 13  FADC defective.
 14  FADC board digital information defective.
 */

/*
 1  Pixel shows no signal
 */

/*
 MCalibrationCalc - valid for the result of a calibration run:

  <-  1  Pedestals not available
  <-  2  FADC samples not available

      3  Hi-Gain saturated, no LoGain available

      4  Conversion Factor HiGain - LoGain not valid

      5  Cannot be calibrated at all
      6  Cannot be fitted - calibrated using Histogram Mean and RMS

      7  Mean Charge smaller than PedRMS
      8  Mean Charge smaller than Mean charge in camera minus 5 sigma
      9  Mean Charge bigger than Mean charge in camera plus 5 sigma

     10  Sigma Charge smaller than PedRMS

     11  Result of F-Factor    method invalid
     12  Result of Blind Pixel method invalid
     13  Result of PIN-Diode   method invalid
     14  Methods inconsistency

     15  Gains oscillate

  <- 16  Arrival Time not calculated
  <- 17  Arrival Time not valid

     18  Sigma  Arrival Time  bigger than  FADC window
     19  Mean   Arrival Time  at edge of   FADC window
     */

/*

Hardware defects which cannot be detected automatically by software. This might be stored at least in the data-base. I think we should wait until we implement these things...
Preamplifier defective.
Optical link defective.
HV cannot be set.
HV readout defective.
DC readout defective.
Discriminator threshold cannot be set.
Trigger delay cannot be set.
Discriminator gives no output.
FADC defective.
FADC board digital information defective.
Pixel out of L1T. (this is an important information, but not necessarily a defect, is it?)

In addition here are some cases which I think can be detected by software:
- no signal
- wrong signal
- hv problem
- dc problem
- Conversion Factor HiGain - LoGain not valid (what does this mean?)
- No calibration possible
- No fit possible - calibrated using Histogram Mean and RMS
- Mean Charge smaller than PedRMS
- Sigma Charge smaller than PedRMS
- Calib.methods inconsistency (there are in pricipal 6 combinations... do we need 6 bits?)
- Gains oscillate (what does it mean?)
- Sigma  Arrival Time  bigger than  FADC window (from calib)
- Mean   Arrival Time  at edge of   FADC window (from calib)
*/
