/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug 02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MHCalibrationChargePINDiode
//
//  Performs all the necessary fits to extract the mean number of photons
//              out of the derived light flux
//
//////////////////////////////////////////////////////////////////////////////
#include "MHCalibrationChargePINDiode.h"

#include <TH1.h>
#include <TPad.h>
#include <TVirtualPad.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MExtractedSignalPINDiode.h"
#include "MCalibrationChargePINDiode.h"

ClassImp(MHCalibrationChargePINDiode);

using namespace std;

const Int_t   MHCalibrationChargePINDiode::fgChargeNbins     = 200;
const Axis_t  MHCalibrationChargePINDiode::fgChargeFirst     = -0.5;
const Axis_t  MHCalibrationChargePINDiode::fgChargeLast      = 199.5;
const Int_t   MHCalibrationChargePINDiode::fgRmsChargeNbins  = 100;
const Axis_t  MHCalibrationChargePINDiode::fgRmsChargeFirst  = 0.;
const Axis_t  MHCalibrationChargePINDiode::fgRmsChargeLast   = 100.;  
const Int_t   MHCalibrationChargePINDiode::fgAbsTimeNbins    = 30;
const Axis_t  MHCalibrationChargePINDiode::fgAbsTimeFirst    = -0.5;
const Axis_t  MHCalibrationChargePINDiode::fgAbsTimeLast     = 29.5;
// --------------------------------------------------------------------------
//
// Default Constructor. 
//
MHCalibrationChargePINDiode::MHCalibrationChargePINDiode(const char *name, const char *title)
    : fPINDiode(NULL), fHRmsCharge()
{

  fName  = name  ? name  : "MHCalibrationChargePINDiode";
  fTitle = title ? title : "Fill the FADC sums of the PINDiode events and perform the fits";
  
  SetChargeNbins();
  SetChargeFirst();
  SetChargeLast();
  
  SetRmsChargeNbins();
  SetRmsChargeFirst();
  SetRmsChargeLast();
  
  SetAbsTimeNbins();
  SetAbsTimeFirst();
  SetAbsTimeLast();

  fHRmsCharge.SetName("HRmsCharge");
  fHRmsCharge.SetTitle("Distribution of Variances of summed FADC slices");  
  fHRmsCharge.SetXTitle("RMS (sum) [FADC slices]");
  fHRmsCharge.SetYTitle("Nr. of events");
  fHRmsCharge.UseCurrentStyle();
  fHRmsCharge.SetDirectory(NULL); 

  Clear();
}


void MHCalibrationChargePINDiode::Init()
{

  fHGausHist.SetName("HCalibrationChargePINDiode");
  fHGausHist.SetTitle("Distribution of Summed FADC slices PIN Diode");  
  fHGausHist.SetXTitle("Sum FADC Slices");
  fHGausHist.SetYTitle("Nr. of events");
  fHGausHist.SetBins(fChargeNbins,fChargeFirst,fChargeLast);
  //  fHGausHist.Sumw2();

  fHAbsTime.SetName("HAbsTimePINDiode");
  fHAbsTime.SetTitle("Distribution of Absolute Arrival Times PIN Diode ");  
  fHAbsTime.SetXTitle("Absolute Arrival Time [FADC slice nr]");
  fHAbsTime.SetYTitle("Nr. of events");
  fHAbsTime.SetBins(fAbsTimeNbins,fAbsTimeFirst,fAbsTimeLast);

  fHRmsCharge.SetBins(fRmsChargeNbins,fRmsChargeFirst,fRmsChargeLast);
}


void MHCalibrationChargePINDiode::Clear(Option_t *o)
{
  
  fExtractSlices = 0;

  MHCalibrationChargePix::Clear();
  return;
}

Bool_t MHCalibrationChargePINDiode::SetupFill(const MParList *pList) 
{
  Init();
  return kTRUE;
}

Bool_t MHCalibrationChargePINDiode::ReInit(MParList *pList)
{

  fPINDiode = (MCalibrationChargePINDiode*)pList->FindCreateObj("MCalibrationChargePINDiode");
  if (!fPINDiode)
  {
      *fLog << err << "MCalibrationChargePINDiode not found... aborting " << endl;
      return kFALSE;
  }

  return kTRUE;
}

Bool_t MHCalibrationChargePINDiode::Fill(const MParContainer *par, const Stat_t w)
{

  MExtractedSignalPINDiode *extractor = (MExtractedSignalPINDiode*)par;
  
  if (!extractor)
    {
      *fLog << err << "No argument in MExtractedSignalPINDiode::Fill... abort." << endl;
      return kFALSE;
    }
  
  Float_t slices = (Float_t)extractor->GetNumFADCSamples();
  
  if (slices == 0.)
    {
      *fLog << err << "Number of used signal slices in MExtractedSignalPINDiode is zero  ... abort." 
            << endl;
      return kFALSE;
    }
  
  if (fExtractSlices != 0. && slices != fExtractSlices )
    {
      *fLog << err << "Number of used signal slices changed in MExtractedSignalCam  ... abort."
           << endl;
      return kFALSE;
    }
  
  fExtractSlices = slices;
  
  const Float_t signal = (float)extractor->GetExtractedSignal();
  const Float_t time   = extractor->GetExtractedTime();
  const Float_t rms    = extractor->GetExtractedRms();

  FillHistAndArray(signal);
  FillAbsTime(time);
  FillRmsCharge(rms);

  return kTRUE;
}

Bool_t MHCalibrationChargePINDiode::Finalize() 
{
  
  if (IsGausFitOK() || IsEmpty())
    return kTRUE;
    
  FitGaus();
  FitRmsCharge();
  CreateFourierSpectrum();

  fPINDiode->SetMeanCharge(     GetMean()     );
  fPINDiode->SetMeanChargeErr(  GetMeanErr()  );
  fPINDiode->SetSigmaCharge(    GetSigma()    );
  fPINDiode->SetSigmaChargeErr( GetSigmaErr() );

  fPINDiode->SetAbsTimeMean(    GetAbsTimeMean() );
  fPINDiode->SetAbsTimeRms(     GetAbsTimeRms()  );

  fPINDiode->SetRmsChargeMean(     GetRmsChargeMean()        );
  fPINDiode->SetRmsChargeMeanErr(  GetRmsChargeMeanErr()     );
  fPINDiode->SetRmsChargeSigma(    GetRmsChargeSigma()       );
  fPINDiode->SetRmsChargeSigmaErr( GetRmsChargeSigmaErr()    );

  return kTRUE;
}

Bool_t MHCalibrationChargePINDiode::FillRmsCharge(const Float_t q)
{
  return fHRmsCharge.Fill(q) > -1;
}

// -----------------------------------------------------------
//
// Fits -- not yet implemented
//
Bool_t MHCalibrationChargePINDiode::FitRmsCharge(Option_t *option)
{
  return 1;
}


void MHCalibrationChargePINDiode::Draw(const Option_t *opt)
{

  TString option(opt);
  option.ToLower();
  
  Int_t win = 1;

  TVirtualPad *pad = gPad ? gPad : MH::MakeDefCanvas(this,600, 600);

  pad->SetTicks();
  pad->SetBorderMode(0);
    
  if (option.Contains("rmscharge"))
    win++;

  pad->Divide(1,win);
  pad->cd(1);

  if (!IsEmpty())
    gPad->SetLogy();

  MHGausEvents::Draw(opt);

  if (win > 1)
    {
      pad->cd(2);
      fHRmsCharge.Draw(opt);
    }
}




