/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.

! *
!
!
!   Author(s): Markus Gaug   02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                               
// MCalibrationQECam                                               
//                                                               
// Storage container for the calibrated Quantum Efficiency of the whole camera.
//
// For a complete description of the quantum efficiency calibration process, 
// see MCalibrationQEPix.
//
// Individual pixels have to be cast when retrieved e.g.:
// MCalibrationQEPix &avpix = (MCalibrationQEPix&)(*fQECam)[i]
// 
// Averaged values over one whole area index (e.g. inner or outer pixels for 
// the MAGIC camera), can be retrieved via: 
// MCalibrationQEPix &avpix = (MCalibrationQEPix&)fQECam->GetAverageArea(i)
//
// Averaged values over one whole camera sector can be retrieved via: 
// MCalibrationQEPix &avpix = (MCalibrationQEPix&)fQECam->GetAverageSector(i)
//
// The following "calibration" constants are used for the calibration of each pixel:
//
// - MCalibrationQEPix::GetQECascadesBlindPixel(): The mean quantum efficiency folded 
//   into a cascades spectrum obtained with the Blind Pixel Method. 
// - MCalibrationQEPix::GetQECascadesFFactor(): The mean quantum efficiency folded 
//   into a cascades spectrum obtained with the F-Factor Method
// - MCalibrationQEPix::GetQECascadesPINDiode(): The mean quantum efficiency folded 
//   into a cascades spectrum obtained with the PIN Diode Method
// - MCalibrationQEPix::GetQECascadesCombined(): The mean quantum efficiency folded 
//   into a cascades spectrum obtained with the combination of the three methods
//
// The following "calibration" constants have been measured to obtain the above values:
//
// - MCalibrationQEPix::GetQEBlindPixel( MCalibrationCam::PulserColor_t color ): 
//   The mean quantum efficiency obtained with the calibration pulser color 
//   (e.g. kGREEN, kBLUE, kUV, kCT1) after the Blind Pixel Method
// - MCalibrationQEPix::GetQEFFactor( MCalibrationCam::PulserColor_t color ): 
//   The mean quantum efficiency obtained with the calibration pulser color
//   (e.g. kGREEN, kBLUE, kUV, kCT1) after the F-Factor Method
// - MCalibrationQEPix::GetQEPINDiode( MCalibrationCam::PulserColor_t color ): 
//   The mean quantum efficiency obtained with the calibration pulser color 
//   (e.g. kGREEN, kBLUE, kUV, kCT1) after the PIN Diode Method
// - MCalibrationQEPix::GetQECombined( MCalibrationCam::PulserColor_t color ): 
//   The mean quantum efficiency obtained with the calibration pulser color 
//   (e.g. kGREEN, kBLUE, kUV, kCT1) after the combination of the three methods
//
// See also: MCalibrationQEPix, MCalibrationChargeCam, MCalibrationChargeCalc
//           MCalibrationChargeBlindPix, MCalibrationChargePINDiode, MCalibrationChargePix
//
//                                                                         
// The calculated values (types of GetPixelContent()) are:
// 
//
// Types used by MCalibrate and MCalibrateData: 
// ============================================
//
// 0: Mean Quantum Efficiency for cascades, obtained with the F-Factor method
// 1: Error of the Mean QE for cascades, obtained with the F-Factor method
// 2: Mean Quantum Efficiency for cascades, obtained with the Blind Pixel method
// 3: Error of the Mean QE for cascades, obtained with the Blind Pixel method
// 4: Mean Quantum Efficiency for cascades, obtained with the PIN Diode method
// 5: Error of the Mean QE for cascades, obtained with the PIN Diode method
// 6: Mean Quantum Efficiency for cascades, obtained with combination of the 3 methods
// 7: Error of the Mean QE for cascades, obtained with combination of the 3 methods
//
// Types filled by MCalibrationChargeCalc in combination of MCalibrationChargePix:
// ===============================================================================
//
// 8:  Mean Quantum Efficiency obtained with F-Factor Method ( color: kCT1)
// 9:  Error of the Mean QE obtained with F-Factor Method ( color: kCT1)
// 10: Mean Quantum Efficiency obtained with F-Factor Method ( color: kGREEN)
// 11: Error of the Mean QE obtained with F-Factor Method ( color: kGREEN)
// 12: Mean Quantum Efficiency obtained with F-Factor Method ( color: kBLUE)
// 13: Error of the Mean QE obtained with F-Factor Method ( color: kBLUE)
// 14: Mean Quantum Efficiency obtained with F-Factor Method ( color: kUV)
// 15: Error of the Mean QE obtained with F-Factor Method ( color: kUV)
//
// Types filled by MCalibrationChargeCalc in combination of MCalibrationChargeBlindPix:
// ====================================================================================
//
// 16: Mean Quantum Efficiency obtained with Blind Pixel Method ( color: kCT1)
// 17: Error of the Mean QE obtained with Blind Pixel Method ( color: kCT1)
// 18: Mean Quantum Efficiency obtained with Blind Pixel Method ( color: kGREEN)
// 19: Error of the Mean QE obtained with Blind Pixel Method ( color: kGREEN)
// 20: Mean Quantum Efficiency obtained with Blind Pixel Method ( color: kBLUE)
// 21: Error of the Mean QE obtained with Blind Pixel Method ( color: kBLUE)
// 22: Mean Quantum Efficiency obtained with Blind Pixel Method ( color: kUV)
// 23: Error of the Mean QE obtained with Blind Pixel Method ( color: kUV)
//
// Types filled by MCalibrationChargeCalc in combination of MCalibrationChargePINDiode:
// ====================================================================================
//
// 24: Mean Quantum Efficiency obtained with PIN Diode Method ( color: kCT1)
// 25: Error of the Mean QE obtained with PIN Diode Method ( color: kCT1)
// 26: Mean Quantum Efficiency obtained with PIN Diode Method ( color: kGREEN)
// 27: Error of the Mean QE obtained with PIN Diode Method ( color: kGREEN)
// 28: Mean Quantum Efficiency obtained with PIN Diode Method ( color: kBLUE)
// 29: Error of the Mean QE obtained with PIN Diode Method ( color: kBLUE)
// 30: Mean Quantum Efficiency obtained with PIN Diode Method ( color: kUV)
// 31: Error of the Mean QE obtained with PIN Diode Method ( color: kUV)
//
// Types filled by MCalibrationChargeCalc in combination of MCalibrationQEPix:
// ===========================================================================
//
// 24: Mean Quantum Efficiency obtained with combination of 3 methods ( color: kCT1)
// 25: Error of the Mean QE obtained with combination of 3 methods ( color: kCT1)
// 26: Mean Quantum Efficiency obtained with combination of 3 methods ( color: kGREEN)
// 27: Error of the Mean QE obtained with combination of 3 methods ( color: kGREEN)
// 28: Mean Quantum Efficiency obtained with combination of 3 methods ( color: kBLUE)
// 29: Error of the Mean QE obtained with combination of 3 methods ( color: kBLUE)
// 30: Mean Quantum Efficiency obtained with combination of 3 methods ( color: kUV)
// 31: Error of the Mean QE obtained with combination of 3 methods ( color: kUV)
//
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationQECam.h"
#include "MCalibrationCam.h"

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MCalibrationQEPix.h"

ClassImp(MCalibrationQECam);

using namespace std;

const Float_t MCalibrationQECam::gkPlexiglassQE         = 0.96;
const Float_t MCalibrationQECam::gkPlexiglassQEErr      = 0.01;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Creates a TClonesArray of MCalibrationQEPix containers, initialized to 1 entry, destinated 
// to hold one container per pixel. Later, a call to MCalibrationQECam::InitSize() 
// has to be performed (in MGeomApply). 
//
// Creates a TClonesArray of MCalibrationQEPix containers, initialized to 1 entry, destinated 
// to hold one container per pixel AREA. Later, a call to MCalibrationQECam::InitAreas() 
// has to be performed (in MGeomApply). 
//
// Creates a TClonesArray of MCalibrationQEPix containers, initialized to 1 entry, destinated
// to hold one container per camera SECTOR. Later, a call to MCalibrationQECam::InitSectors() 
// has to be performed (in MGeomApply). 
//
MCalibrationQECam::MCalibrationQECam(const char *name, const char *title)
    : fFlags(MCalibrationCam::gkNumPulserColors)
{
    fName  = name  ? name  : "MCalibrationQECam";
    fTitle = title ? title : "Storage container for the calibrated Quantrum Efficiency of the camera";

    fPixels           = new TClonesArray("MCalibrationQEPix",1);
    fAverageAreas     = new TClonesArray("MCalibrationQEPix",1);
    fAverageSectors   = new TClonesArray("MCalibrationQEPix",1);

    Clear();
}

// ------------------------------------------------------------------------
//
// Sets all bits to kFALSE
// 
// Calls:
// - MCalibrationCam::Clear()
//
void MCalibrationQECam::Clear(Option_t *o)
{

  SetBlindPixelMethodValid ( kFALSE, MCalibrationCam::kGREEN);
  SetFFactorMethodValid    ( kFALSE, MCalibrationCam::kGREEN);
  SetCombinedMethodValid   ( kFALSE, MCalibrationCam::kGREEN);
  SetPINDiodeMethodValid   ( kFALSE, MCalibrationCam::kGREEN);
  SetBlindPixelMethodValid ( kFALSE, MCalibrationCam::kBLUE);
  SetFFactorMethodValid    ( kFALSE, MCalibrationCam::kBLUE);
  SetCombinedMethodValid   ( kFALSE, MCalibrationCam::kBLUE);
  SetPINDiodeMethodValid   ( kFALSE, MCalibrationCam::kBLUE);
  SetBlindPixelMethodValid ( kFALSE, MCalibrationCam::kUV);
  SetFFactorMethodValid    ( kFALSE, MCalibrationCam::kUV);
  SetCombinedMethodValid   ( kFALSE, MCalibrationCam::kUV);
  SetPINDiodeMethodValid   ( kFALSE, MCalibrationCam::kUV);
  SetBlindPixelMethodValid ( kFALSE, MCalibrationCam::kCT1);
  SetFFactorMethodValid    ( kFALSE, MCalibrationCam::kCT1);
  SetCombinedMethodValid   ( kFALSE, MCalibrationCam::kCT1);
  SetPINDiodeMethodValid   ( kFALSE, MCalibrationCam::kCT1);

  MCalibrationCam::Clear();
}


// --------------------------------------------------------------------------
//
// Not yet implemented
//
void MCalibrationQECam::DrawPixelContent(Int_t idx) const
{
  return;
}

// --------------------------------------------------------------------
//
// Types used by MCalibrate and MCalibrateData: 
// ============================================
//
// 0: Mean Quantum Efficiency for cascades, obtained with the F-Factor method
// 1: Error of the Mean QE for cascades, obtained with the F-Factor method
// 2: Mean Quantum Efficiency for cascades, obtained with the Blind Pixel method
// 3: Error of the Mean QE for cascades, obtained with the Blind Pixel method
// 4: Mean Quantum Efficiency for cascades, obtained with the PIN Diode method
// 5: Error of the Mean QE for cascades, obtained with the PIN Diode method
// 6: Mean Quantum Efficiency for cascades, obtained with combination of the 3 methods
// 7: Error of the Mean QE for cascades, obtained with combination of the 3 methods
//
// Types filled by MCalibrationChargeCalc in combination of MCalibrationChargePix:
// ===============================================================================
//
// 8:  Mean Quantum Efficiency obtained with F-Factor Method ( color: kCT1)
// 9:  Error of the Mean QE obtained with F-Factor Method ( color: kCT1)
// 10: Mean Quantum Efficiency obtained with F-Factor Method ( color: kGREEN)
// 11: Error of the Mean QE obtained with F-Factor Method ( color: kGREEN)
// 12: Mean Quantum Efficiency obtained with F-Factor Method ( color: kBLUE)
// 13: Error of the Mean QE obtained with F-Factor Method ( color: kBLUE)
// 14: Mean Quantum Efficiency obtained with F-Factor Method ( color: kUV)
// 15: Error of the Mean QE obtained with F-Factor Method ( color: kUV)
//
// Types filled by MCalibrationChargeCalc in combination of MCalibrationChargeBlindPix:
// ====================================================================================
//
// 16: Mean Quantum Efficiency obtained with Blind Pixel Method ( color: kCT1)
// 17: Error of the Mean QE obtained with Blind Pixel Method ( color: kCT1)
// 18: Mean Quantum Efficiency obtained with Blind Pixel Method ( color: kGREEN)
// 19: Error of the Mean QE obtained with Blind Pixel Method ( color: kGREEN)
// 20: Mean Quantum Efficiency obtained with Blind Pixel Method ( color: kBLUE)
// 21: Error of the Mean QE obtained with Blind Pixel Method ( color: kBLUE)
// 22: Mean Quantum Efficiency obtained with Blind Pixel Method ( color: kUV)
// 23: Error of the Mean QE obtained with Blind Pixel Method ( color: kUV)
//
// Types filled by MCalibrationChargeCalc in combination of MCalibrationChargePINDiode:
// ====================================================================================
//
// 24: Mean Quantum Efficiency obtained with PIN Diode Method ( color: kCT1)
// 25: Error of the Mean QE obtained with PIN Diode Method ( color: kCT1)
// 26: Mean Quantum Efficiency obtained with PIN Diode Method ( color: kGREEN)
// 27: Error of the Mean QE obtained with PIN Diode Method ( color: kGREEN)
// 28: Mean Quantum Efficiency obtained with PIN Diode Method ( color: kBLUE)
// 29: Error of the Mean QE obtained with PIN Diode Method ( color: kBLUE)
// 30: Mean Quantum Efficiency obtained with PIN Diode Method ( color: kUV)
// 31: Error of the Mean QE obtained with PIN Diode Method ( color: kUV)
//
// Types filled by MCalibrationChargeCalc in combination of MCalibrationQEPix:
// ===========================================================================
//
// 24: Mean Quantum Efficiency obtained with combination of 3 methods ( color: kCT1)
// 25: Error of the Mean QE obtained with combination of 3 methods ( color: kCT1)
// 26: Mean Quantum Efficiency obtained with combination of 3 methods ( color: kGREEN)
// 27: Error of the Mean QE obtained with combination of 3 methods ( color: kGREEN)
// 28: Mean Quantum Efficiency obtained with combination of 3 methods ( color: kBLUE)
// 29: Error of the Mean QE obtained with combination of 3 methods ( color: kBLUE)
// 30: Mean Quantum Efficiency obtained with combination of 3 methods ( color: kUV)
// 31: Error of the Mean QE obtained with combination of 3 methods ( color: kUV)
//
Bool_t MCalibrationQECam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  if (idx > GetSize())
    return kFALSE;

  MCalibrationQEPix &pix = (MCalibrationQEPix&)(*this)[idx];

  if (pix.IsExcluded())
    return kFALSE;

  switch (type)
    {
    case 0:
      val = pix.GetQECascadesFFactor();
      break;
    case 1:
      val = pix.GetQECascadesFFactorErr();
      break;
    case 2:
      val = pix.GetQECascadesBlindPixel();
      break;
    case 3:
      val = pix.GetQECascadesBlindPixelErr();
      break;
    case 4:
      val = pix.GetQECascadesPINDiode();
      break;
    case 5:
      val = pix.GetQECascadesPINDiodeErr();
      break;
    case 6:
      val = pix.GetQECascadesCombined();
      break;
    case 7:
      val = pix.GetQECascadesCombinedErr();
      break;
    case 8:
      val = pix.GetQEFFactor(kCT1);
      break;
    case 9:
      val = pix.GetQEFFactorErr(kCT1);
      break;
    case 10:
      val = pix.GetQEFFactor(kGREEN);
      break;
    case 11:
      val = pix.GetQEFFactorErr(kGREEN);
      break;
    case 12:
      val = pix.GetQEFFactor(kBLUE);
      break;
    case 13:
      val = pix.GetQEFFactorErr(kBLUE);
      break;
    case 14:
      val = pix.GetQEFFactor(kUV);
      break;
    case 15:
      val = pix.GetQEFFactorErr(kUV);
      break;
    case 16:
      val = pix.GetQEBlindPixel(kCT1);
      break;
    case 17:
      val = pix.GetQEBlindPixelErr(kCT1);
      break;
    case 18:
      val = pix.GetQEBlindPixel(kGREEN);
      break;
    case 19:
      val = pix.GetQEBlindPixelErr(kGREEN);
      break;
    case 20:
      val = pix.GetQEBlindPixel(kBLUE);
      break;
    case 21:
      val = pix.GetQEBlindPixelErr(kBLUE);
      break;
    case 22:
      val = pix.GetQEBlindPixel(kUV);
      break;
    case 23:
      val = pix.GetQEBlindPixelErr(kUV);
      break;
    case 24:
      val = pix.GetQEPINDiode(kCT1);
      break;
    case 25:
      val = pix.GetQEPINDiodeErr(kCT1);
      break;
    case 26:
      val = pix.GetQEPINDiode(kGREEN);
      break;
    case 27:
      val = pix.GetQEPINDiodeErr(kGREEN);
      break;
    case 28:
      val = pix.GetQEPINDiode(kBLUE);
      break;
    case 29:
      val = pix.GetQEPINDiodeErr(kBLUE);
      break;
    case 30:
      val = pix.GetQEPINDiode(kUV);
      break;
    case 31:
      val = pix.GetQEPINDiodeErr(kUV);
      break;
    case 32:
      val = pix.GetQECombined(kCT1);
      break;
    case 33:
      val = pix.GetQECombinedErr(kCT1);
      break;
    case 34:
      val = pix.GetQECombined(kGREEN);
      break;
    case 35:
      val = pix.GetQECombinedErr(kGREEN);
      break;
    case 36:
      val = pix.GetQECombined(kBLUE);
      break;
    case 37:
      val = pix.GetQECombinedErr(kBLUE);
      break;
    case 38:
      val = pix.GetQECombined(kUV);
      break;
    case 39:
      val = pix.GetQECombinedErr(kUV);
      break;
    default:
      return kFALSE;
    }
  return val!=-1.;
}

// --------------------------------------------------------------------------
//
// Return -1 if gkPlexiglassQEErr is smaller than 0.
// Return -1 if gkPlexiglassQE    is 0.
// Return gkPlexiglassQEErr^2 / gkPlexiglassQE^2
// 
Float_t MCalibrationQECam::GetPlexiglassQERelVar() const
{
  if (gkPlexiglassQEErr < 0.)
    return -1.;
  
  if (gkPlexiglassQE  == 0.)
    return -1.;
  
  return gkPlexiglassQEErr * gkPlexiglassQEErr / gkPlexiglassQE / gkPlexiglassQE ;
}


// --------------------------------------------------------------------------------
//
// Returns kTRUE if ANY of the four colours have the bit kBlindPixelMethodValid set,
// otherwise kFALSE
// 
Bool_t MCalibrationQECam::IsBlindPixelMethodValid () const  
{
  if (IsBlindPixelMethodValid (MCalibrationCam::kGREEN))
    return kTRUE;
  if (IsBlindPixelMethodValid (MCalibrationCam::kBLUE ))
    return kTRUE;
  if (IsBlindPixelMethodValid (MCalibrationCam::kUV   ))
    return kTRUE;
  if (IsBlindPixelMethodValid (MCalibrationCam::kCT1  ))
    return kTRUE;

  return kFALSE;
}

// --------------------------------------------------------------------------------
//
// Returns kTRUE if ANY of the four colours have the bit kCombinedMethodValid set,
// otherwise kFALSE
// 
Bool_t MCalibrationQECam::IsCombinedMethodValid () const
{
  if (IsCombinedMethodValid (MCalibrationCam::kGREEN))
    return kTRUE;
  if (IsCombinedMethodValid (MCalibrationCam::kBLUE ))
    return kTRUE;
  if (IsCombinedMethodValid (MCalibrationCam::kUV   ))
    return kTRUE;
  if (IsCombinedMethodValid (MCalibrationCam::kCT1  ))
    return kTRUE;

  return kFALSE;
}

// --------------------------------------------------------------------------------
//
// Returns kTRUE if ANY of the four colours have the bit kFFactorMethodValid set,
// otherwise kFALSE
// 
Bool_t MCalibrationQECam::IsFFactorMethodValid () const
{
  if (IsFFactorMethodValid (MCalibrationCam::kGREEN))
    return kTRUE;
  if (IsFFactorMethodValid (MCalibrationCam::kBLUE ))
    return kTRUE;
  if (IsFFactorMethodValid (MCalibrationCam::kUV   ))
    return kTRUE;
  if (IsFFactorMethodValid (MCalibrationCam::kCT1  ))
    return kTRUE;

  return kFALSE;
}


// --------------------------------------------------------------------------------
//
// Returns kTRUE if ANY of the four colours have the bit kPINDiodeMethodValid set,
// otherwise kFALSE
// 
Bool_t MCalibrationQECam::IsPINDiodeMethodValid () const
{
  if (IsPINDiodeMethodValid (MCalibrationCam::kGREEN))
    return kTRUE;
  if (IsPINDiodeMethodValid (MCalibrationCam::kBLUE ))
    return kTRUE;
  if (IsPINDiodeMethodValid (MCalibrationCam::kUV   ))
    return kTRUE;
  if (IsPINDiodeMethodValid (MCalibrationCam::kCT1  ))
    return kTRUE;

  return kFALSE;
}

// --------------------------------------------------------------------------------
//
// Returns kTRUE if ANY of the bit kBlindPixelMethodValid is set for colour "col"
// otherwise kFALSE
// 
Bool_t MCalibrationQECam::IsBlindPixelMethodValid (MCalibrationCam::PulserColor_t col)  const
{
  return TESTBIT(fFlags[ MCalibrationCam::kGREEN ],kBlindPixelMethodValid);
}

// --------------------------------------------------------------------------------
//
// Returns kTRUE if ANY of the bit kCombinedMethodValid is set for colour "col"
// otherwise kFALSE
// 
Bool_t MCalibrationQECam::IsCombinedMethodValid (MCalibrationCam::PulserColor_t col)  const
{
  return TESTBIT(fFlags[ MCalibrationCam::kGREEN ],kCombinedMethodValid);
}

// --------------------------------------------------------------------------------
//
// Returns kTRUE if ANY of the bit kFFactorMethodValid is set for colour "col"
// otherwise kFALSE
// 
Bool_t MCalibrationQECam::IsFFactorMethodValid (MCalibrationCam::PulserColor_t col)  const
{
  return TESTBIT(fFlags[ MCalibrationCam::kGREEN ],kFFactorMethodValid);
}

// --------------------------------------------------------------------------------
//
// Returns kTRUE if ANY of the bit kPINDiodeMethodValid is set for colour "col"
// otherwise kFALSE
// 
Bool_t MCalibrationQECam::IsPINDiodeMethodValid (MCalibrationCam::PulserColor_t col)  const
{
  return TESTBIT(fFlags[ MCalibrationCam::kGREEN ],kPINDiodeMethodValid);
}

// --------------------------------------------------------------------------
//
// Print the 
// - MCalibrationQEPix::GetQECascadesFFactor() 
// - MCalibrationQEPix::GetQECascadesBlindPixel() 
// - MCalibrationQEPix::GetQECascadesPINDiode() 
// - MCalibrationQEPix::GetQECascadesCombined()
// for all pixels 
//
void MCalibrationQECam::Print(Option_t *o) const
{

  *fLog << all << GetDescriptor() << ":" << endl;
  int id = 0;
  
  *fLog << all << endl;
  *fLog << all << "Quantum Efficiencies averaged over cascades spectra, measured with F-Factor method:" << endl;
  *fLog << all << endl;

  TIter Next(fPixels);
  MCalibrationQEPix *pix;
  while ((pix=(MCalibrationQEPix*)Next()))
    {
      
      if (!pix->IsExcluded() && pix->IsAverageQEFFactorAvailable()) 
	{
          *fLog << all 
                << Form("%s%4i%s%4.2f%s%4.2f","Pix  ",pix->GetPixId(),
                        ":  QE: ",pix->GetQECascadesFFactor()," +- ",pix->GetQECascadesFFactorErr())
		<< endl;
          id++;
	}
    }
  
  *fLog << all << id << " succesful pixels :-))" << endl;
  id = 0;
  
  *fLog << all << endl;
  *fLog << all << "Quantum Efficiencies averaged over cascades spectra, " 
        << "measured with Blind Pixel method:" << endl;
  *fLog << all << endl;
  
  TIter Next2(fPixels);
    while ((pix=(MCalibrationQEPix*)Next2()))
      {
        
      if (!pix->IsExcluded() && pix->IsAverageQEBlindPixelAvailable()) 
	{
          *fLog << all 
                << Form("%s%4i%s%4.2f%s%4.2f","Pix  ",pix->GetPixId(),
                        ":  QE: ",pix->GetQECascadesBlindPixel()," +- ",pix->GetQECascadesBlindPixelErr())
		<< endl;
          id++;
	}
      }

  *fLog << all << id << " succesful pixels :-))" << endl;
  id = 0;
  
  *fLog << all << endl;
  *fLog << all << "Quantum Efficiencies averaged over cascades spectra, " 
        << "measured with PIN Diode method:" << endl;
  *fLog << all << endl;
  
  TIter Next3(fPixels);
    while ((pix=(MCalibrationQEPix*)Next3()))
      {
        
      if (!pix->IsExcluded() && pix->IsAverageQEPINDiodeAvailable()) 
	{
          *fLog << all 
                << Form("%s%4i%s%4.2f%s%4.2f","Pix  ",pix->GetPixId(),
                        ":  QE: ",pix->GetQECascadesPINDiode()," +- ",pix->GetQECascadesPINDiodeErr())
		<< endl;
          id++;
	}
      }

  *fLog << all << id << " succesful pixels :-))" << endl;
  id = 0;

    
  *fLog << all << endl;
  *fLog << all << "Quantum Efficiencies averaged over cascades spectra, " 
        << "measured with combination of the 3 methods:" << endl;
  *fLog << all << endl;
  
  TIter Next4(fPixels);
    while ((pix=(MCalibrationQEPix*)Next4()))
      {
        
      if (!pix->IsExcluded() && pix->IsAverageQECombinedAvailable()) 
	{
          *fLog << all 
                << Form("%s%4i%s%4.2f%s%4.2f","Pix  ",pix->GetPixId(),
                        ":  QE: ",pix->GetQECascadesCombined()," +- ",pix->GetQECascadesCombinedErr())
		<< endl;
          id++;
	}
      }

  *fLog << all << id << " succesful pixels :-))" << endl;
  id = 0;

  *fLog << all << endl;
  *fLog << all << "Excluded pixels:" << endl;
  *fLog << all << endl;
  
  TIter Next5(fPixels);
  while ((pix=(MCalibrationQEPix*)Next5()))
  {
      if (pix->IsExcluded())
      {
	  *fLog << all << pix->GetPixId() << endl;
	  id++;
      }
  }
  *fLog << all << id << " Excluded pixels " << endl;
}


// --------------------------------------------------------------------------
//
// Sets the validity flag (according to b) for the Blind Pixel Method, 
// for all colours (kGREEN, kBLUE, kUV, kCT1)
// 
void MCalibrationQECam::SetBlindPixelMethodValid ( const Bool_t b ) 
{
  SetBlindPixelMethodValid ( b, MCalibrationCam::kGREEN);
  SetBlindPixelMethodValid ( b, MCalibrationCam::kBLUE );
  SetBlindPixelMethodValid ( b, MCalibrationCam::kUV   );
  SetBlindPixelMethodValid ( b, MCalibrationCam::kCT1  );
}

// ----------------------------------------------------------------------------
//
// Sets the validity flag (according to b) for the combination of the 3 methods
// for all colours (kGREEN, kBLUE, kUV, kCT1)
// 
void MCalibrationQECam::SetCombinedMethodValid ( const Bool_t b ) 
{
  SetCombinedMethodValid ( b, MCalibrationCam::kGREEN);
  SetCombinedMethodValid ( b, MCalibrationCam::kBLUE );
  SetCombinedMethodValid ( b, MCalibrationCam::kUV   );
  SetCombinedMethodValid ( b, MCalibrationCam::kCT1  );
}

// --------------------------------------------------------------------------
//
// Sets the validity flag (according to b) for the F-Factor Method
// for all colours (kGREEN, kBLUE, kUV, kCT1)
// 
void MCalibrationQECam::SetFFactorMethodValid ( const Bool_t b ) 
{
  SetFFactorMethodValid ( b, MCalibrationCam::kGREEN);
  SetFFactorMethodValid ( b, MCalibrationCam::kBLUE );
  SetFFactorMethodValid ( b, MCalibrationCam::kUV   );
  SetFFactorMethodValid ( b, MCalibrationCam::kCT1  );
}

// --------------------------------------------------------------------------
//
// Sets the validity flag (according to b) for the PIN Diode Method, 
// for all colours (kGREEN, kBLUE, kUV, kCT1)
// 
void MCalibrationQECam::SetPINDiodeMethodValid ( const Bool_t b ) 
{
  SetPINDiodeMethodValid ( b, MCalibrationCam::kGREEN);
  SetPINDiodeMethodValid ( b, MCalibrationCam::kBLUE );
  SetPINDiodeMethodValid ( b, MCalibrationCam::kUV   );
  SetPINDiodeMethodValid ( b, MCalibrationCam::kCT1  );
}

// --------------------------------------------------------------------------
//
// Sets the validity flag (according to b) for the Blind Pixel Method, 
// for colour "col"
// 
void MCalibrationQECam::SetBlindPixelMethodValid ( const Bool_t b,  MCalibrationCam::PulserColor_t col ) 
{
  if (b) 
    SETBIT(fFlags[ MCalibrationCam::kGREEN ],kBlindPixelMethodValid);
  else
    CLRBIT(fFlags[ MCalibrationCam::kGREEN ],kBlindPixelMethodValid);    
}

// --------------------------------------------------------------------------
//
// Sets the validity flag (according to b) for the combination of 3 methods
// for colour "col"
// 
void MCalibrationQECam::SetCombinedMethodValid ( const Bool_t b,  MCalibrationCam::PulserColor_t col ) 
{
  if (b) 
    SETBIT(fFlags[ MCalibrationCam::kGREEN ],kCombinedMethodValid);
  else
    CLRBIT(fFlags[ MCalibrationCam::kGREEN ],kCombinedMethodValid);    
}

// --------------------------------------------------------------------------
//
// Sets the validity flag (according to b) for the F-Factor Method, 
// for colour "col"
// 
void MCalibrationQECam::SetFFactorMethodValid ( const Bool_t b,  MCalibrationCam::PulserColor_t col ) 
{
  if (b) 
    SETBIT(fFlags[ MCalibrationCam::kGREEN ],kFFactorMethodValid);
  else
    CLRBIT(fFlags[ MCalibrationCam::kGREEN ],kFFactorMethodValid);    
}

// --------------------------------------------------------------------------
//
// Sets the validity flag (according to b) for the PIN Diode Method, 
// for colour "col"
// 
void MCalibrationQECam::SetPINDiodeMethodValid ( const Bool_t b,  MCalibrationCam::PulserColor_t col ) 
{
  if (b) 
    SETBIT(fFlags[ MCalibrationCam::kGREEN ],kPINDiodeMethodValid);
  else
    CLRBIT(fFlags[ MCalibrationCam::kGREEN ],kPINDiodeMethodValid);    
}








