/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Josep Flix 04/2001 <mailto:jflix@ifae.es>
!   Author(s): Thomas Bretz 05/2001 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Sebastian Commichau 12/2003 
!   Author(s): Javier Rico 01/2004 <mailto:jrico@ifae.es>
!   Author(s): Markus Gaug 01/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//   MPedCalcPedRun
//
// This task takes a pedestal run file and fills MPedestalCam during
// the Process() with the pedestal and rms computed in an event basis.
// In the PostProcess() MPedestalCam is finally filled with the pedestal
// mean and rms computed in a run basis.
// More than one run (file) can be merged
//
//
// Actually, MPedCalcPedRun applies the following formula (1):
// 
// PedRMS = Sqrt(  (sum(x_i^2) - sum(x_i)^2/n) / n-1 / 14 )
// 
// where x_i is the sum of 14 FADC slices and sum means the sum over all
// events, n is the number of events.
// 
// For a high number of events, this formula is equivalent to formula (2):
// 
// PedRMS = Sqrt(  (<x_i*x_i> - <x_i>*<x_i>) / 14  )
// 
// where <> is the mean over all events and x_i again the sum over the 14
// slices.
// 
// If you assume statistical equivalence of all slices (say, all have equal
// offset and are not correlated and fluctuate Gaussian), it should also be
// equivalent to (old formula) (3):
// 
// PedRMS = Sqrt(  (<p_i*p_i> - <p_i>*<p_i>) ) 
// 
// which is the RMS per slice of a single slice (p_i) and 
// <> the mean over the total number of measurements, i.e. n*14.
// 
// If we assume that at least our pairs fluctuate independently and Gaussian,
// then we can use the actual formula (1) in order to get 
// fluctuations of pairs by the transformation:
// 
// PedRMS/pair = PedRMS (form. (3)) * Sqrt(2)
// 
// (However, we know that our slice-to-slice fluctuations are not Gaussian
// (and moreover asymmetric) and that they are also correlated.)
// 
// 
//  Input Containers:
//   MRawEvtData
//
//  Output Containers:
//   MPedestalCam
//
//
/////////////////////////////////////////////////////////////////////////////
#include "MPedCalcPedRun.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MRawRunHeader.h"  
#include "MRawEvtPixelIter.h"
#include "MRawEvtData.h"

#include "MPedestalPix.h"
#include "MPedestalCam.h"

#include "MExtractedSignalPix.h"
#include "MExtractedSignalCam.h"

#include "MGeomPix.h"
#include "MGeomCam.h"

#include "MGeomCamMagic.h"

ClassImp(MPedCalcPedRun);

using namespace std;

// --------------------------------------------------------------------------
//
// default constructor
//
MPedCalcPedRun::MPedCalcPedRun(const char *name, const char *title)
    : fRawEvt(NULL), fPedestals(NULL)
{
    fName  = name  ? name  : "MPedCalcPedRun";
    fTitle = title ? title : "Task to calculate pedestals from pedestal runs raw data";

    AddToBranchList("fHiGainPixId");
    AddToBranchList("fHiGainFadcSamples");

    fNumHiGainSamples = 0;
    Clear();
}

void MPedCalcPedRun::Clear(const Option_t *o)
{

  fNumSamplesTot    = 0;

  fRawEvt    = NULL;
  fPedestals = NULL;
}


// --------------------------------------------------------------------------
//
// Look for the following input containers:
//
//  - MRawEvtData
// 
// The following output containers are also searched and created if
// they were not found:
//
//  - MPedestalCam
//
Int_t MPedCalcPedRun::PreProcess( MParList *pList )
{

  Clear();
  
  fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
  if (!fRawEvt)
    {
      *fLog << err << "MRawEvtData not found... aborting." << endl;
      return kFALSE;
    }
  
  fGeom   =  (MGeomCam*)pList->FindObject("MGeomCam");
  if (!fGeom)
    {
      *fLog << err << "MGeomCam not found... aborting." << endl;
      return kFALSE;
    }

  fPedestals = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
  if (!fPedestals)
    return kFALSE;
  
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// The ReInit searches for the following input containers:
//  - MRawRunHeader
//
// It also initializes the data arrays fSumx and fSumx2 
// (only for the first read file)
// 
Bool_t MPedCalcPedRun::ReInit(MParList *pList)
{
    const MRawRunHeader *runheader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!runheader)
    {
        *fLog << warn << dbginf;
        *fLog << "Warning - cannot check file type, MRawRunHeader not found." << endl;
    }
    else
      if (runheader->IsMonteCarloRun())
        return kTRUE;

    Int_t npixels  = fPedestals->GetSize();
    Int_t areas    = fPedestals->GetAverageAreas();
    Int_t sectors  = fPedestals->GetAverageSectors();

    if (fSumx.GetSize()==0)
    {
	fSumx. Set(npixels);
	fSumx2.Set(npixels);

	fAreaSumx. Set(areas);
	fAreaSumx2.Set(areas);
	fAreaValid.Set(areas);

	fSectorSumx. Set(sectors);
	fSectorSumx2.Set(sectors);
	fSectorValid.Set(sectors);

	fSumx.Reset();
	fSumx2.Reset();
    }

    // Calculate an even number for the hi gain samples to avoid
    // biases due to the fluctuation in pedestal from one slice to
    // the other one
    fNumHiGainSamples = runheader->GetNumSamplesHiGain() & ~1;

    return kTRUE;
}
// --------------------------------------------------------------------------
//
// Fill the MPedestalCam container with the signal mean and rms for the event.
// Store the measured signal in arrays fSumx and fSumx2 so that we can 
// calculate the overall mean and rms in the PostProcess()
//
Int_t MPedCalcPedRun::Process()
{

  MRawEvtPixelIter pixel(fRawEvt);
  
  while (pixel.Next())
    {

      const UInt_t idx    = pixel.GetPixelId();
      const UInt_t aidx   = (*fGeom)[idx].GetAidx();
      const UInt_t sector = (*fGeom)[idx].GetSector();      

      Byte_t *ptr = pixel.GetHiGainSamples();
      const Byte_t *end = ptr + fNumHiGainSamples;
      
      UInt_t sum = 0;
      UInt_t sqr = 0;

      do
        {
          sum += *ptr;
          sqr += *ptr * *ptr;
        }
      while (++ptr != end);
      
      const Float_t msum = (Float_t)sum;
      
      //
      // These three lines have been uncommented by Markus Gaug
      // If anybody needs them, please contact me!!
      //
      //	const Float_t higainped = msum/fNumHiGainSamples;
      //	const Float_t higainrms = TMath::Sqrt((msqr-msum*msum/fNumHiGainSamples)/(fNumHiGainSamples-1.));
      //	(*fPedestals)[idx].Set(higainped, higainrms);
      
      fSumx[idx]          += msum;
      fAreaSumx[aidx]     += msum;
      fSectorSumx[sector] += msum;      
      //
      // The old version:
      //
      //       const Float_t msqr = (Float_t)sqr;
      //	fSumx2[idx] += msqr;
      //
      // The new version:
      //
      const Float_t sqrsum  = msum*msum;
      fSumx2[idx]          += sqrsum;
      fAreaSumx2[aidx]     += sqrsum;
      fSectorSumx2[sector] += sqrsum;      
    }
  
  fPedestals->SetReadyToSave();
  fNumSamplesTot += fNumHiGainSamples;
  
  
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Compute signal mean and rms in the whole run and store it in MPedestalCam
//
Int_t MPedCalcPedRun::PostProcess()
{

  // Compute pedestals and rms from the whole run
  const ULong_t n     = fNumSamplesTot;
  const ULong_t nevts = GetNumExecutions();

  MRawEvtPixelIter pixel(fRawEvt);
  
  while (pixel.Next())
    {

      const Int_t  pixid  = pixel.GetPixelId();
      const UInt_t aidx   = (*fGeom)[pixid].GetAidx();
      const UInt_t sector = (*fGeom)[pixid].GetSector();      
      
      fAreaValid  [aidx]++;
      fSectorValid[sector]++;

      const Float_t sum  = fSumx.At(pixid);
      const Float_t sum2 = fSumx2.At(pixid);
      
      const Float_t higainped = sum/n;
      //
      // The old version:
      //
      //      const Float_t higainrms = TMath::Sqrt((sum2-sum*sum/n)/(n-1.));
      //
      // The new version:
      //
      // 1. Calculate the Variance of the sums:
      Float_t higainVar = (sum2-sum*sum/nevts)/(nevts-1.);
      // 2. Scale the variance to the number of slices:
      higainVar /= (Float_t)fNumHiGainSamples;
      // 3. Calculate the RMS from the Variance:
      const Float_t higainrms = TMath::Sqrt(higainVar);

      (*fPedestals)[pixid].Set(higainped, higainrms);

    }

  //
  // Loop over the (two) area indices to get the averaged pedestal per aidx
  //
  for (Int_t aidx=0; aidx<fAreaValid.GetSize(); aidx++)
    {
      
      const Int_t   napix = fAreaValid.At(aidx);
      const Float_t sum   = fAreaSumx.At(aidx);
      const Float_t sum2  = fAreaSumx2.At(aidx);
      const ULong_t an    = napix * n;
      const ULong_t aevts = napix * nevts;
      
      const Float_t higainped = sum/an;

      // 1. Calculate the Variance of the sums:
      Float_t higainVar = (sum2-sum*sum/aevts)/(aevts-1.);
      // 2. Scale the variance to the number of slices:
      higainVar /= (Float_t)fNumHiGainSamples;
      // 3. Calculate the RMS from the Variance:
      Float_t higainrms = TMath::Sqrt(higainVar);
      // 4. Re-scale it with the square root of the number of involved pixels 
      //    in order to be comparable to the mean of pedRMS of that area
      higainrms *= TMath::Sqrt((Float_t)napix);

      fPedestals->GetAverageArea(aidx).Set(higainped, higainrms);
    }
  
  //
  // Loop over the (six) sector indices to get the averaged pedestal per sector
  //
  for (Int_t sector=0; sector<fSectorValid.GetSize(); sector++)
    {
      
      const Int_t   nspix = fSectorValid.At(sector);
      const Float_t sum   = fSectorSumx.At(sector);
      const Float_t sum2  = fSectorSumx2.At(sector);
      const ULong_t sn    = nspix * n;
      const ULong_t sevts = nspix * nevts;
      
      const Float_t higainped = sum/sn;

      // 1. Calculate the Variance of the sums:
      Float_t higainVar = (sum2-sum*sum/sevts)/(sevts-1.);
      // 2. Scale the variance to the number of slices:
      higainVar /= (Float_t)fNumHiGainSamples;
      // 3. Calculate the RMS from the Variance:
      Float_t higainrms = TMath::Sqrt(higainVar);
      // 4. Re-scale it with the square root of the number of involved pixels 
      //    in order to be comparable to the mean of pedRMS of that sector
      higainrms *= TMath::Sqrt((Float_t)nspix);

      fPedestals->GetAverageSector(sector).Set(higainped, higainrms);
    }
  
  fPedestals->SetTotalEntries(fNumSamplesTot);

  return kTRUE;
}


