/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 2001 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHHillasExt
//
// This class contains histograms for every Hillas parameter
//
/////////////////////////////////////////////////////////////////////////////
#include "MHHillasExt.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TLegend.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"

#include "MParList.h"

#include "MBinning.h"
#include "MHillasExt.h"
#include "MHillasSrc.h"

ClassImp(MHHillasExt);

using namespace std;

// --------------------------------------------------------------------------
//
// Setup four histograms for Width, Length
//
MHHillasExt::MHHillasExt(const char *name, const char *title)
    : fMm2Deg(1), fUseMmScale(kTRUE), fHilName("MHillasExt")
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHHillasExt";
    fTitle = title ? title : "Container for extended Hillas histograms";

    //
    // loop over all Pixels and create two histograms
    // one for the Low and one for the High gain
    // connect all the histogram with the container fHist
    //
    fHAsym.SetDirectory(NULL);
    fHM3Long.SetDirectory(NULL);
    fHM3Trans.SetDirectory(NULL);
    fHMaxDist.SetDirectory(NULL);

    fHAsym.UseCurrentStyle();
    fHM3Long.UseCurrentStyle();
    fHM3Trans.UseCurrentStyle();
    fHMaxDist.UseCurrentStyle();

    fHAsym.SetName("Asymmetry");
    fHM3Long.SetName("3rd Mom Long");
    fHM3Trans.SetName("3rd Mom Trans");
    fHMaxDist.SetName("Max Dist");

    fHAsym.SetTitle("Asymmetry");
    fHM3Long.SetTitle("3^{rd} Moment Longitudinal");
    fHM3Trans.SetTitle("3^{rd} Moment Transverse");
    fHMaxDist.SetTitle("Distance of max distant pixel");

    fHAsym.SetXTitle("Asym [mm]");
    fHM3Long.SetXTitle("3^{rd} M_{l} [mm]");
    fHM3Trans.SetXTitle("3^{rd} M_{t} [mm]");
    fHMaxDist.SetXTitle("D_{max} [mm]");

    fHAsym.SetYTitle("Counts");
    fHM3Long.SetYTitle("Counts");
    fHM3Trans.SetYTitle("Counts");
    fHMaxDist.SetYTitle("Counts");

    fHAsym.SetFillStyle(4000);
    fHM3Long.SetFillStyle(4000);
    fHM3Trans.SetFillStyle(4000);
    fHMaxDist.SetFillStyle(4000);

    fHM3Trans.SetLineColor(kBlue);

    MBinning bins;

    bins.SetEdges(51, -326, 326);
    bins.Apply(fHM3Long);
    bins.Apply(fHM3Trans);

    bins.SetEdges(51, -593, 593);
    bins.Apply(fHAsym);

    bins.SetEdges(101, 0, 593);
    bins.Apply(fHMaxDist);
}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning (with the names 'BinningWidth' and 'BinningLength')
// are found in the parameter list
// Use this function if you want to set the conversion factor which
// is used to convert the mm-scale in the camera plain into the deg-scale
// used for histogram presentations. The conversion factor is part of
// the camera geometry. Please create a corresponding MGeomCam container.
//
Bool_t MHHillasExt::SetupFill(const MParList *plist)
{
    fHillasExt = (MHillasExt*)plist->FindObject(fHilName, "MHillasExt");
    if (!fHillasExt)
    {
        *fLog << err << fHilName << "[MHillasExt] not found in parameter list... aborting." << endl;
        return kFALSE;
    }

    const MGeomCam *geom = (MGeomCam*)plist->FindObject("MGeomCam");
    if (!geom)
        *fLog << warn << GetDescriptor() << ": No Camera Geometry available. Using mm-scale for histograms." << endl;
    else
    {
        fMm2Deg = geom->GetConvMm2Deg();
        SetMmScale(kFALSE);
    }

    ApplyBinning(*plist, "Asym",    &fHAsym);
    ApplyBinning(*plist, "M3Long",  &fHM3Long);
    ApplyBinning(*plist, "M3Trans", &fHM3Trans);
    ApplyBinning(*plist, "MaxDist", &fHMaxDist);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the four histograms with data from a MHillas-Container.
// Be careful: Only call this with an object of type MHillas
//
Bool_t MHHillasExt::Fill(const MParContainer *par, const Stat_t w)
{
    const MHillasSrc *src = (MHillasSrc*)par;

    const Double_t scale = TMath::Sign(fUseMmScale?1:fMm2Deg, (src ? src->GetCosDeltaAlpha() : 1));

    fHAsym.Fill(scale*fHillasExt->GetAsym(), w);
    fHM3Long.Fill(scale*fHillasExt->GetM3Long(), w);
    fHM3Trans.Fill(scale*fHillasExt->GetM3Trans(), w);
    fHMaxDist.Fill(TMath::Abs(scale*fHillasExt->GetMaxDist()), w);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// With this function you can convert the histogram ('on the fly') between
// degrees and millimeters.
//
void MHHillasExt::SetMmScale(Bool_t mmscale)
{
    if (fUseMmScale == mmscale)
        return;

    if (fMm2Deg<0)
    {
        *fLog << warn << dbginf << "Warning - Sorry, no conversion factor for conversion available." << endl;
        return;
    }

    const Double_t scale = mmscale ? 1./fMm2Deg : fMm2Deg;
    MH::ScaleAxis(&fHAsym,    scale);
    MH::ScaleAxis(&fHM3Long,  scale);
    MH::ScaleAxis(&fHM3Trans, scale);
    MH::ScaleAxis(&fHMaxDist, scale);

    if (mmscale)
    {
        fHAsym.SetXTitle("Asym [mm]");
        fHM3Long.SetXTitle("3^{rd} M_{l} [mm]");
        fHM3Trans.SetXTitle("3^{rd} M_{t} [mm]");
        fHMaxDist.SetXTitle("D_{max} [mm]");
    }
    else
    {
        fHAsym.SetXTitle("Asym [\\circ]");
        fHM3Long.SetXTitle("3^{rd} M_{l} [\\circ]");
        fHM3Trans.SetXTitle("3^{rd} M_{t} [\\circ]");
        fHMaxDist.SetXTitle("D_{max} [\\circ]");
    }

    fUseMmScale = mmscale;
}

// --------------------------------------------------------------------------
//
// Use this function to setup your own conversion factor between degrees
// and millimeters. The conversion factor should be the one calculated in
// MGeomCam. Use this function with Caution: You could create wrong values
// by setting up your own scale factor.
//
void MHHillasExt::SetMm2Deg(Float_t mmdeg)
{
    if (mmdeg<0)
    {
        *fLog << warn << dbginf << "Warning - Conversion factor < 0 - nonsense. Ignored." << endl;
        return;
    }

    if (fMm2Deg>=0)
        *fLog << warn << dbginf << "Warning - Conversion factor already set. Overwriting" << endl;

    fMm2Deg = mmdeg;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the four histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHHillasExt::Draw(Option_t *)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->Divide(2, 2);

    pad->cd(1);
    gPad->SetBorderMode(0);
    MH::DrawSame(fHM3Long, fHM3Trans, "3^{rd} Moments");

    pad->cd(3);
    gPad->SetBorderMode(0);
    fHAsym.Draw();

    pad->cd(2);
    gPad->SetBorderMode(0);
    fHMaxDist.Draw();

    delete pad->GetPad(4);

    pad->Modified();
    pad->Update();
}

TH1 *MHHillasExt::GetHistByName(const TString name)
{
    if (name.Contains("Asym", TString::kIgnoreCase))
        return &fHAsym;
    if (name.Contains("M3Long", TString::kIgnoreCase))
        return &fHM3Long;
    if (name.Contains("M3Trans", TString::kIgnoreCase))
        return &fHM3Trans;
    if (name.Contains("MaxDist", TString::kIgnoreCase))
        return &fHMaxDist;

    return NULL;
}
