/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!              Markus Gaug ,04/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJPedestal
//
/////////////////////////////////////////////////////////////////////////////
#include "MJPedestal.h"

#include <TF1.h>
#include <TFile.h>
#include <TStyle.h>
#include <TString.h>
#include <TCanvas.h>
#include <TSystem.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MRunIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"
#include "MExtractor.h"

#include "MStatusDisplay.h"

#include "MGeomCam.h"
#include "MHCamera.h"
#include "MPedestalCam.h"
#include "MBadPixelsCam.h"

#include "MReadMarsFile.h"
#include "MRawFileRead.h"
#include "MGeomApply.h"
#include "MBadPixelsMerge.h"
#include "MPedCalcPedRun.h"

ClassImp(MJPedestal);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets fRuns to 0, fExtractor to NULL, fDataCheck to kFALSE
//
MJPedestal::MJPedestal(const char *name, const char *title) 
    : fRuns(0), fExtractor(NULL), fDataCheck(kFALSE)
{
    fName  = name  ? name  : "MJPedestal";
    fTitle = title ? title : "Tool to create a pedestal file (MPedestalCam)";
}

TString MJPedestal::GetOutputFile() const
{
    if (!fRuns)
        return "";

    return Form("%s/%s-F0.root", (const char*)fOutputPath, (const char*)fRuns->GetRunsAsFileName());
}

Bool_t MJPedestal::ReadPedestalCam()
{
    const TString fname = GetOutputFile();

    if (gSystem->AccessPathName(fname, kFileExists))
    {
        *fLog << err << "Input file " << fname << " doesn't exist." << endl;
        return kFALSE;
    }

    *fLog << inf << "Reading from file: " << fname << endl;

    TFile file(fname, "READ");
    if (fPedestalCam.Read()<=0)
    {
        *fLog << err << "Unable to read MPedestalCam from " << fname << endl;
        return kFALSE;
    }

    if (file.FindKey("MBadPixelsCam"))
    {
        MBadPixelsCam bad;
        if (bad.Read()<=0)
        {
            *fLog << err << "Unable to read MBadPixelsCam from " << fname << endl;
            return kFALSE;
        }
        fBadPixels.Merge(bad);
    }

    if (fDisplay && !fDisplay->GetCanvas("Pedestals"))
        fDisplay->Read();

    return kTRUE;
}


void MJPedestal::DisplayResult(MParList &plist)
{

    if (!fDisplay)
        return;

    //
    // Update display
    //
    TString title = fDisplay->GetTitle();
    title += "--  Pedestal ";
    title += fRuns->GetRunsAsString();
    title += "  --";
    fDisplay->SetTitle(title);

    //
    // Get container from list
    //
    MGeomCam &geomcam = *(MGeomCam*)plist.FindObject("MGeomCam");

    //
    // Create container to display
    //
    MHCamera disp0(geomcam, "MPedestalCam;ped", "Pedestal");
    MHCamera disp1(geomcam, "MPedestalCam;RMS", "Ped. RMS");

    disp0.SetCamContent(fPedestalCam, 0);
    disp0.SetCamError  (fPedestalCam, 1);

    disp1.SetCamContent(fPedestalCam, 2);
    disp1.SetCamError  (fPedestalCam, 3);

    disp0.SetYTitle("P [fadc/slice]");
    disp1.SetYTitle("\\sigma_{P} [fadc/slice]");

    //
    // Display data
    //
    TCanvas &c3 = fDisplay->AddTab("Pedestals");
    c3.Divide(2,3);

    CamDraw(c3, 1, 2, disp0, 1);
    CamDraw(c3, 2, 2, disp1, 6);
}

Bool_t MJPedestal::WriteResult()
{
    if (fOutputPath.IsNull())
        return kTRUE;

    const TString oname(GetOutputFile());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "UPDATE");

    if (fDisplay && fDisplay->Write()<=0)
    {
        *fLog << err << "Unable to write MStatusDisplay to " << oname << endl;
        return kFALSE;
    }

    if (fPedestalCam.Write()<=0)
    {
        *fLog << err << "Unable to write MPedestalCam to " << oname << endl;
        return kFALSE;
    }

    if (fBadPixels.Write()<=0)
    {
        *fLog << err << "Unable to write MBadPixelsCam to " << oname << endl;
        return kFALSE;
    }

    return kTRUE;
}

void MJPedestal::SetOutputPath(const char *path)
{
    fOutputPath = path;
    if (fOutputPath.EndsWith("/"))
        fOutputPath = fOutputPath(0, fOutputPath.Length()-1);
}

Bool_t MJPedestal::Process()
{
    if (!ReadPedestalCam())
        return ProcessFile();

    return kTRUE;
}

Bool_t MJPedestal::ProcessFile()
{
    if (!fRuns)
    {
        *fLog << err << "No Runs choosen... abort." << endl;
        return kFALSE;
    }
    if (fRuns->GetNumRuns() != fRuns->GetNumEntries())
    {
        *fLog << err << "Number of files found doesn't match number of runs... abort." << endl;
        return kFALSE;
    }

    *fLog << inf;
    fLog->Separator(GetDescriptor());
    *fLog << "Calculate MPedestalCam from Runs " << fRuns->GetRunsAsString() << endl;
    *fLog << endl;

    MParList plist;
    MTaskList tlist;
    plist.AddToList(&tlist);

    MReadMarsFile read("Events");
    MRawFileRead rawread("");

    if (fDataCheck)
    {
//	rawread.AddFiles(*fRuns);
	tlist.AddToList(&rawread);
    }
    else
    {
	read.DisableAutoScheme();
	static_cast<MRead&>(read).AddFiles(*fRuns);
	tlist.AddToList(&read);
    }
    // Enable logging to file
    //*fLog.SetOutputFile(lname, kTRUE);

    // Setup Tasklist
    plist.AddToList(&fPedestalCam);
    plist.AddToList(&fBadPixels);

    MGeomApply      geomapl;
    // MBadPixelsMerge merge(&fBadPixels);
    MPedCalcPedRun  pedcalc;

    if (fExtractor)
      {
        pedcalc.SetWindowSize((Int_t)fExtractor->GetNumHiGainSamples());
        pedcalc.SetRange(fExtractor->GetHiGainFirst(), fExtractor->GetHiGainLast());
      }
    else
      {
        *fLog << warn << GetDescriptor() 
            << ": No extractor has been chosen, take default number of FADC samples " << endl;
      }

    tlist.AddToList(&geomapl);
    //    tlist.AddToList(&merge);
    //tlist.AddToList(&sigcalc);
    tlist.AddToList(&pedcalc);

    //
    // Execute the eventloop
    //
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);

    // Execute first analysis
    if (!evtloop.Eventloop())
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    DisplayResult(plist);

    if (!WriteResult())
        return kFALSE;

    *fLog << inf << GetDescriptor() << ": Done." << endl;

    return kTRUE;
}
