/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!   Author(s): Javier Rico     04/2004 <mailto:jrico@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
// MSrcRotate
//
// Task to rotate the position of the source as a function of time
//
//  Input Containers:
//    MSrcPosCam
//
//  Output Containers:
//    MSrcPosCam
//    MDCA
//
//////////////////////////////////////////////////////////////////////////////

#include <fstream>
#include <math.h>

#include "MParList.h"
#include "MRawRunHeader.h"
#include "MRawEvtHeader.h"
#include "MSrcRotate.h"

#include "MAstroSky2Local.h"
#include "MObservatory.h"
#include "MSrcPosCam.h"
#include "MDCA.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MSrcRotate);

using namespace std;

static const TString gsDefName  = "MSrcRotate";
static const TString gsDefTitle = "Rotate position of the source";

// -------------------------------------------------------------------------
//
// Default constructor. The first (second) argument is the name of a container
// containing the source position in the camera plain, MScrPosCam (MDCA).
// The default is "MSrcPosCam" ("MDCA"). 
//
MSrcRotate::MSrcRotate(const char* srcPos, const char* dca, const char *name, const char *title)
  : fRA(0), fDEC(0), fRefMJD(0), fRunNumber(0)
{
    fName  = name  ? name  : gsDefName.Data();
    fTitle = title ? title : gsDefTitle.Data();

    SetSrcPosName(srcPos);
    SetDCAName(dca);
}

// -------------------------------------------------------------------------
//
// Look for/create the needed containers
// Check whether RA and DEC have been initialized
//
Int_t MSrcRotate::PreProcess(MParList *pList)
{
  if(!SearchForSrcPos(pList))
    return kFALSE;
  
  // look for MRawRunHeader
  fRunHeader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
  if (!fRunHeader)
    {
      *fLog << err << "MSrcRotate::PreProcess Error: MRawRunHeader not found... aborting" << endl;
      return kFALSE;
    }
  
  // look for MRawEvtHeader
  fEvtHeader = (MRawEvtHeader*)pList->FindObject("MRawEvtHeader");
  if (!fEvtHeader)
    {
      *fLog << err << "MSrcRotate::PreProcess Error: MRawEvtHeader not found... aborting." << endl;
      return kFALSE;
    }
  
  // look for/create the MObservatory
  fObservatory = (MObservatory*)pList->FindCreateObj("MObservatory");
  if(!fObservatory)
    {
      *fLog << err << "MSrcRotate::PreProcess Error: MObservatory not found... aborting." << endl;
      return kFALSE;
    }

  // check RA and DEC
  if(fRA==0. || fDEC ==0)
    *fLog << warn << "MSrcRotate::PreProcess Warning: RA=" << fRA << "or DEC=" << fDEC << " could be not initialized" << endl;    
	
  return kTRUE;
}

// -------------------------------------------------------------------------
//
// Perform the rotation of the source position 
// 
// FIXME: for the time being, this is computed by assuming constant event rate
//
Int_t MSrcRotate::ComputeNewSrcPosition()
{  
  if(fRunHeader->GetRunNumber()!=fRunNumber)
    {
      fRunNumber=fRunHeader->GetRunNumber();
      
      // save the number of events, initial and final times
      fNEvts    = fRunHeader->GetNumEvents();
      fIniTime  = fRunHeader->GetRunStart();
      fFinTime  = fRunHeader->GetRunEnd();
      fDeltaT   = (fFinTime.GetMjd()-fIniTime.GetMjd())/fNEvts;
      
      if((ULong_t)TMath::Nint(fIniTime.GetMjd())!=(ULong_t)TMath::Nint(fFinTime.GetMjd()))
	{
	  *fLog << err << "MSrcRotate::ComputeNewSrcPosition Error: Inial and final MJDs are different ("<<fIniTime.GetMjd()<<"!="<<fFinTime.GetMjd()<<")" << endl;
	  return kFALSE;
	}
      
#ifdef DEBUG
      cout << endl << "********************"  << endl;
      cout << "Run number: " << fRunHeader->GetRunNumber() << endl;
      cout << "Number of events: " << fNEvts << endl;
      cout << "Initial MJD date: " << fIniTime.GetMjd() << endl;
      cout << "Final MJD   date: " << fFinTime.GetMjd() << endl;
      cout << "Delta t: " << fDeltaT << endl;
#endif
    }
  
  
  // Compute the event time
  // FIXME: for the time being, this is computed by assuming constant event rate
  MTime eventTime;
  Double_t newMJD = fIniTime.GetMjd() + (fFinTime.GetMjd()-fIniTime.GetMjd())*fEvtHeader->GetDAQEvtNumber()/fNEvts;  
  eventTime.SetMjd(newMJD);    
  MTime refTime;
  refTime.SetMjd(fRefMJD);
  
  // de-rotate the source position
  const MAstroSky2Local Observation(eventTime, *fObservatory);
  const MAstroSky2Local RefObservation(refTime, *fObservatory);
  
#ifdef DEBUG
  printf("Run:%d, Event:%d, iniMJD=%15.5f, finMJD=%15.5f, fDeltaT=%15.5f, newMJD=%15.5f, fRefMJD=%15.5f, rotation=%15.5f, ref=%15.5f\n",
	 fRunHeader->GetRunNumber(),fEvtHeader->GetDAQEvtNumber(),
	 fIniTime.GetMjd(),fFinTime.GetMjd(),fDeltaT,
	 newMJD,fRefMJD,Observation.RotationAngle(fRA,fDEC),
	 RefObservation.RotationAngle(fRA,fDEC));
  //  cout << "newMJD=" << newMJD << ", fRefMJD="<<fRefMJD<<", rotation="<<Observation.RotationAngle(fRA,fDEC)<<", ref="<<RefObservation.RotationAngle(fRA,fDEC)<< endl;
#endif
  
  Double_t rotationAngle = fRefMJD ? 
    Observation.RotationAngle(fRA,fDEC)-RefObservation.RotationAngle(fRA,fDEC) :
    Observation.RotationAngle(fRA,fDEC) ;
  MSrcPosCam* srcpos = GetSrcPosCam();
  MDCA* dca = GetDCA();
  
  Float_t c = TMath::Cos(rotationAngle);
  Float_t s = TMath::Sin(rotationAngle);
  // perform a rotation of -rotationAngle to move the source back to the "initial" position
  Float_t newX = c*srcpos->GetX()-s*srcpos->GetY();
  Float_t newY = s*srcpos->GetX()+c*srcpos->GetY();
  
#ifdef DEBUG
  Double_t rotationAngleComp = fObservatory->RotationAngle(0.1256,2.63);
  cout << "Event " << fEvtHeader->GetDAQEvtNumber() << endl;
  cout << "newMJD=" << newMJD <<", rotationAngle=" << rotationAngle <<", rotationAngleComp=" << rotationAngleComp << ", oldX="<<fIniSrcPos.GetX()<< ", oldY="<<fIniSrcPos.GetY()<< ", newX="<<newX<< ", newY="<<newY << endl;
#endif
  
  srcpos->SetX(newX);
  srcpos->SetY(newY);
  dca->SetRefPoint(newX,newY);

  return kTRUE;
}
