/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 10/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Hendrik Bartko, 08/2004 <mailto:hbartko@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MCameraData
//
// This is a generalized class storing camera data. For example the cleaning
// level for the image cleaning is one possibility.
//
/////////////////////////////////////////////////////////////////////////////
#include "MCameraData.h"

#include "MCerPhotEvt.h"
#include "MCerPhotPix.h"

#include "MGeomCam.h"
#include "MGeomPix.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MPedPhotCam.h"
#include "MPedPhotPix.h"

#include "MSigmabar.h"

ClassImp(MCameraData);

using namespace std;

// --------------------------------------------------------------------------
//
// Creates a MCerPhotPix object for each pixel in the event
//
MCameraData::MCameraData(const char *name, const char *title)
{
    fName  = name  ? name  : "MCameraData";
    fTitle = title ? title : "Generalized storage container for camera contents";
}

/*
// --------------------------------------------------------------------------
//
// This is not yet implemented like it should.
//

void MCameraData::Draw(Option_t* option) 
{
    //
    //   FIXME!!! Here the Draw function of the CamDisplay
    //   should be called to add the CamDisplay to the Pad.
    //   The drawing should be done in MCamDisplay::Paint
    //

    //    MGeomCam *geom = fType ? new MGeomCamMagic : new MGeomCamCT1;
    //    MCamDisplay *disp = new MCamDisplay(geom);
    //    delete geom;
    //    disp->DrawPhotNum(this);
}
*/


// --------------------------------------------------------------------------
//
// Function to calculate the cleaning level for all pixels in a given event
// as the ratio between the measured photons and the pedestal rms.
// In order to do the image cleaning on average in the same photon flux
// (reconstructed photons per pixel area) for the inner and outer pixels,
// a correction factor is applied to the outer pixels (see TDAS 02-14).
// The correction factor assumes the ideal case that the pedestal rms 
// scales with the inverse square root of the pixel area.
//
// FIXME: Should the check noise<=0 be replaced by MBadPixels?
//
void MCameraData::CalcCleaningLevel(const MCerPhotEvt &evt, const MPedPhotCam &cam,
                                    const MGeomCam &geom)
{
    const Int_t n = geom.GetNumPixels();

    fData.Set(n);
    fData.Reset();

    fValidity.Set(n);
    fValidity.Reset();

    const Int_t entries = evt.GetNumPixels();

    //
    // check the number of all pixels against the noise level and
    // set them to 'unused' state if necessary
    // 
    for (Int_t i=0; i<entries; i++)
    {
        const MCerPhotPix &pix = evt[i];

        const Int_t idx = pix.GetPixId();
        const Float_t noise = cam[idx].GetRms();

        if (noise<=0) // fData[idx]=0, fValidity[idx]=0
            continue;

        //
	// We calculate a correction factor which accounts for the 
	// fact that pixels have different size (see TDAS 02-14).
	//
        fData[idx] = pix.GetNumPhotons() * geom.GetPixRatioSqrt(idx) / noise;
        fValidity[idx] = 1;
    }
}

// --------------------------------------------------------------------------
//
// Function to calculate the cleaning level for all pixels in a given event
// as the ratio between the measured photons and the pedestal rms.
// In order to do the image cleaning on average in the same photon flux
// (reconstructed photons per pixel area) for the inner and outer pixels,
// a correction factor is applied to the outer pixels (see TDAS 02-14).
// The correction factor takes the actual average pedestal RMS of the
// inner and outer pixels into account.
//
// FIXME: Should the check noise<=0 be replaced by MBadPixels?
//
void MCameraData::CalcCleaningLevel2(const MCerPhotEvt &evt, const MPedPhotCam &cam,
                                     const MGeomCam &geom)
{
    const Int_t n = geom.GetNumPixels();

    fData.Set(n);
    fData.Reset();

    fValidity.Set(n);
    fValidity.Reset();

    const Int_t entries   = evt.GetNumPixels();
    const Float_t anoise0 = cam.GetArea(0).GetRms();
    if (anoise0<=0)
        return;

    //
    // check the number of all pixels against the noise level and
    // set them to 'unused' state if necessary
    // 
    for (Int_t i=0; i<entries; i++)
    {
        const MCerPhotPix &pix = evt[i];

        const Int_t   idx   = pix.GetPixId();
        const Float_t noise = cam[idx].GetRms();

        if (noise<=0) // fData[idx]=0, fValidity[idx]=0
            continue;

        //
	// We calculate a correction factor which accounts for the 
	// fact that pixels have different size (see TDAS 02-14).
	// We also take into account that the RMS does not scale 
	// with the square root of the pixel area.
	// 
        const UInt_t  aidx  = geom[idx].GetAidx();
        const Float_t ratio = cam.GetArea(aidx).GetRms()/anoise0;

        fData[idx] = pix.GetNumPhotons() * geom.GetPixRatio(idx) * ratio / noise;
        fValidity[idx] = 1;
    }
}


void MCameraData::CalcCleaningLevel(const MCerPhotEvt &evt, const MSigmabar &sgb,
                                    const MGeomCam &geom)
{
    CalcCleaningLevel(evt, sgb.GetSigmabarInner(), geom);
}

void MCameraData::CalcCleaningLevel(const MCerPhotEvt &evt, Double_t noise,
                                    const MGeomCam &geom)
{
    const Int_t n = geom.GetNumPixels();

    fData.Set(n);
    fData.Reset();

    fValidity.Set(n);
    fValidity.Reset();

    if (noise<=0)
        return;

    const Int_t entries = evt.GetNumPixels();

    //
    // check the number of all pixels against the noise level and
    // set them to 'unused' state if necessary
    // 
    for (Int_t i=0; i<entries; i++)
    {
        const MCerPhotPix &pix = evt[i];

        const Int_t idx = pix.GetPixId();

        //
	// We calculate a correction factor which accounts for the 
	// fact that pixels have different size (see TDAS 02-14).
	//
        fData[idx] = pix.GetNumPhotons() * geom.GetPixRatio(idx) / noise;
        fValidity[idx] = 1;
    }
}

// --------------------------------------------------------------------------
//
// Function to calculate the cleaning level for all pixels in a given event
// as the ratio between the reconstructed number of photons per area of an
// inner pixel and the average pedestal RMS of the inner pixels (democratic
// image cleaning, see TDAS 02-14).
//
// FIXME: Should the check noise<=0 be replaced by MBadPixels?
//
void MCameraData::CalcCleaningLevelDemocratic(const MCerPhotEvt &evt, const MPedPhotCam &cam,
                                              const MGeomCam &geom)
{
    const Int_t n = geom.GetNumPixels();

    fData.Set(n);
    fData.Reset();

    fValidity.Set(n);
    fValidity.Reset();

    const Int_t   entries = evt.GetNumPixels();
    const Float_t noise0  = cam.GetArea(0).GetRms();
    if (noise0<=0)
        return;

    //
    // check the number of all pixels against the noise level and
    // set them to 'unused' state if necessary
    // 
    for (Int_t i=0; i<entries; i++)
    {
        const MCerPhotPix &pix = evt[i];

        const Int_t idx     = pix.GetPixId();
        const Float_t noise = cam[idx].GetRms();

        if (noise<=0)
            continue;

        //
        // We calculate a correction factor which accounts for the
        // fact that pixels have different size (see TDAS 02-14).
        //
        fData[idx] = pix.GetNumPhotons() * geom.GetPixRatio(idx) / noise0;
        fValidity[idx] = 1;
    }
}

// --------------------------------------------------------------------------
//
// Returns the contents of the pixel.
//
Bool_t MCameraData::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{
    if (idx<0 || idx>=fData.GetSize())
        return kFALSE;

    val = fData[idx];
    return fValidity[idx];
}

void MCameraData::DrawPixelContent(Int_t num) const
{
    *fLog << warn << "MCameraData::DrawPixelContent - not available." << endl;
}
