/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJCalibrateSignal
//
// This class is reading the output written by callisto. It calibrates
// signal and time.
//
// The signal and time extractors are read from the callisto-output. In
// pricipal you could overwrite these default using the resource file,
// but this is NOT recommended!
//
/////////////////////////////////////////////////////////////////////////////
#include "MJCalibrateSignal.h"

#include <TEnv.h>
#include <TFile.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MDirIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MStatusDisplay.h"

#include "MGeomCam.h"
#include "MHCamEvent.h"
#include "MPedestalCam.h"
#include "MBadPixelsCam.h"

#include "MCalibrationChargePINDiode.h"

#include "MCalibrationQECam.h"
#include "MCalibrationBlindCam.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationRelTimeCam.h"
/*
#include "MCalibrationIntensityQECam.h"
#include "MCalibrationIntensityBlindCam.h"
#include "MCalibrationIntensityChargeCam.h"
#include "MCalibrationIntensityRelTimeCam.h"
*/
#include "MReadReports.h"
#include "MReadMarsFile.h"
#include "MRawFileRead.h"
#include "MGeomApply.h"
#include "MMcPedestalCopy.h"
#include "MPointingPosCalc.h"
#include "MPedCalcFromLoGain.h"
#include "MExtractor.h"
#include "MTaskEnv.h"
#include "MCalibrateData.h"
#include "MCalibrateRelTimes.h"
#include "MBadPixelsMerge.h"
#include "MBadPixelsCalc.h"
#include "MBadPixelsTreat.h"
#include "MFillH.h"
#include "MWriteRootFile.h"

ClassImp(MJCalibrateSignal);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets fRuns to 0, fExtractor to NULL, fDataCheck to kFALSE
//
MJCalibrateSignal::MJCalibrateSignal(const char *name, const char *title)
    : fIsDataCheck(kFALSE)
{
    fName  = name  ? name  : "MJCalibrateSignal";
    fTitle = title ? title : "Tool to calibrate data";
}

Bool_t MJCalibrateSignal::WriteResult()
{
    if (fPathOut.IsNull())
    {
        *fLog << inf << "No output path specified via SetPathOut - no output written." << endl;
        return kTRUE;
    }

    const TString oname = Form("%s/signal%06d.root", (const char*)fPathOut, fSequence.GetSequence());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "RECREATE");

    *fLog << inf << " - MStatusDisplay..." << flush;
    if (fDisplay && fDisplay->Write()<=0)
    {
        *fLog << err << "Unable to write MStatusDisplay to " << oname << endl;
        return kFALSE;
    }
    *fLog << inf << "ok." << endl;

    return kTRUE;
}

Bool_t MJCalibrateSignal::ReadCalibration(TObjArray &l, MBadPixelsCam &cam, MTask* &ext1, MTask* &ext2, TString &geom) const
{
    const TString fname = Form("%s/calib%06d.root", fPathIn.Data(), fSequence.GetSequence());

    *fLog << inf << "Reading from file: " << fname << endl;

    TFile file(fname, "READ");
    if (!file.IsOpen())
    {
        *fLog << err << dbginf << "ERROR - Could not open file " << fname << endl;
        return kFALSE;
    }

    TObject *o = file.Get("ExtractSignal");
    if (o && !o->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << dbginf << "ERROR - ExtractSignal read from " << fname << " doesn't inherit from MExtractor!" << endl;
        return kFALSE;
    }
    ext1 = o ? (MTask*)o->Clone() : NULL;

    o = file.Get("ExtractTime");
    if (o && !o->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << dbginf << "ERROR - ExtractTime read from " << fname << " doesn't inherit from MExtractor!" << endl;
        return kFALSE;
    }
    ext2 = o ? (MTask*)o->Clone() : NULL;
    if (!ext1 && !ext2)
    {
        *fLog << err << dbginf << "ERROR - Neither ExtractSignal nor ExrtractTime found in " << fname << "!" << endl;
        return kFALSE;
    }

    o = file.Get("MGeomCam");
    if (o && !o->InheritsFrom(MGeomCam::Class()))
    {
        *fLog << err << dbginf << "ERROR - MGeomCam read from " << fname << " doesn't inherit from MGeomCam!" << endl;
        return kFALSE;
    }
    geom = o ? o->ClassName() : "";

    TObjArray cont(l);
    cont.Add(&cam);
    return ReadContainer(cont);
}

// --------------------------------------------------------------------------
//
// MJCalibration allows to setup several option by a resource file:
//   MJCalibration.Display: full, datacheck, normal
//   MJCalibration.RelTimeCalibration: yes,no
//   MJCalibration.DataCheck: yes,no
//   MJCalibration.Debug: yes,no
//   MJCalibration.Intensity: yes,no
//   MJCalibration.UseBlindPixel: yes,no
//   MJCalibration.UsePINDiode: yes,no
//   MJCalibration.Geometry: MGeomCamMagic, MGeomCamECO1000
//
// For more details see the class description and the corresponding Getters
// 
Bool_t MJCalibrateSignal::CheckEnvLocal()
{
    SetDataCheck(GetEnv("DataCheck", IsDataCheck()));
    return kTRUE;
}

Bool_t MJCalibrateSignal::ProcessFile(MPedestalCam &pedcam)
{
    if (!fSequence.IsValid())
    {
        *fLog << err << "ERROR - Sequence invalid!" << endl;
        return kFALSE;
    }

    //if (!CheckEnv())
    //    return kFALSE;

    CheckEnv();

    // --------------------------------------------------------------------------------

    *fLog << inf;
    fLog->Separator(GetDescriptor());
    *fLog << "Calculate calibrated data from runs ";
    *fLog << fSequence.GetName() << endl;
    *fLog << endl;

    // --------------------------------------------------------------------------------

    MDirIter iter;
    const Int_t n0 = fSequence.SetupDatRuns(iter, fPathData, "D", IsDataCheck());
    const Int_t n1 = fSequence.GetNumDatRuns();
    if (n0==0)
    {
        *fLog << err << "ERROR - No input files of sequence found!" << endl;
        return kFALSE;
    }
    if (n0!=n1)
    {
        *fLog << err << "ERROR - Number of files found (" << n0 << ") doesn't match number of files in sequence (" << n1 << ")" << endl;
        return kFALSE;
    }

    // Read File
    /*
     MCalibrationIntensityChargeCam      calcam;
     MCalibrationIntensityQECam          qecam;
     MCalibrationIntensityBlindCam       bndcam;
     MCalibrationIntensityRelTimeCam     tmcam;
     */
    MCalibrationChargeCam      calcam;
    MCalibrationQECam          qecam;
    MCalibrationBlindCam       bndcam;
    MCalibrationChargePINDiode pind;
    MCalibrationRelTimeCam     tmcam;
    MBadPixelsCam              badpix;

    MTask *extractor1=0;
    MTask *extractor2=0;
    TString geom;

    TObjArray calibcont;
    calibcont.Add(&calcam);
    calibcont.Add(&qecam);
    calibcont.Add(&bndcam);
    calibcont.Add(&pind);
    calibcont.Add(&tmcam);

    if (!ReadCalibration(calibcont, badpix, extractor1, extractor2, geom))
        return kFALSE;

    *fLog << all;
    if (extractor1)
    {
        *fLog << underline << "Signal Extractor found in calibration file" << endl;
        extractor1->Print();
        *fLog << endl;
    }
    else
        *fLog << inf << "No Signal Extractor: ExtractSignal in file." << endl;

    if (extractor2)
    {
        *fLog << underline << "Time Extractor found in calibration file" << endl;
        extractor2->Print();
        *fLog << endl;
    }
    else
        *fLog << inf << "No Time Extractor: ExtractTime in file." << endl;

    if (!geom.IsNull())
        *fLog << inf << "Camera geometry found in file: " << geom << endl;
    else
        *fLog << inf << "No Camera geometry found using default <MGeomCamMagic>" << endl;

    // This is necessary for the case in which it is not in the files
    MBadPixelsCam badcam;

    // Setup Parlist
    MParList plist;
    plist.AddToList(this); // take care of fDisplay!
    plist.AddToList(&badcam);
    plist.AddToList(&pedcam);
    plist.AddToList(&calibcont);

    // Setup Tasklist
    MTaskList tlist;
    plist.AddToList(&tlist);

    // FIXME: Move this to an intermediate class MJMagic
    Byte_t filetype = 2;
    /*
    TString name = iter.Next();
    Byte_t filetype = MRawFileRead::IsFileValid(name);
    if (!filetype)
        filetype = MReadMarsFile::IsFileValid(name)+1;
    if (filetype<1||filetype>3)
    {
        gLog << err << "ERROR - FileType #" << (int)filetype << " of first file " << name << " unknown..." << endl;
        return kFALSE;
    } */
    // 1 = raw-file
    // 2 = raw-root file
    // 3 = mc-raw file

    MReadReports readreal;
    readreal.AddTree("Events", "MTime.", kTRUE);
    readreal.AddTree("Trigger");
    readreal.AddTree("Camera");
    readreal.AddTree("Drive");
    readreal.AddTree("CC");
    readreal.AddTree("Currents");

    //MReadMarsFile read("Events");
    //read.DisableAutoScheme();
    MRawFileRead rawread(NULL);
    if (IsDataCheck())
        rawread.AddFiles(iter);
    else
        readreal.AddFiles(iter);

    MGeomApply             apply; // Only necessary to create geometry
    if (!geom.IsNull())
        apply.SetGeometry(geom);
    MBadPixelsMerge        merge(&badpix);

    MPedCalcFromLoGain     pedlo;
    pedlo.SetPedestalUpdate(kTRUE);
    // FIXME: How to get the fixed value 15 automatically?
    const Float_t win = static_cast<MExtractor*>(extractor1)->GetNumHiGainSamples();
    pedlo.SetExtractWindow(15, (UShort_t)TMath::Nint(win));

    MMcPedestalCopy        pcopy;
    MTaskEnv taskenv1("ExtractSignal");
    MTaskEnv taskenv2("ExtractTime");
    taskenv1.SetDefault(extractor1);
    taskenv2.SetDefault(extractor2);
    MCalibrateData         calib;
    if (filetype==3)
        calib.SetCalibrationMode(MCalibrateData::kFfactor);
    MCalibrateRelTimes     caltm;
    MBadPixelsCalc         bpcal;
    MBadPixelsTreat        treat;

    MHCamEvent evt0(0, "PedestalFLG", "Pedestal from Lo Gain;;P [fadc/sl]");
    MHCamEvent evt1(0, "Extra'd",     "Extracted Signal;;S [fadc/sl]");
    MHCamEvent evt2(0, "PedPhot",     "Calibrated Pedestal;;P [\\gamma]");
    MHCamEvent evt3(1, "PedRMS",      "Calibrated Pedestal RMS;;\\sigma_{p} [\\gamma]");
    MHCamEvent evt4(0, "Interp'd",    "Interpolated Signal;;S [\\gamma]");
    MHCamEvent evt5(2, "Unsuitable",  "Unsuitable event ratio;;%");
    MHCamEvent evt6(0, "Times",       "Arrival Time;;T [slice]");
    evt0.EnableVariance();
    evt1.EnableVariance();
    evt2.EnableVariance();
    evt3.EnableVariance();
    evt4.EnableVariance();
    evt6.EnableVariance();

    MFillH fill0(&evt0, "MPedestalCam",        "FillPedestalFLG");
    MFillH fill1(&evt1, "MExtractedSignalCam", "FillExtracted");
    MFillH fill2(&evt2, "MPedPhotCam",         "FillPedPhot");
    MFillH fill3(&evt3, "MPedPhotCam",         "FillPedRMS");
    MFillH fill4(&evt4, "MCerPhotEvt",         "FillInterpolated");
    MFillH fill5(&evt5, "MBadPixelsCam",       "FillUnsuitable");
    MFillH fill6(&evt6, "MArrivalTime",        "FillTimes");

    MWriteRootFile write(2, Form("%s{s/_D_/_Y_}", fPathOut.Data()), fOverwrite);
    // Run Header
    write.AddContainer("MRawRunHeader",       "RunHeaders");
    write.AddContainer("MBadPixelsCam",       "RunHeaders");
    write.AddContainer("MGeomCam",            "RunHeaders");
    // Monte Carlo Headers
    write.AddContainer("MMcTrigHeader",       "RunHeaders", kFALSE);
    write.AddContainer("MMcConfigRunHeader",  "RunHeaders", kFALSE);
    write.AddContainer("MMcCorsikaRunHeader", "RunHeaders", kFALSE);
    // Monte Carlo
    write.AddContainer("MMcEvt",              "Events", kFALSE);
    write.AddContainer("MMcTrig",             "Events", kFALSE);
    // Data
    write.AddContainer("MCerPhotEvt",         "Events");
    write.AddContainer("MPedPhotCam",         "Events");
    write.AddContainer("MTime",               "Events", kFALSE);
    write.AddContainer("MRawEvtHeader",       "Events");
    write.AddContainer("MArrivalTime",        "Events");
    // Slow-Control: Current
    write.AddContainer("MTimeCurrents",       "Currents", kFALSE);
    write.AddContainer("MCameraDC",           "Currents", kFALSE);
    write.AddContainer("MReportCurrents",     "Currents", kFALSE);
    // Slow-Control: Camera
    write.AddContainer("MReportCamera",       "Camera", kFALSE);
    write.AddContainer("MTimeCamera",         "Camera", kFALSE);
    write.AddContainer("MCameraAUX",          "Camera", kFALSE);
    write.AddContainer("MCameraCalibration",  "Camera", kFALSE);
    write.AddContainer("MCameraCooling",      "Camera", kFALSE);
    write.AddContainer("MCameraHV",           "Camera", kFALSE);
    write.AddContainer("MCameraLV",           "Camera", kFALSE);
    write.AddContainer("MCameraLids",         "Camera", kFALSE);
    // Slow-Control: Trigger
    write.AddContainer("MReportTrigger",      "Trigger", kFALSE);
    write.AddContainer("MTimeTrigger",        "Trigger", kFALSE);
    // Slow-Control: Drive
    write.AddContainer("MPointingPos",        "Drive", kFALSE);
    write.AddContainer("MReportDrive",        "Drive", kFALSE);
    write.AddContainer("MTimeDrive",          "Drive", kFALSE);
    // Slow-Control: Central Control
    write.AddContainer("MReportCC",           "CC", kFALSE);
    write.AddContainer("MTimeCC",             "CC", kFALSE);

    // Now setup tasklist for events
    MTaskList tlist2;
    tlist2.AddToList(&apply);
    tlist2.AddToList(&merge);
    tlist2.AddToList(filetype==3 ? (MTask*)&pcopy : (MTask*)&pedlo);
    tlist2.AddToList(&fill0);
    if (extractor1)
        tlist2.AddToList(&taskenv1);
    if (extractor2)
        tlist2.AddToList(&taskenv2);
    tlist2.AddToList(&fill1);
    tlist2.AddToList(&calib);
    tlist2.AddToList(&caltm);
    tlist2.AddToList(&fill2);
    tlist2.AddToList(&bpcal);
    tlist2.AddToList(&treat);
    tlist2.AddToList(&fill3);
    tlist2.AddToList(&fill4);
    tlist2.AddToList(&fill5);
    tlist2.AddToList(&fill6);

    // Setup List for Drive-tree
    MPointingPosCalc pcalc;

    // Now setup main tasklist
    tlist.AddToList(IsDataCheck() ? (MTask*)&rawread : (MTask*)&readreal);
    tlist.AddToList(&tlist2, IsDataCheck()?"All":"Events");
    if (!IsDataCheck())
        tlist.AddToList(&pcalc, "Drive");
    tlist.AddToList(&write);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);
    if (!SetupEnv(evtloop))
        return kFALSE;

    // Execute first analysis
    if (!evtloop.Eventloop(fMaxEvents))
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    if (!WriteResult())
        return kFALSE;

    *fLog << all << GetDescriptor() << ": Done." << endl;
    *fLog << endl << endl;

    return kTRUE;
}
