/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 8/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MImagePar
//
// Storage Container for new image parameters
//
//    Short_t fNumIslands;           // number of islands found
//
//    Short_t fNumHGSaturatedPixels; // number of pixels with saturating hi-gains
//    Short_t fNumSaturatedPixels;   // number of pixels with saturating lo-gains
//
/////////////////////////////////////////////////////////////////////////////
#include "MImagePar.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MCerPhotEvt.h"
#include "MCerPhotPix.h"

ClassImp(MImagePar);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor.
//
MImagePar::MImagePar(const char *name, const char *title)
{
    fName  = name  ? name  : "MImagePar";
    fTitle = title ? title : "New image parameters";

    Reset();
}

// --------------------------------------------------------------------------
//
void MImagePar::Reset()
{
    fNumIslands = -1;

    fNumSatPixelsHG = -1;
    fNumSatPixelsLG = -1;
}

// --------------------------------------------------------------------------
//
//  Calculation of new image parameters
//
void MImagePar::Calc(const MCerPhotEvt &evt)
{
    // Get number of saturating pixels
    fNumSatPixelsHG = 0;
    fNumSatPixelsLG = 0;

    const UInt_t npixevt = evt.GetNumPixels();
    for (UInt_t i=0; i<npixevt; i++)
    {
        const MCerPhotPix &pix = evt[i];

	if (pix.IsPixelHGSaturated())
            fNumSatPixelsHG++;
	if (pix.IsPixelSaturated())
            fNumSatPixelsLG++;
    }

    // Get number of islands
    fNumIslands = evt.GetNumIslands();

    SetReadyToSave();
}

// --------------------------------------------------------------------------
//
void MImagePar::Print(Option_t *) const
{
    *fLog << all;
    *fLog << "Image Parameters (" << GetName() << ")" << endl;
    *fLog << " - Num Islands    [#]   = " << fNumIslands << " Islands" << endl;
    *fLog << " - Sat.Pixels/HG  [#]   = " << fNumSatPixelsHG << " Pixels" << endl;
    *fLog << " - Sat.Pixels/LG  [#]   = " << fNumSatPixelsLG << " Pixels" << endl;
}
