/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek, 03/2003 <mailto:wittek@mppmu.mpg.de>
!   Author(s): Thomas Bretz, 04/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHNewImagePar
//
////////////////////////////////////////////////////////////////////////////
#include "MHNewImagePar.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"
#include "MBinning.h"
#include "MParList.h"

#include "MHillas.h"
#include "MNewImagePar.h"

ClassImp(MHNewImagePar);

using namespace std;

// --------------------------------------------------------------------------
//
// Setup histograms 
//
MHNewImagePar::MHNewImagePar(const char *name, const char *title)
{
    fName  = name  ? name  : "MHNewImagePar";
    fTitle = title ? title : "Histograms of new image parameters";

    fHistLeakage1.SetName("Leakage1");
    fHistLeakage1.SetTitle("Leakage_{1}");
    fHistLeakage1.SetXTitle("Leakage");
    fHistLeakage1.SetYTitle("Counts");
    fHistLeakage1.SetDirectory(NULL);
    fHistLeakage1.UseCurrentStyle();
    fHistLeakage1.SetFillStyle(4000);

    fHistLeakage2.SetName("Leakage2");
    fHistLeakage2.SetTitle("Leakage_{2}");
    fHistLeakage2.SetXTitle("Leakage");
    fHistLeakage2.SetYTitle("Counts");
    fHistLeakage2.SetDirectory(NULL);
    fHistLeakage2.UseCurrentStyle();
    fHistLeakage2.SetLineColor(kBlue);
    fHistLeakage2.SetFillStyle(4000);
 
    fHistUsedPix.SetName("UsedPix");
    fHistUsedPix.SetTitle("Number of used pixels");
    fHistUsedPix.SetXTitle("Number of Pixels");
    fHistUsedPix.SetYTitle("Counts");
    fHistUsedPix.SetDirectory(NULL);
    fHistUsedPix.UseCurrentStyle();
    fHistUsedPix.SetLineColor(kBlue);
    fHistUsedPix.SetFillStyle(4000);

    fHistCorePix.SetName("CorePix");
    fHistCorePix.SetTitle("Number of core pixels");
    fHistCorePix.SetXTitle("Number of Pixels");
    fHistCorePix.SetYTitle("Counts");
    fHistCorePix.SetDirectory(NULL);
    fHistCorePix.UseCurrentStyle();
    fHistCorePix.SetLineColor(kBlack);
    fHistCorePix.SetFillStyle(4000);

    fHistUsedArea.SetName("UsedArea");
    fHistUsedArea.SetTitle("Area of used pixels");
    fHistUsedArea.SetXTitle("Area [m^2]");
    fHistUsedArea.SetYTitle("Counts");
    fHistUsedArea.SetDirectory(NULL);
    fHistUsedArea.UseCurrentStyle();
    fHistUsedArea.SetLineColor(kBlue);
    fHistUsedArea.SetFillStyle(4000);

    fHistCoreArea.SetName("CoreArea");
    fHistCoreArea.SetTitle("Area of core pixels");
    fHistCoreArea.SetXTitle("Area [m^2]");
    fHistCoreArea.SetYTitle("Counts");
    fHistCoreArea.SetDirectory(NULL);
    fHistCoreArea.UseCurrentStyle();
    fHistCoreArea.SetLineColor(kBlack);
    fHistCoreArea.SetFillStyle(4000);

    fHistConc.SetDirectory(NULL);
    fHistConc1.SetDirectory(NULL);
    fHistConc.SetName("Conc2");
    fHistConc1.SetName("Conc1");
    fHistConc.SetTitle("Ratio: Conc");
    fHistConc1.SetTitle("Ratio: Conc1");
    fHistConc.SetXTitle("Ratio");
    fHistConc1.SetXTitle("Ratio");
    fHistConc.SetYTitle("Counts");
    fHistConc1.SetYTitle("Counts");
    fHistConc1.UseCurrentStyle();
    fHistConc.UseCurrentStyle();
    fHistConc.SetFillStyle(4000);
    fHistConc1.SetFillStyle(4000);
    fHistConc1.SetLineColor(kBlue);


    MBinning bins;

    bins.SetEdges(100, 0, 1);
    bins.Apply(fHistLeakage1);
    bins.Apply(fHistLeakage2);
    bins.Apply(fHistConc);
    bins.Apply(fHistConc1);

    bins.SetEdges(75, 0.5, 150.5);
    bins.Apply(fHistUsedPix);
    bins.Apply(fHistCorePix);

    bins.SetEdges(75, 0, 0.249);
    bins.Apply(fHistUsedArea);
    bins.Apply(fHistCoreArea);
}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning
//
Bool_t MHNewImagePar::SetupFill(const MParList *plist)
{
    ApplyBinning(*plist, "Leakage", &fHistLeakage1);
    ApplyBinning(*plist, "Leakage", &fHistLeakage2);

    ApplyBinning(*plist, "Pixels",  &fHistUsedPix);
    ApplyBinning(*plist, "Pixels",  &fHistCorePix);

    ApplyBinning(*plist, "Area",    &fHistUsedArea);
    ApplyBinning(*plist, "Area",    &fHistCoreArea);

    ApplyBinning(*plist, "Conc",    &fHistConc);
    ApplyBinning(*plist, "Conc1",   &fHistConc1);

    return kTRUE;
}


// --------------------------------------------------------------------------
//
// Fill the histograms with data from a MNewImagePar container.
//
Bool_t MHNewImagePar::Fill(const MParContainer *par, const Stat_t w)
{
    if (!par)
    {
        *fLog << err << "MHNewImagePar::Fill: Pointer (!=NULL) expected." << endl;
        return kFALSE;
    }

    const MNewImagePar &h = *(MNewImagePar*)par;

    fHistLeakage1.Fill(h.GetLeakage1(), w);
    fHistLeakage2.Fill(h.GetLeakage2(), w);

    fHistUsedPix.Fill(h.GetNumUsedPixels(), w);
    fHistCorePix.Fill(h.GetNumCorePixels(), w);

    fHistUsedArea.Fill(h.GetUsedArea()/1000000, w);
    fHistCoreArea.Fill(h.GetCoreArea()/1000000, w);

    fHistConc.Fill(h.GetConc(), w);
    fHistConc1.Fill(h.GetConc1(), w);

    return kTRUE;
}

void MHNewImagePar::Paint(Option_t *o)
{
    if (TString(o)==(TString)"log" && fHistLeakage1.GetMaximum()>0)
        gPad->SetLogy();
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the two histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHNewImagePar::Draw(Option_t *)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->Divide(2,2);

    pad->cd(1);
    gPad->SetBorderMode(0);
    TAxis &x = *fHistLeakage1.GetXaxis();
    x.SetRangeUser(0.0, x.GetXmax());
    MH::DrawSame(fHistLeakage1, fHistLeakage2, "Leakage1 and Leakage2");
    fHistLeakage1.SetMinimum();
    fHistLeakage2.SetMinimum();
    fHistLeakage2.SetMaximum(0.1);   // dummy value to allow log-scale
    AppendPad("log");

    pad->cd(2);
    gPad->SetBorderMode(0);
    MH::DrawSame(fHistCorePix, fHistUsedPix, "Number of core/used Pixels");

    pad->cd(3);
    gPad->SetBorderMode(0);
    MH::DrawSame(fHistConc1, fHistConc, "Concentrations");

    pad->cd(4);
    gPad->SetBorderMode(0);
    MH::DrawSame(fHistCoreArea, fHistUsedArea, "Area of core/used Pixels");
}

TH1 *MHNewImagePar::GetHistByName(const TString name)
{
    if (name.Contains("Leakage1", TString::kIgnoreCase))
        return &fHistLeakage1;
    if (name.Contains("Leakage2", TString::kIgnoreCase))
        return &fHistLeakage2;
    if (name.Contains("Conc", TString::kIgnoreCase))
        return &fHistConc;
    if (name.Contains("Conc1", TString::kIgnoreCase))
        return &fHistConc1;
    if (name.Contains("UsedPix", TString::kIgnoreCase))
        return &fHistUsedPix;
    if (name.Contains("CorePix", TString::kIgnoreCase))
        return &fHistCorePix;
    if (name.Contains("UsedArea", TString::kIgnoreCase))
        return &fHistUsedArea;
    if (name.Contains("CoreArea", TString::kIgnoreCase))
        return &fHistCoreArea;

    return NULL;
}
