/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug  04/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MCalibrateRelTimes
//
//   This task takes the extracted arrival times from MArrivalTimeCam for each  
//   pixel and applies the offset calibrated in MCalibrationRelTimeCam 
//   The calibrated arrival time and its error gets stored in MCerPhotEvt. 
//
//   Input Containers:
//    MArrivalTimeCam 
//    MCalibrationRelTimeCam
//
//   Output Containers:
//    MArrivalTime
//
//////////////////////////////////////////////////////////////////////////////
#include "MCalibrateRelTimes.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"
#include "MH.h"

#include "MGeomCam.h"

#include "MCalibrationRelTimeCam.h"
#include "MCalibrationRelTimePix.h"

#include "MArrivalTimeCam.h"
#include "MArrivalTimePix.h"

#include "MBadPixelsCam.h"
#include "MBadPixelsPix.h"

#include "MArrivalTime.h"

ClassImp(MCalibrateRelTimes);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
MCalibrateRelTimes::MCalibrateRelTimes(const char *name, const char *title) 
    : fGeomCam(NULL), fCalibrations(NULL), fBadPixels(NULL), fSignals(NULL), 
      fArrivalTime(NULL)
{
    fName  = name  ? name  : "MCalibrateRelTimes";
    fTitle = title ? title : "Task to calculate the calibrated arrival times of photons in one event";
}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MGeomCam
//  - MCalibrationRelTimesCam
//  - MArrivalTimeCam
//  - MBadPixelsCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MArrivalTime
//
Int_t MCalibrateRelTimes::PreProcess(MParList *pList)
{

    fSignals = (MArrivalTimeCam*)pList->FindObject(AddSerialNumber("MArrivalTimeCam"));

    if (!fSignals)
    {
      *fLog << err << AddSerialNumber("MArrivalTimeCam") << " not found ... aborting" << endl;
        return kFALSE;
    }

    fBadPixels = (MBadPixelsCam*)pList->FindObject(AddSerialNumber("MBadPixelsCam"));
    if (!fBadPixels)
      *fLog << warn << AddSerialNumber("MBadPixelsCam") << " not found ... no action" << endl;
    

    fCalibrations = (MCalibrationRelTimeCam*)pList->FindObject(AddSerialNumber("MCalibrationRelTimeCam"));
    if (!fCalibrations)
      {
        *fLog << err << AddSerialNumber("MCalibrationRelTimeCam") << " not found ... aborting." << endl;
        return kFALSE;
      }


    fArrivalTime = (MArrivalTime*)pList->FindCreateObj(AddSerialNumber("MArrivalTime"));
    if (!fArrivalTime)
      {
        *fLog << err << AddSerialNumber("MArrivalTime") << ": Cannot create ... aborting." << endl;
        return kFALSE;
      }
    
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Check for validity of the selected calibration method, switch to a 
// different one in case of need
//
Bool_t MCalibrateRelTimes::ReInit(MParList *pList)
{
  return kTRUE;
}
// --------------------------------------------------------------------------
//
// Apply the calibration factors to the extracted signal according to the 
// selected calibration method
//
Int_t MCalibrateRelTimes::Process()
{
    /*
     if (fCalibrations->GetNumPixels() != (UInt_t)fSignals->GetSize())
     {
     // FIXME: MArrivalTime must be of variable size -
     //        like MCerPhotEvt - because we must be able
     //        to reduce size by zero supression
     //        For the moment this check could be done in ReInit...
     *fLog << err << "MArrivalTime and MCalibrationCam have different sizes... abort." << endl;
     return kFALSE;
     }
     */

    const UInt_t npix = fSignals->GetSize();

    for (UInt_t pixidx=0; pixidx<npix; pixidx++)
    {
        MCalibrationRelTimePix &pix = (MCalibrationRelTimePix&)(*fCalibrations)[pixidx];

        if (fBadPixels && (*fBadPixels)[pixidx].IsUnsuitable())
            continue;

        const Float_t offset    = pix.GetTimeOffset();
        const Float_t precision = pix.GetTimePrecision();

        MArrivalTimePix &sig = (*fSignals)[pixidx];

        Float_t signal;
        Float_t sigerr;

        if (sig.IsLoGainUsed())
        {
            signal = sig.GetArrivalTimeLoGain();
            sigerr = sig.GetArrivalTimeLoGainError();
        }
        else
        {
            signal = sig.GetArrivalTimeHiGain();
            sigerr = sig.GetArrivalTimeHiGainError();
        }

        const Float_t time = signal - offset;
        const Float_t err  = sigerr*sigerr + precision*precision;

        fArrivalTime->SetTime(pixidx, time);
        fArrivalTime->SetTimeErr(pixidx, err>0 ? TMath::Sqrt(err) : 0);

    } /* for (UInt_t pixidx=0; pixidx<npix; pixidx++) */

    fArrivalTime->SetReadyToSave();

    return kTRUE;
}
