/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug, 04/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//   MExtractTime
//
//   Base class for the signal extractors, used the functions 
//   FindTimeHiGain() and FindTimeLoGain() to extract the signal and 
//   substract the pedestal value    
//
//   The following figure gives and example of possible inheritance trees. 
//   An extractor class can inherit from each of the following base classes:
//    - MExtractor
//    - MExtractTime
//    - MExtractTimeAndCharge
//
//Begin_Html
/*
<img src="images/ExtractorClasses.gif">
*/
//End_Html
//
//   The following variables have to be set by the derived class and 
//   do not have defaults:
//   - fNumHiGainSamples
//   - fNumLoGainSamples
//   - fSqrtHiGainSamples
//   - fSqrtLoGainSamples
//
// Input Containers:
//   MRawEvtData
//   MRawRunHeader
//   MPedestalCam
//
// Output Containers:
//   MArrivalTimeCam
//
//////////////////////////////////////////////////////////////////////////////
#include "MExtractTime.h"

#include <fstream>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRawEvtData.h"
#include "MRawEvtPixelIter.h"
#include "MRawRunHeader.h"

#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include "MArrivalTimeCam.h"
#include "MArrivalTimePix.h"

ClassImp(MExtractTime);

using namespace std;

const TString MExtractTime::fgNameTimeCam   = "MArrivalTimeCam";
// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Set: 
// - all pointers to NULL
// - all variables to 0
// - fSaturationLimit to fgSaturationLimit
// - fNameTimeCam to fgNameTimeCam
//
// Call:
// -  AddToBranchList("MRawEvtData.*")
//
MExtractTime::MExtractTime(const char *name, const char *title)
  : fArrTime(NULL)
{

  fName  = name  ? name  : "MExtractTime";
  fTitle = title ? title : "Base class for signal extractors";
  
  SetNameTimeCam();
}



// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawEvtData
//  - MRawRunHeader
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//
//  - MArrivalTimeCam
//
Int_t MExtractTime::PreProcess(MParList *pList)
{
  fArrTime = (MArrivalTimeCam*)pList->FindCreateObj("MArrivalTimeCam",AddSerialNumber(fNameTimeCam));
  if (!fArrTime)
      return kFALSE;

  return PreProcessStd(pList);
}

// --------------------------------------------------------------------------
//
// The ReInit calls:
// -  MExtractor::ReInit()
//
// Call: 
// - MArrivalTimeCam::SetUsedFADCSlices(fHiGainFirst, fHiGainLast, fNumHiGainSamples,
//                                      fLoGainFirst, fLoGainLast, fNumLoGainSamples);
//
Bool_t MExtractTime::ReInit(MParList *pList)
{
    if (!MExtractor::ReInit(pList))
        return kFALSE;

    if (fArrTime)
      fArrTime->SetUsedFADCSlices(fHiGainFirst, fHiGainLast+fHiLoLast, fLoGainFirst, fLoGainLast);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slices and store them as a new
// pixel in the MArrivalTimeCam container.
//
Int_t MExtractTime::Process()
{


  MRawEvtPixelIter pixel(fRawEvt);
  fArrTime->Clear();

  while (pixel.Next())
    {
      //
      // Find signal in hi- and lo-gain
      //
      Float_t timehi=0., deltatimehi=0.;
      Byte_t sathi=0;

      const Int_t pixid = pixel.GetPixelId();
      const MPedestalPix  &ped = (*fPedestals)[pixid];
      MArrivalTimePix &pix = (*fArrTime)[pixid];

      FindTimeHiGain(pixel.GetHiGainSamples()+fHiGainFirst, timehi, deltatimehi, sathi, ped);

      Float_t timelo=0., deltatimelo=0.;
      Byte_t satlo=0;

      if ((sathi)&&pixel.HasLoGain())
	  FindTimeLoGain(pixel.GetLoGainSamples()+fLoGainFirst, timelo, deltatimelo, satlo, ped);

      pix.SetArrivalTime(timehi, deltatimehi, timelo-fOffsetLoGain, deltatimelo);
      pix.SetGainSaturation(sathi, sathi, satlo);
 
    } /* while (pixel.Next()) */

    fArrTime->SetReadyToSave();

    return kTRUE;
}

void MExtractTime::Print(Option_t *o) const
{
    if (IsA()==MExtractTime::Class())
        *fLog << GetDescriptor() << ":" << endl;
    *fLog << " Offset Lo-Gain: " << fOffsetLoGain << endl;
    MExtractor::Print(o);
}
