#include "MCamDisplay.h"

#include <math.h>
#include <TClonesArray.h>
#include <TCanvas.h>
#include <TStyle.h>
#include <TBox.h>
#include <TText.h>

#include "MHexagon.h"
#include "MCamGeom.h"

#include "MCerPhotEvt.h" 


ClassImp(MCamDisplay)

MCamDisplay::MCamDisplay (Int_t type ) 
{ 
  //    default constructor

  //    set the color palette 

  gStyle->SetPalette(1,0) ; 

  fAutoScale  = kTRUE ; 
  
  MCamGeom *geometry  = new MCamGeom( type ) ; 

  fNbPixels = geometry->GetNbPixels() ; 
  fPixels = new TClonesArray("MHexagon", fNbPixels ) ; 

  //  create the hexagons of the display
  
  TClonesArray &obj = *fPixels ; 
  
  for (Int_t i=0; i< fNbPixels; i++ ) 
    { 
      new (obj[i]) MHexagon(geometry->GetX(i) , 
			    geometry->GetY(i) , 
			    geometry->GetR(i) ) ; 
    } 

  // set the range to default

  fMinPhe  = -2.  ; 
  fMaxPhe  = 50. ; 

  // set up the Legend 

  fLegend = new TClonesArray("TBox", ITEMS_LEGEND ) ; 
  TClonesArray &obj1 = *fLegend ; 

  fLegText = new TClonesArray("TText", ITEMS_LEGEND ) ; 
  TClonesArray &obj2 = *fLegText ; 
  
  char text[100] ; 

  Float_t help ; 

  help = 50. / ITEMS_LEGEND ; 

  for ( Int_t il = 0 ; il < ITEMS_LEGEND ; il++ ) 
    { 
      new ( obj1[il] ) TBox(650, il*40-500 , 700, il*40-460 ) ; 
      ( (TBox*) fLegend->At(il))->SetFillColor( GetColor ( (Float_t) help*il) ) ;
  
      sprintf ( text, "%5.1f",  (Float_t) help * il ) ; 
      new ( obj2[il] ) TText(720, il*40-480, text ) ;
      ( (TText*) fLegText->At(il))->SetTextSize (0.025) ; 
      ( (TText*) fLegText->At(il))->SetTextAlign(12) ; 
  } 
} 

MCamDisplay::~MCamDisplay() 
{ 
  delete fPixels ; 
} 


void MCamDisplay::Init() 
{ 

  // Set the right colors
  
  gStyle->SetPalette(1, 0) ; 

  if ( ! gPad ) new TCanvas("display", "MAGIC display", 0, 0, 650, 500) ;
  
  for (Int_t i=0; i< fNbPixels; i++) 
    { 
      ( (MHexagon*) fPixels->At(i))->Draw() ; 
    } 

  for (Int_t i=0; i< ITEMS_LEGEND; i++) 
    { 
      
      ( (TBox*) fLegend->At(i))->Draw() ; 

      ( (TText*) fLegText->At(i))->Draw() ; 
    } 
  
} 


void MCamDisplay::Draw(Option_t *option  ) 
{
  // 

  //  check if there a pad exists

  if ( ! gPad ) Init() ; 

  gPad->Range (-600, -600, 900, 600) ; 
  gPad->SetFillColor(22) ; 

  //
  
  gPad->Modified() ; 
  gPad->Update() ; 

  //gPad->Update() ; 
}  

void MCamDisplay::Draw( MCerPhotEvt *event) 
{

  // loop over all pixels in the MCerPhotEvt and
  // determine the Pixel Id and the content

  Reset() ; 

  //  if the autoscale is true, set the values for the range for 
  //  each event 
  
  if ( fAutoScale == kTRUE ) 
    { 
      fMinPhe = event->GetMinimumPhoton() ; 
      fMaxPhe = event->GetMaximumPhoton() ;

      if ( fMaxPhe < 20. ) fMaxPhe = 20. ; 


      UpdateLegend() ; 
    }  

  //   update the picture 

  for (Int_t i=0 ; i<event->GetNbPixels() ; i++ )
    {
      if ( event->IsPixelUsed(i) == kTRUE ) 
	( (MHexagon*) fPixels->At( event->GetPixelId(i) ))->SetFillColor( GetColor(event->GetPhotons(i))) ; 
   
    } 
 
  Draw() ; 
  
}  

void MCamDisplay::DrawError( MCerPhotEvt *event) 
{
  // 

  // loop over all pixels in the MCerPhotEvt and
  // determine the Pixel Id and the content
  Reset() ; 

  for (Int_t i=0 ; i<event->GetNbPixels() ; i++ )
    {
      ( (MHexagon*) fPixels->At( event->GetPixelId(i) ))->SetFillColor( GetColor(event->GetErrorPhot(i)) ) ; 
    } 
  
  Draw() ; 
  
}  


void MCamDisplay::Reset() 
{
  for ( Int_t i=0 ; i< fNbPixels ; i++  )
    { 
      ( (MHexagon*) fPixels->At(i))->SetFillColor(10) ;
    } 

} 

Int_t MCamDisplay::GetColor(Float_t wert ) 
{
  //   Here we calculate the color index for the current value. 
  //   The color index is defined with the class TStyle and the 
  //   Color palette inside. We use the command gStyle->SetPalette(1,0) 
  //   for the display. So we have to convert the value "wert" into
  //   a color index that fits the color palette. 
  //   The range of the color palette is defined by the values fMinPhe 
  //   and fMaxRange. Between this values we have 50 color index, starting
  //   with 0 up to 49. 
  // 

  //   first treat the over- and under-flows

  if ( wert >= fMaxPhe ) 
    return gStyle->GetColorPalette( 49 )   ; 

  if ( wert <= fMinPhe ) 
    return gStyle->GetColorPalette( 0 )  ;

  // calculate the color index 

  Int_t ColIndex ; 

  ColIndex =  (Int_t) wert ; 
  
  ColIndex =  (Int_t) ( .5 + ( (wert-fMinPhe) * 49. / (fMaxPhe-fMinPhe) ) ) ; 

  return (gStyle->GetColorPalette(ColIndex) ) ; 

} 


void MCamDisplay::UpdateLegend() 
{ 
  //    change the text on the legend according to the range of the 
  //    Display
  
  char text[100] ; 
  
  Float_t x, y, wert,  help ;
  
  help = 50./ITEMS_LEGEND ;  
  
  for (Int_t il=0; il < ITEMS_LEGEND; il++)
    { 
      wert = fMinPhe + (il*help)/50 * (fMaxPhe-fMinPhe) ; 
      
      sprintf ( text, "%5.1f", wert ) ; 
      
      x = ( (TText*) fLegText->At(il))->GetX () ; 
      y = ( (TText*) fLegText->At(il))->GetY () ; 
      ( (TText*) fLegText->At(il))->SetText (x, y,  text ) ; 
    }  
  
} 
