#ifndef MARS_MJCalibration
#define MARS_MJCalibration

#ifndef MARS_MJob
#include "MJob.h"
#endif
#ifndef MARS_MCalibrationIntensityChargeCam
#include "MCalibrationIntensityChargeCam.h"
#endif
#ifndef MARS_MCalibrationChargeCam
#include "MCalibrationChargeCam.h"
#endif
#ifndef MARS_MCalibrationIntensityBlindCam
#include "MCalibrationIntensityBlindCam.h"
#endif
#ifndef MARS_MCalibrationBlindCam
#include "MCalibrationBlindCam.h"
#endif
#ifndef MARS_MCalibrationChargePINDiode
#include "MCalibrationChargePINDiode.h"
#endif
#ifndef MARS_MCalibrationIntensityRelTimeCam
#include "MCalibrationIntensityRelTimeCam.h"
#endif
#ifndef MARS_MCalibrationRelTimeCam
#include "MCalibrationRelTimeCam.h"
#endif
#ifndef MARS_MCalibrationIntensityQECam
#include "MCalibrationIntensityQECam.h"
#endif
#ifndef MARS_MCalibrationQECam
#include "MCalibrationQECam.h"
#endif
#ifndef MARS_MBadPixelsIntensityCam
#include "MBadPixelsIntensityCam.h"
#endif
#ifndef MARS_MBadPixelsCam
#include "MBadPixelsCam.h"
#endif
#ifndef MARS_MPedestalCam
#include "MPedestalCam.h"
#endif

class TH1D;

class MTask;
class MRunIter;
class MParList;
class MPedestalCam;
class MExtractor;
class MExtractTime;
class MEvtLoop;
class MExtractBlindPixel;
class MHCalibrationChargeBlindCam;
class MHCamera;

class MJCalibration : public MJob
{
private:

  static const Int_t gkIFAEBoxInaugurationRun;          //! Run number of first IFAE box calibration
  static const Int_t gkSecondBlindPixelInstallation;    //! Run number upon which second blind pixel was installed
  static const Int_t gkSpecialPixelsContInstallation;   //! Run number upon which third blind pixel was installed
  static const Int_t gkThirdBlindPixelInstallation;     //! Run number upon which third blind pixel was installed  

  static const TString  fgReferenceFile;   //! default for fReferenceFile ("mjobs/calibrationref.rc")

  Double_t fConvFADC2PheMin;               //! Histogram minimum for conversion factor to phes
  Double_t fConvFADC2PheMax;               //! Histogram maximum for conversion factor to phes
  Double_t fConvFADC2PhotMin;              //! Histogram minimum for conversion factor to phs
  Double_t fConvFADC2PhotMax;              //! Histogram maixmum for conversion factor to phs
  Double_t fQEMin;                         //! Histogram minimum for quantum efficiency
  Double_t fQEMax;                         //! Histogram maximum for quantum efficiency
  Double_t fArrivalTimeMin;                //! Histogram minimum for arrival times 
  Double_t fArrivalTimeMax;                //! Histogram maximum for arrival times
  Double_t fTimeOffsetMin;                 //! Histogram minimum for time offsets
  Double_t fTimeOffsetMax;                 //! Histogram maximum for time offsets
  Double_t fTimeResolutionMin;             //! Histogram minimum for time resolutions
  Double_t fTimeResolutionMax;             //! Histogram maximum for time resolutions
  					   
  Float_t  fRefConvFADC2PheInner;          //! Reference value for the conversion factor to phes - inner
  Float_t  fRefConvFADC2PheOuter;          //! Reference value for the conversion factor to phes - outer
  Float_t  fRefConvFADC2PhotInner;         //! Reference value for the conversion factor to phs - inner
  Float_t  fRefConvFADC2PhotOuter;         //! Reference value for the conversion factor to phs - outer
  Float_t  fRefQEInner;                    //! Reference value for the quantum eff. cascades - inner
  Float_t  fRefQEOuter;                    //! Reference value for the quantum eff. cascades - outer
  Float_t  fRefArrivalTimeInner;           //! Reference value for the arrival time - inner
  Float_t  fRefArrivalTimeOuter;           //! Reference value for the arrival time - outer
  Float_t  fRefArrivalTimeRmsInner;        //! Reference value for the arrival time - inner
  Float_t  fRefArrivalTimeRmsOuter;        //! Reference value for the arrival time - outer
  Float_t  fRefTimeOffsetInner;            //! Reference value for the time offset - inner
  Float_t  fRefTimeOffsetOuter;            //! Reference value for the time offset - outer
  Float_t  fRefTimeResolutionInner;  	   //! Reference value for the time resolution - inner
  Float_t  fRefTimeResolutionOuter;	   //! Reference value for the time resolution - outer
					   
  TString  fReferenceFile;                 // File name containing the reference values

  MRunIter       *fRuns;                                // Calibration files
  
  MExtractor     *fExtractor;                           // Signal extractor
  MExtractTime   *fTimeExtractor;                       // Arrival Time extractor
  
  MBadPixelsIntensityCam          fIntensBadCam;        // Bad Pixels cam, can be set from previous runs
  MBadPixelsCam                   fBadPixels;           // Bad Pixels cam, can be set from previous runs  
  MCalibrationIntensityChargeCam  fIntensCalibCam;      // Calibration conversion factors FADC2Phe for diff. intens.
  MCalibrationChargeCam           fCalibrationCam;      // Calibration conversion factors FADC2Phe  
  MCalibrationIntensityBlindCam   fIntensBlindCam;      // Calibration from Blind Pixel(s)
  MCalibrationBlindCam            fCalibrationBlindCam; // Calibration from Blind Pixel(s)  
  MCalibrationChargePINDiode      fCalibrationPINDiode; // Calibration from PIN Diode
  MCalibrationIntensityQECam      fIntensQECam;         // Quantum efficiency for diff. intensities, 
  MCalibrationQECam               fQECam;               // Quantum efficiency, can be set from previous runs
  MCalibrationIntensityRelTimeCam fIntensRelTimeCam;    // Calibration constants rel. times for diff. intensities
  MCalibrationRelTimeCam          fRelTimeCam;          // Calibration constants rel. times
  
  MPedestalCam                    fExtractorCam;        // Pedestal Cam with extractor resolution

  MCalibrationCam::PulserColor_t fColor;                // Colour of the pulsed LEDs
  
  enum  Display_t                                       // Possible Display types
    {
      kFullDisplay,
      kDataCheckDisplay,
      kNormalDisplay
    };
  
  Display_t fDisplayType;                              // Chosen Display type
  
  enum  Device_t                                       // Possible devices for calibration
    {
      kUseBlindPixel,
      kUsePINDiode
    }; 
  
  Byte_t fDevices;                                     // Bit-field for used devices for calibration

  enum  Storage_t                                       // Possible devices for calibration
    {
      kNoStorage,
      kHistsStorage
    };                                                 // Possible flags for the storage of results
  
  Byte_t fStorage;                                     // Bit-field for chosen storage type

  enum { kRelTimes, kDebug, kIntensity };  // Possible flags

  Byte_t fFlags;                                       // Bit-field for the flags
  
    enum DataType_t
      {
        kIsUseRawData,
        kIsUseRootData,
        kIsUseMC
      };

    Byte_t    fDataFlag;      // Bit-field to store the data type

    Bool_t IsUseRawData()  const { return TESTBIT(fDataFlag,kIsUseRawData);  }
    Bool_t IsUseRootData() const { return TESTBIT(fDataFlag,kIsUseRootData); }
    Bool_t IsUseMC()       const { return TESTBIT(fDataFlag,kIsUseMC);       }

    void   SetUseRawData ( const Bool_t b=kTRUE )  { b ? SETBIT(fDataFlag,kIsUseRawData)  : CLRBIT(fDataFlag,kIsUseRawData);  }
    void   SetUseRootData( const Bool_t b=kTRUE )  { b ? SETBIT(fDataFlag,kIsUseRootData) : CLRBIT(fDataFlag,kIsUseRootData); }
    void   SetUseMC      ( const Bool_t b=kTRUE )  { b ? SETBIT(fDataFlag,kIsUseMC)       : CLRBIT(fDataFlag,kIsUseMC);       }


  TString fGeometry;                                   // Class name geometry
  
  Bool_t IsUseBlindPixel() const { return TESTBIT(fDevices,kUseBlindPixel); }
  Bool_t IsUsePINDiode  () const { return TESTBIT(fDevices,kUsePINDiode);   }

  Bool_t IsRelTimes     () const { return TESTBIT(fFlags,kRelTimes);       }
  Bool_t IsDebug        () const { return TESTBIT(fFlags,kDebug);          }
  Bool_t IsIntensity    () const { return TESTBIT(fFlags,kIntensity);      }

  Bool_t IsNoStorage    () const { return TESTBIT(fStorage,kNoStorage);    }
  Bool_t IsHistsStorage () const { return TESTBIT(fStorage,kHistsStorage); }

  void   DrawTab(MParList &plist, const char *cont, const char *name, Option_t *opt);
  
  void   DisplayResult        ( MParList &plist );
  void   DisplayReferenceLines( MHCamera *cam , const Int_t what                           ) const;
  void   DisplayOutliers      ( TH1D     *hist, const char* whatsmall, const char* whatbig ) const;
  void   DisplayDoubleProject ( MHCamera *cam , const char* whatsmall, const char* whatbig ) const;
  const  Int_t CountBadPixels ( MHCamera *cam , const Int_t what                           ) const;
  void   FixDataCheckHist     ( TH1D     *hist  ) const;
  
  Bool_t WriteResult   (MParList &plist);
  //Bool_t WriteEventloop( MEvtLoop &evtloop   ) const;
  Bool_t WriteTasks    ( MTask *t1, MTask *t2) const;
  Bool_t CheckEnvLocal ();
  
  // WORKAROUNDS!!!
  void   InitBlindPixel(MExtractBlindPixel &blindext,
                        MHCalibrationChargeBlindCam &blindcam);

  const char* GetOutputFileName() const;
  
public:
  MJCalibration(const char *name=NULL, const char *title=NULL);
  
  const char* GetOutputFile() const;
  
  MCalibrationIntensityChargeCam  &GetIntensCalibrationCam() { return fIntensCalibCam;   }
  MCalibrationIntensityRelTimeCam &GetIntensRelTimeCam()     { return fIntensRelTimeCam; }
  MCalibrationIntensityQECam      &GetIntensQECam()          { return fIntensQECam;      }
  MBadPixelsIntensityCam          &GetIntensBadCam()         { return fIntensBadCam;     }  

  MCalibrationChargeCam           &GetCalibrationCam()       { return fCalibrationCam;   }
  MCalibrationRelTimeCam          &GetRelTimeCam()           { return fRelTimeCam;       }
  MCalibrationQECam               &GetQECam()                { return fQECam;            }
  MBadPixelsCam                   &GetBadPixels()            { return fBadPixels;        }
  
  // Containers and pointers
  void SetBadPixels    ( const MBadPixelsCam &bad)                   { bad.Copy(fBadPixels); }
  void SetColor        ( const MCalibrationCam::PulserColor_t color) { fColor = color;       }
  void SetExtractor    ( MExtractor* ext)                            { fExtractor = ext;     }
  void SetExtractorCam ( const MPedestalCam  &ped)                   { ped.Copy(fExtractorCam); fExtractorCam.SetName(ped.GetName()); }
  void SetInput        ( MRunIter *iter)                             { fRuns = iter;         }
  void SetQECam        ( const MCalibrationQECam &qe)                { qe.Copy(fQECam);      }
  void SetTimeExtractor( MExtractTime* ext)                          { fTimeExtractor = ext; }
    
  // Displays
  void SetFullDisplay()      { fDisplayType = kFullDisplay;      }
  void SetDataCheckDisplay() { fDisplayType = kDataCheckDisplay; }
  void SetNormalDisplay()    { fDisplayType = kNormalDisplay;    }

  // Camera Geomtry
  void SetGeometry(const char *geom) { fGeometry = geom; }
  
  // Flags
  void SetRelTimeCalibration( const Bool_t b=kTRUE ) { b ? SETBIT(fFlags,kRelTimes)  : CLRBIT(fFlags,kRelTimes); }
  void SetDebug             ( const Bool_t b=kTRUE ) { b ? SETBIT(fFlags,kDebug)     : CLRBIT(fFlags,kDebug); }
  void SetIntensity         ( const Bool_t b=kTRUE ) { b ? SETBIT(fFlags,kIntensity) : CLRBIT(fFlags,kIntensity); }
  
  // Devices
  void SetUseBlindPixel(const Bool_t b=kTRUE);
  void SetUsePINDiode(const Bool_t b=kTRUE);

  // Storage
  void SetNoStorage    ( const Bool_t b=kTRUE ) { b ? SETBIT(fStorage,kNoStorage)    : CLRBIT(fStorage,kNoStorage); }
  void SetHistsStorage ( const Bool_t b=kTRUE ) { b ? SETBIT(fStorage,kHistsStorage) : CLRBIT(fStorage,kHistsStorage); }
  
  void SetReferenceFile( const TString ref=fgReferenceFile ) { fReferenceFile = ref; }

  // Precessing
  Bool_t ReadCalibrationCam();
  Bool_t ProcessFile(MPedestalCam &pedcam);
  Bool_t Process(MPedestalCam &pedcam);

  ClassDef(MJCalibration, 0) // Tool to run a calibration per pulser colour and intensity
};

#endif
